/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.primitive;

import java.util.Arrays;

import com.gs.collections.api.LongIterable;
import com.gs.collections.api.LazyLongIterable;
import com.gs.collections.api.bag.primitive.MutableLongBag;
import com.gs.collections.api.block.predicate.primitive.LongPredicate;
import com.gs.collections.api.list.primitive.MutableLongList;
import com.gs.collections.api.set.primitive.MutableLongSet;
import com.gs.collections.impl.bag.mutable.primitive.LongHashBag;
import com.gs.collections.impl.lazy.primitive.LazyLongIterableAdapter;
import com.gs.collections.impl.list.mutable.primitive.LongArrayList;
import com.gs.collections.impl.set.mutable.primitive.LongHashSet;

/**
 * This file was automatically generated from template file abstractPrimitiveIterable.stg.
 * @since 6.0
 */
public abstract class AbstractLongIterable implements LongIterable
{
    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public long minIfEmpty(long defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.min();
    }

    public long maxIfEmpty(long defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.max();
    }

    public double average()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        return (double) this.sum() / (double) this.size();
    }

    public double median()
    {
        if (this.isEmpty())
        {
            throw new ArithmeticException();
        }
        long[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            long first = sortedArray[middleIndex];
            long second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public long[] toSortedArray()
    {
        long[] array = this.toArray();
        Arrays.sort(array);
        return array;
    }

    public MutableLongList toSortedList()
    {
        return LongArrayList.newList(this).sortThis();
    }

    public LazyLongIterable asLazy()
    {
        return new LazyLongIterableAdapter(this);
    }

    public MutableLongList toList()
    {
        return LongArrayList.newList(this);
    }

    public MutableLongSet toSet()
    {
        return LongHashSet.newSet(this);
    }

    public MutableLongBag toBag()
    {
        return LongHashBag.newBag(this);
    }

    public boolean containsAll(long... source)
    {
        for (long item : source)
        {
            if (!this.contains(item))
            {
                return false;
            }
        }
        return true;
    }


    public boolean containsAll(LongIterable source)
    {
        return source.allSatisfy(new LongPredicate()
        {
            public boolean accept(long each)
            {
                return AbstractLongIterable.this.contains(each);
            }
        });
    }

    public boolean isEmpty()
    {
        return this.size() == 0;
    }

    public boolean notEmpty()
    {
        return this.size() != 0;
    }
}
