/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.set.immutable.primitive;

import java.io.IOException;
import java.io.Serializable;

import com.gs.collections.api.ByteIterable;
import com.gs.collections.api.LazyByteIterable;
import com.gs.collections.api.bag.primitive.MutableByteBag;
import com.gs.collections.api.set.ImmutableSet;
import com.gs.collections.api.set.primitive.ImmutableByteSet;
import com.gs.collections.api.set.primitive.ByteSet;
import com.gs.collections.api.set.primitive.MutableByteSet;
import com.gs.collections.api.block.function.primitive.ByteToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectByteToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.BytePredicate;
import com.gs.collections.api.block.procedure.primitive.ByteIntProcedure;
import com.gs.collections.api.block.procedure.primitive.ByteProcedure;
import com.gs.collections.api.iterator.ByteIterator;
import com.gs.collections.api.list.primitive.MutableByteList;
import com.gs.collections.impl.bag.mutable.primitive.ByteHashBag;
import com.gs.collections.impl.iterator.UnmodifiableByteIterator;
import com.gs.collections.impl.set.mutable.UnifiedSet;
import com.gs.collections.impl.set.mutable.primitive.ByteHashSet;
import com.gs.collections.impl.factory.primitive.ByteSets;
import com.gs.collections.impl.lazy.primitive.LazyByteIterableAdapter;
import com.gs.collections.impl.list.mutable.primitive.ByteArrayList;
import net.jcip.annotations.Immutable;

/**
 * ImmutableByteSingletonSet is an optimization for {@link ImmutableByteSet} of size 1.
 * This file was automatically generated from template file immutablePrimitiveSingletonSet.stg.
 *
 * @since 4.0.
 */
@Immutable
final class ImmutableByteSingletonSet implements ImmutableByteSet, Serializable
{
    private static final long serialVersionUID = 1L;

    private final byte element;

    ImmutableByteSingletonSet(byte element)
    {
        this.element = element;
    }

    public ImmutableByteSet newWith(byte element)
    {
        return ByteSets.immutable.with(this.element, element);
    }

    public ImmutableByteSet newWithout(byte element)
    {
        return this.element == element ? ByteSets.immutable.with() : this;
    }

    public ImmutableByteSet newWithAll(ByteIterable elements)
    {
        return ByteHashSet.newSet(elements).with(this.element).toImmutable();
    }

    public ImmutableByteSet newWithoutAll(ByteIterable elements)
    {
        return elements.contains(this.element) ? ByteSets.immutable.with() : this;
    }

    public int size()
    {
        return 1;
    }

    public boolean isEmpty()
    {
        return false;
    }

    public boolean notEmpty()
    {
        return true;
    }

    public boolean contains(byte value)
    {
        return this.element == value;
    }

    public boolean containsAll(ByteIterable source)
    {
        for (ByteIterator iterator = source.byteIterator(); iterator.hasNext(); )
        {
            if (this.element != iterator.next())
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(byte... source)
    {
        for (byte value : source)
        {
            if (this.element != value)
            {
                return false;
            }
        }
        return true;
    }

    public void forEach(ByteProcedure procedure)
    {
        procedure.value(this.element);
    }

    public ImmutableByteSet select(BytePredicate predicate)
    {
        return predicate.accept(this.element) ? ByteHashSet.newSetWith(this.element).toImmutable()
                : new ByteHashSet().toImmutable();
    }

    public ImmutableByteSet reject(BytePredicate predicate)
    {
        return predicate.accept(this.element) ? new ByteHashSet().toImmutable()
                : ByteHashSet.newSetWith(this.element).toImmutable();
    }

    public <V> ImmutableSet<V> collect(ByteToObjectFunction<? extends V> function)
    {
        return UnifiedSet.newSetWith(function.valueOf(this.element)).toImmutable();
    }

    public MutableByteList toList()
    {
        return ByteArrayList.newListWith(this.element);
    }

    public int sizeDistinct()
    {
        return 1;
    }

    public int occurrencesOf(byte item)
    {
        return this.element == item ? 1 : 0;
    }

    public void forEachWithOccurrences(ByteIntProcedure byteIntProcedure)
    {
        byteIntProcedure.value(this.element, 1);
    }

    public byte detectIfNone(BytePredicate predicate, byte ifNone)
    {
        return predicate.accept(this.element) ? this.element : ifNone;
    }

    public int count(BytePredicate predicate)
    {
        return predicate.accept(this.element) ? 1 : 0;
    }

    public boolean anySatisfy(BytePredicate predicate)
    {
        return predicate.accept(this.element);
    }

    public long sum()
    {
        return this.element;
    }

    public byte min()
    {
        return this.element;
    }

    public byte max()
    {
        return this.element;
    }

    public byte maxIfEmpty(byte defaultValue)
    {
        return this.element;
    }

    public byte minIfEmpty(byte defaultValue)
    {
        return this.element;
    }

    public double average()
    {
        return this.element;
    }

    public double median()
    {
        return this.element;
    }

    public byte[] toSortedArray()
    {
        return new byte[]{this.element};
    }

    public MutableByteList toSortedList()
    {
        return ByteArrayList.newListWith(this.element);
    }

    public boolean noneSatisfy(BytePredicate predicate)
    {
        return !predicate.accept(this.element);
    }

    public boolean allSatisfy(BytePredicate predicate)
    {
        return predicate.accept(this.element);
    }

    public <T> T injectInto(T injectedValue, ObjectByteToObjectFunction<? super T, ? extends T> function)
    {
        return function.valueOf(injectedValue, this.element);
    }

    @Override
    public boolean equals(Object obj)
    {
        if (obj == this)
        {
            return true;
        }
        if (!(obj instanceof ByteSet))
        {
            return false;
        }
        ByteSet set = (ByteSet) obj;
        if (set.size() != 1)
        {
            return false;
        }
        return set.contains(this.element);
    }

    @Override
    public int hashCode()
    {
        return (int) this.element;
    }

    public MutableByteSet toSet()
    {
        return ByteHashSet.newSetWith(this.element);
    }

    public MutableByteBag toBag()
    {
        return ByteHashBag.newBagWith(this.element);
    }

    public ByteSet freeze()
    {
        return this;
    }

    public ImmutableByteSet toImmutable()
    {
        return this;
    }

    public LazyByteIterable asLazy()
    {
        return new LazyByteIterableAdapter(this);
    }

    public byte[] toArray()
    {
        return new byte[]{this.element};
    }

    @Override
    public String toString()
    {
        return '[' + this.makeString() + ']';
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);
            appendable.append(String.valueOf(this.element));
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public ByteIterator byteIterator()
    {
        return new UnmodifiableByteIterator(ByteHashSet.newSetWith(this.element).byteIterator());
    }
}

