package io.smallrye.reactive.messaging.kafka;

import java.util.Optional;
import org.eclipse.microprofile.config.Config;

/**
 * Extract the incoming configuration for the {@code smallrye-kafka} connector.
*/
public class KafkaConnectorIncomingConfiguration extends KafkaConnectorCommonConfiguration {

  /**
   * Creates a new KafkaConnectorIncomingConfiguration.
   */
  public KafkaConnectorIncomingConfiguration(Config config) {
    super(config);
    validate();
  }

  /**
  * Gets the topics value from the configuration.
  * Attribute Name: topics
  * Description: A comma-separating list of topics to be consumed. Cannot be used with the `topic` or `pattern` properties
  * @return the topics
  */
  public Optional<String> getTopics() {
    return config.getOptionalValue("topics", String.class);
  }

  /**
  * Gets the pattern value from the configuration.
  * Attribute Name: pattern
  * Description: Indicate that the `topic` property is a regular expression. Must be used with the `topic` property. Cannot be used with the `topics` property
  * Default Value: false
  * @return the pattern
  */
  public Boolean getPattern() {
    return config.getOptionalValue("pattern", Boolean.class)
     .orElse(Boolean.valueOf("false"));
  }

  /**
  * Gets the key.deserializer value from the configuration.
  * Attribute Name: key.deserializer
  * Description: The deserializer classname used to deserialize the record's key
  * Default Value: org.apache.kafka.common.serialization.StringDeserializer
  * @return the key.deserializer
  */
  public String getKeyDeserializer() {
    return config.getOptionalValue("key.deserializer", String.class)
     .orElse("org.apache.kafka.common.serialization.StringDeserializer");
  }

  /**
  * Gets the value.deserializer value from the configuration.
  * Attribute Name: value.deserializer
  * Description: The deserializer classname used to deserialize the record's value
  * Mandatory: yes
  * @return the value.deserializer
  */
  public String getValueDeserializer() {
    return config.getOptionalValue("value.deserializer", String.class)
        .orElseThrow(() -> new IllegalArgumentException("The attribute `value.deserializer` on connector 'smallrye-kafka' (channel: " + getChannel() + ") must be set"));
  }

  /**
  * Gets the fetch.min.bytes value from the configuration.
  * Attribute Name: fetch.min.bytes
  * Description: The minimum amount of data the server should return for a fetch request. The default setting of 1 byte means that fetch requests are answered as soon as a single byte of data is available or the fetch request times out waiting for data to arrive.
  * Default Value: 1
  * @return the fetch.min.bytes
  */
  public Integer getFetchMinBytes() {
    return config.getOptionalValue("fetch.min.bytes", Integer.class)
     .orElse(Integer.valueOf("1"));
  }

  /**
  * Gets the group.id value from the configuration.
  * Attribute Name: group.id
  * Description: A unique string that identifies the consumer group the application belongs to. If not set, a unique, generated id is used
  * @return the group.id
  */
  public Optional<String> getGroupId() {
    return config.getOptionalValue("group.id", String.class);
  }

  /**
  * Gets the enable.auto.commit value from the configuration.
  * Attribute Name: enable.auto.commit
  * Description: If enabled, consumer's offset will be periodically committed in the background by the underlying Kafka client, ignoring the actual processing outcome of the records. It is recommended to NOT enable this setting and let Reactive Messaging handles the commit.
  * Default Value: false
  * @return the enable.auto.commit
  */
  public Boolean getEnableAutoCommit() {
    return config.getOptionalValue("enable.auto.commit", Boolean.class)
     .orElse(Boolean.valueOf("false"));
  }

  /**
  * Gets the retry value from the configuration.
  * Attribute Name: retry
  * Description: Whether or not the connection to the broker is re-attempted in case of failure
  * Default Value: true
  * @return the retry
  */
  public Boolean getRetry() {
    return config.getOptionalValue("retry", Boolean.class)
     .orElse(Boolean.valueOf("true"));
  }

  /**
  * Gets the retry-attempts value from the configuration.
  * Attribute Name: retry-attempts
  * Description: The maximum number of reconnection before failing. -1 means infinite retry
  * Default Value: -1
  * @return the retry-attempts
  */
  public Integer getRetryAttempts() {
    return config.getOptionalValue("retry-attempts", Integer.class)
     .orElse(Integer.valueOf("-1"));
  }

  /**
  * Gets the retry-max-wait value from the configuration.
  * Attribute Name: retry-max-wait
  * Description: The max delay (in seconds) between 2 reconnects
  * Default Value: 30
  * @return the retry-max-wait
  */
  public Integer getRetryMaxWait() {
    return config.getOptionalValue("retry-max-wait", Integer.class)
     .orElse(Integer.valueOf("30"));
  }

  /**
  * Gets the broadcast value from the configuration.
  * Attribute Name: broadcast
  * Description: Whether the Kafka records should be dispatched to multiple consumer
  * Default Value: false
  * @return the broadcast
  */
  public Boolean getBroadcast() {
    return config.getOptionalValue("broadcast", Boolean.class)
     .orElse(Boolean.valueOf("false"));
  }

  /**
  * Gets the auto.offset.reset value from the configuration.
  * Attribute Name: auto.offset.reset
  * Description: What to do when there is no initial offset in Kafka.Accepted values are earliest, latest and none
  * Default Value: latest
  * @return the auto.offset.reset
  */
  public String getAutoOffsetReset() {
    return config.getOptionalValue("auto.offset.reset", String.class)
     .orElse("latest");
  }

  /**
  * Gets the failure-strategy value from the configuration.
  * Attribute Name: failure-strategy
  * Description: Specify the failure strategy to apply when a message produced from a record is acknowledged negatively (nack). Values can be `fail` (default), `ignore`, or `dead-letter-queue`
  * Default Value: fail
  * @return the failure-strategy
  */
  public String getFailureStrategy() {
    return config.getOptionalValue("failure-strategy", String.class)
     .orElse("fail");
  }

  /**
  * Gets the commit-strategy value from the configuration.
  * Attribute Name: commit-strategy
  * Description: Specify the commit strategy to apply when a message produced from a record is acknowledged. Values can be `latest`, `ignore` or `throttled`. If `enable.auto.commit` is true then the default is `ignore` otherwise it is `throttled`
  * @return the commit-strategy
  */
  public Optional<String> getCommitStrategy() {
    return config.getOptionalValue("commit-strategy", String.class);
  }

  /**
  * Gets the throttled.unprocessed-record-max-age.ms value from the configuration.
  * Attribute Name: throttled.unprocessed-record-max-age.ms
  * Description: While using the `throttled` commit-strategy, specify the max age in milliseconds that an unprocessed message can be before the connector is marked as unhealthy. Setting this attribute to 0 disables this monitoring.
  * Default Value: 60000
  * @return the throttled.unprocessed-record-max-age.ms
  */
  public Integer getThrottledUnprocessedRecordMaxAgeMs() {
    return config.getOptionalValue("throttled.unprocessed-record-max-age.ms", Integer.class)
     .orElse(Integer.valueOf("60000"));
  }

  /**
  * Gets the dead-letter-queue.topic value from the configuration.
  * Attribute Name: dead-letter-queue.topic
  * Description: When the `failure-strategy` is set to `dead-letter-queue` indicates on which topic the record is sent. Defaults is `dead-letter-topic-$channel`
  * @return the dead-letter-queue.topic
  */
  public Optional<String> getDeadLetterQueueTopic() {
    return config.getOptionalValue("dead-letter-queue.topic", String.class);
  }

  /**
  * Gets the dead-letter-queue.key.serializer value from the configuration.
  * Attribute Name: dead-letter-queue.key.serializer
  * Description: When the `failure-strategy` is set to `dead-letter-queue` indicates the key serializer to use. If not set the serializer associated to the key deserializer is used
  * @return the dead-letter-queue.key.serializer
  */
  public Optional<String> getDeadLetterQueueKeySerializer() {
    return config.getOptionalValue("dead-letter-queue.key.serializer", String.class);
  }

  /**
  * Gets the dead-letter-queue.value.serializer value from the configuration.
  * Attribute Name: dead-letter-queue.value.serializer
  * Description: When the `failure-strategy` is set to `dead-letter-queue` indicates the value serializer to use. If not set the serializer associated to the value deserializer is used
  * @return the dead-letter-queue.value.serializer
  */
  public Optional<String> getDeadLetterQueueValueSerializer() {
    return config.getOptionalValue("dead-letter-queue.value.serializer", String.class);
  }

  /**
  * Gets the partitions value from the configuration.
  * Attribute Name: partitions
  * Description: The number of partitions to be consumed concurrently. The connector creates the specified amount of Kafka consumers. It should match the number of partition of the targeted topic
  * Default Value: 1
  * @return the partitions
  */
  public Integer getPartitions() {
    return config.getOptionalValue("partitions", Integer.class)
     .orElse(Integer.valueOf("1"));
  }

  /**
  * Gets the requests value from the configuration.
  * Attribute Name: requests
  * Description: When `partitions` is greater than 1, this attribute allows configuring how many records are requested by each consumers every time.
  * Default Value: 128
  * @return the requests
  */
  public Integer getRequests() {
    return config.getOptionalValue("requests", Integer.class)
     .orElse(Integer.valueOf("128"));
  }

  /**
  * Gets the consumer-rebalance-listener.name value from the configuration.
  * Attribute Name: consumer-rebalance-listener.name
  * Description: The name set in `@Identifier` of a bean that implements `io.smallrye.reactive.messaging.kafka.KafkaConsumerRebalanceListener`. If set, this rebalance listener is applied to the consumer.
  * @return the consumer-rebalance-listener.name
  */
  public Optional<String> getConsumerRebalanceListenerName() {
    return config.getOptionalValue("consumer-rebalance-listener.name", String.class);
  }

  /**
  * Gets the key-deserialization-failure-handler value from the configuration.
  * Attribute Name: key-deserialization-failure-handler
  * Description: The name set in `@Identifier` of a bean that implements `io.smallrye.reactive.messaging.kafka.DeserializationFailureHandler`. If set, deserialization failure happening when deserializing keys are delegated to this handler which may retry or provide a fallback value.
  * @return the key-deserialization-failure-handler
  */
  public Optional<String> getKeyDeserializationFailureHandler() {
    return config.getOptionalValue("key-deserialization-failure-handler", String.class);
  }

  /**
  * Gets the value-deserialization-failure-handler value from the configuration.
  * Attribute Name: value-deserialization-failure-handler
  * Description: The name set in `@Identifier` of a bean that implements `io.smallrye.reactive.messaging.kafka.DeserializationFailureHandler`. If set, deserialization failure happening when deserializing values are delegated to this handler which may retry or provide a fallback value.
  * @return the value-deserialization-failure-handler
  */
  public Optional<String> getValueDeserializationFailureHandler() {
    return config.getOptionalValue("value-deserialization-failure-handler", String.class);
  }

  /**
  * Gets the fail-on-deserialization-failure value from the configuration.
  * Attribute Name: fail-on-deserialization-failure
  * Description: When no deserialization failure handler is set and a deserialization failure happens, report the failure and mark the application as unhealthy. If set to `false` and a deserialization failure happens, a `null` value is forwarded.
  * Default Value: true
  * @return the fail-on-deserialization-failure
  */
  public Boolean getFailOnDeserializationFailure() {
    return config.getOptionalValue("fail-on-deserialization-failure", Boolean.class)
     .orElse(Boolean.valueOf("true"));
  }

  /**
  * Gets the graceful-shutdown value from the configuration.
  * Attribute Name: graceful-shutdown
  * Description: Whether or not a graceful shutdown should be attempted when the application terminates.
  * Default Value: true
  * @return the graceful-shutdown
  */
  public Boolean getGracefulShutdown() {
    return config.getOptionalValue("graceful-shutdown", Boolean.class)
     .orElse(Boolean.valueOf("true"));
  }

  /**
  * Gets the poll-timeout value from the configuration.
  * Attribute Name: poll-timeout
  * Description: The polling timeout in milliseconds. When polling records, the poll will wait at most that duration before returning records. Default is 1000ms
  * Default Value: 1000
  * @return the poll-timeout
  */
  public Integer getPollTimeout() {
    return config.getOptionalValue("poll-timeout", Integer.class)
     .orElse(Integer.valueOf("1000"));
  }

  /**
  * Gets the pause-if-no-requests value from the configuration.
  * Attribute Name: pause-if-no-requests
  * Description: Whether the polling must be paused when the application does not request items and resume when it does. This allows implementing back-pressure based on the application capacity. Note that polling is not stopped, but will not retrieve any records when paused.
  * Default Value: true
  * @return the pause-if-no-requests
  */
  public Boolean getPauseIfNoRequests() {
    return config.getOptionalValue("pause-if-no-requests", Boolean.class)
     .orElse(Boolean.valueOf("true"));
  }

  /**
  * Gets the batch value from the configuration.
  * Attribute Name: batch
  * Description: Whether the Kafka records are consumed in batch. The channel injection point must consume a compatible type, such as `List<Payload>` or `KafkaRecordBatch<Payload>`.
  * Default Value: false
  * @return the batch
  */
  public Boolean getBatch() {
    return config.getOptionalValue("batch", Boolean.class)
     .orElse(Boolean.valueOf("false"));
  }

  /**
  * Gets the max-queue-size-factor value from the configuration.
  * Attribute Name: max-queue-size-factor
  * Description: Multiplier factor to determine maximum number of records queued for processing, using `max.poll.records` * `max-queue-size-factor`. Defaults to 2. In `batch` mode `max.poll.records` is considered `1`.
  * Default Value: 2
  * @return the max-queue-size-factor
  */
  public Integer getMaxQueueSizeFactor() {
    return config.getOptionalValue("max-queue-size-factor", Integer.class)
     .orElse(Integer.valueOf("2"));
  }

  public void validate() {
    super.validate();
    getValueDeserializer();
  }
}
