/*
 *    GeoTools - The Open Source Java GIS Toolkit
 *    http://geotools.org
 *
 *    (C) 2002-2008, Open Source Geospatial Foundation (OSGeo)
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation;
 *    version 2.1 of the License.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 */
package org.geotools.arcsde.session;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Vector;
import java.util.WeakHashMap;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.FutureTask;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.commons.pool.ObjectPool;
import org.geotools.arcsde.ArcSdeException;
import org.geotools.arcsde.session.Commands.GetVersionCommand;

import com.esri.sde.sdk.client.SeColumnDefinition;
import com.esri.sde.sdk.client.SeConnection;
import com.esri.sde.sdk.client.SeDBMSInfo;
import com.esri.sde.sdk.client.SeDelete;
import com.esri.sde.sdk.client.SeError;
import com.esri.sde.sdk.client.SeException;
import com.esri.sde.sdk.client.SeInsert;
import com.esri.sde.sdk.client.SeLayer;
import com.esri.sde.sdk.client.SeObjectId;
import com.esri.sde.sdk.client.SeQuery;
import com.esri.sde.sdk.client.SeRasterColumn;
import com.esri.sde.sdk.client.SeRegistration;
import com.esri.sde.sdk.client.SeRelease;
import com.esri.sde.sdk.client.SeRow;
import com.esri.sde.sdk.client.SeSqlConstruct;
import com.esri.sde.sdk.client.SeState;
import com.esri.sde.sdk.client.SeStreamOp;
import com.esri.sde.sdk.client.SeTable;
import com.esri.sde.sdk.client.SeUpdate;
import com.esri.sde.sdk.client.SeVersion;
import com.esri.sde.sdk.geom.GeometryFactory;

/**
 * Provides thread safe access to an SeConnection.
 * <p>
 * This class has become more and more magic over time! It no longer represents a Connection but
 * provides "safe" access to a connection.
 * <p>
 * 
 * @author Gabriel Roldan
 * @author Jody Garnett
 * @version $Id: Session.java 34115 2009-10-10 01:09:29Z groldan $
 * @since 2.3.x
 */
class Session implements ISession {

    private static final Logger LOGGER = Logger.getLogger("org.geotools.arcsde.pool");

    /**
     * How many seconds must have elapsed since the last connection round trip to the server for
     * {@link #testServer()} to actually check the connection's validity
     */
    protected static final long TEST_SERVER_ROUNDTRIP_INTERVAL_SECONDS = 5;

    /** Actual SeConnection being protected */
    private final SeConnection connection;

    /**
     * ObjectPool used to manage open connections (shared).
     */
    private final ObjectPool pool;

    private final ArcSDEConnectionConfig config;

    private static int sessionCounter;

    private final int sessionId;

    private boolean transactionInProgress;

    private boolean isPassivated;

    private Map<String, SeTable> cachedTables = new WeakHashMap<String, SeTable>();

    private Map<String, SeLayer> cachedLayers = new WeakHashMap<String, SeLayer>();

    /**
     * The SeConnection bound task executor, ensures all operations against a given connection are
     * performed in the same thread regardless of the thread the {@link #issue(Command)} is being
     * called from.
     */
    private final ExecutorService taskExecutor;

    /**
     * Thread used by the taskExecutor; so we can detect recursion.
     */
    private Thread commandThread;

    /**
     * Provides safe access to an SeConnection.
     * 
     * @param pool
     *            ObjectPool used to manage SeConnection
     * @param config
     *            Used to set up a SeConnection
     * @throws SeException
     *             If we cannot connect
     */
    Session(final ObjectPool pool, final ArcSDEConnectionConfig config) throws IOException {
        this.config = config;
        this.pool = pool;
        this.taskExecutor = Executors.newSingleThreadExecutor();

        // grab command thread, held by taskExecutor
        updateCommandThread();

        /*
         * This ensures the connection runs always on the same thread. Will fail if its accessed by
         * different threads
         */
        try {
            this.connection = issue(new CreateSessionCommand(config));
        } catch (IOException e) {
            // make sure a connection creation failure does not leave a stale thread
            this.taskExecutor.shutdownNow();
            throw e;
        } catch (RuntimeException shouldntHappen) {
            this.taskExecutor.shutdownNow();
            throw shouldntHappen;
        }

        synchronized (Session.class) {
            sessionCounter++;
            sessionId = sessionCounter;
        }
    }

    /**
     * @see ISession#issue(org.geotools.arcsde.session.Command)
     */
    public <T> T issue(final Command<T> command) throws IOException {
        final Thread callingThread = Thread.currentThread();
        if (callingThread == commandThread) {
            // Called command inside command
            try {
                return command.execute(this, connection);
            } catch (SeException e) {
                Throwable cause = e.getCause();
                if (cause instanceof IOException) {
                    throw (IOException) cause;
                }
                throw new ArcSdeException(e);
            }
        } else {
            final FutureTask<T> task = new FutureTask<T>(new Callable<T>() {
                public T call() throws Exception {
                    final Thread currentThread = Thread.currentThread();

                    if (commandThread != currentThread) {
                        LOGGER.fine("updating command thread from " + commandThread + " to "
                                + currentThread);
                        commandThread = currentThread;

                    }
                    if (currentThread != commandThread) {
                        throw new IllegalStateException("currentThread != commandThread");
                    }
                    try {
                        return command.execute(Session.this, connection);
                    } catch (Exception e) {
                        if (LOGGER.isLoggable(Level.FINEST)) {
                            LOGGER.log(Level.FINEST, "Command execution failed for Session "
                                    + Session.this.sessionId + " in thread "
                                    + currentThread.getId(), e);
                        } else if (LOGGER.isLoggable(Level.FINE)) {
                            LOGGER.fine("Command execution failed for Session "
                                    + Session.this.sessionId + " in thread "
                                    + currentThread.getId());
                        }

                        if (e instanceof SeException) {
                            throw new ArcSdeException((SeException) e);
                        } else if (e instanceof IOException) {
                            throw e;
                        }
                        throw new RuntimeException("Command execution failed for Session "
                                + Session.this.sessionId + " in thread " + currentThread.getId(), e);
                    }
                }
            });

            T result;
            taskExecutor.execute(task);
            try {
                result = task.get();
            } catch (InterruptedException e) {
                updateCommandThread();
                throw new RuntimeException("Command execution abruptly interrupted", e);
            } catch (ExecutionException e) {
                updateCommandThread();
                Throwable cause = e.getCause();
                if (cause instanceof IOException) {
                    throw (IOException) cause;
                } else if (cause instanceof SeException) {
                    throw new ArcSdeException((SeException) cause);
                }
                throw (IOException) new IOException().initCause(cause);
            }
            return result;
        }
    }

    private void updateCommandThread() {
        final FutureTask<?> task = new FutureTask<Object>(new Callable<Object>() {
            public Object call() throws Exception {
                final Thread currentThread = Thread.currentThread();
                if (currentThread != commandThread) {
                    LOGGER.fine("updating command thread from " + commandThread + " to "
                            + currentThread);
                    commandThread = currentThread;
                }
                return null;
            }
        });
        // used to detect when thread has been
        // restarted after error
        taskExecutor.execute(task);
        // block until task is executed
        try {
            task.get();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * @see ISession#testServer()
     */
    public final void testServer() throws IOException {
        /*
         * This method is called often (every time a session is to be returned from the pool) to
         * check if it's still valid. We can call getTimeSinceLastRT safely since it does not
         * require a server roundtrip and hence there's no risk of violating thread safety. So we do
         * it before issuing the command to avoid the perf penalty imposed by running the command if
         * not needed.
         */
        final long secondsSinceLastServerRoundTrip = this.connection.getTimeSinceLastRT();

        if (TEST_SERVER_ROUNDTRIP_INTERVAL_SECONDS < secondsSinceLastServerRoundTrip) {
            issue(new Command<Void>() {
                @Override
                public Void execute(final ISession session, final SeConnection connection)
                        throws SeException, IOException {
                    connection.testServer(TEST_SERVER_ROUNDTRIP_INTERVAL_SECONDS);

                    return null;
                }
            });
        }
    }

    /**
     * @see ISession#isClosed()
     */
    public final boolean isClosed() {
        return this.connection.isClosed();
    }

    /**
     * Marks the connection as being active (i.e. its out of the pool and ready to be used).
     * <p>
     * Shall be called just before being returned from the connection pool
     * </p>
     * 
     * @see #markInactive()
     * @see #isPassivated
     * @see #checkActive()
     */
    void markActive() {
        this.isPassivated = false;
    }

    /**
     * Marks the connection as being inactive (i.e. laying on the connection pool)
     * <p>
     * Shall be callled just before sending it back to the pool
     * </p>
     * 
     * @see #markActive()
     * @see #isPassivated
     * @see #checkActive()
     */
    void markInactive() {
        this.isPassivated = true;
    }

    /**
     * @see ISession#isPassivated()
     */
    public boolean isDisposed() {
        return isPassivated;
    }

    /**
     * Sanity check method called before every public operation delegates to the superclass.
     * 
     * @throws IllegalStateException
     *             if {@link #isDisposed() isPassivated() == true} as this is a serious workflow
     *             breackage.
     */
    private void checkActive() {
        if (isDisposed()) {
            throw new IllegalStateException("Unrecoverable error: " + toString()
                    + " is passivated, shall not be used!");
        }
    }

    /**
     * @see ISession#getLayer(java.lang.String)
     */
    public SeLayer getLayer(final String layerName) throws IOException {
        checkActive();
        if (!cachedLayers.containsKey(layerName)) {
            issue(new Command<Void>() {
                @Override
                public Void execute(final ISession session, final SeConnection connection)
                        throws SeException, IOException {
                    synchronized (cachedLayers) {
                        if (!cachedLayers.containsKey(layerName)) {
                            SeTable table = getTable(layerName);
                            String shapeColumn = getShapeColumn(table);
                            if (shapeColumn == null) {
                                return null;
                            }
                            SeLayer layer = new SeLayer(connection, layerName, shapeColumn);
                            cachedLayers.put(layerName, layer);
                        }
                    }
                    return null;
                }
            });
        }

        SeLayer seLayer = cachedLayers.get(layerName);
        if (seLayer == null) {
            throw new NoSuchElementException("Layer '" + layerName + "' not found");
        }
        return seLayer;

    }

    private String getShapeColumn(SeTable table) throws ArcSdeException {
        try {
            for (SeColumnDefinition aDef : table.describe()) {
                if (aDef.getType() == SeColumnDefinition.TYPE_SHAPE) {
                    return aDef.getName();
                }
            }
        } catch (SeException e) {
            throw new ArcSdeException("Exception describing table " + table.getName(), e);
        }
        return null;
    }

    /**
     * @see ISession#getRasterColumn(java.lang.String)
     */
    public synchronized SeRasterColumn getRasterColumn(final String rasterName) throws IOException {
        throw new UnsupportedOperationException("Waiting for a proper implementation");
    }

    /**
     * @see org.geotools.arcsde.session.ISession#getRasterColumns()
     */
    public List<String> getRasterColumns() throws IOException {
        checkActive();
        List<String> rasterNames = issue(new Command<List<String>>() {
            @SuppressWarnings("unchecked")
            @Override
            public List<String> execute(final ISession session, final SeConnection connection)
                    throws SeException, IOException {

                final Vector<SeRasterColumn> rasterColumns = connection.getRasterColumns();
                List<String> names = new ArrayList<String>(rasterColumns.size());

                for (SeRasterColumn col : rasterColumns) {
                    names.add(col.getQualifiedTableName());
                }
                return names;
            }
        });
        return rasterNames;
    }

    /**
     * @see ISession#getTable(java.lang.String)
     */
    public SeTable getTable(final String tableName) throws IOException {
        checkActive();
        if (!cachedTables.containsKey(tableName)) {
            issue(new Command<Void>() {
                @Override
                public Void execute(final ISession session, final SeConnection connection)
                        throws SeException, IOException {
                    synchronized (cachedTables) {
                        if (!cachedTables.containsKey(tableName)) {
                            SeTable table = new SeTable(connection, tableName);
                            try {
                                table.describe();
                            } catch (SeException e) {
                                throw new NoSuchElementException("Table '" + tableName
                                        + "' not found");
                            }
                            cachedTables.put(tableName, table);
                        }
                    }
                    return null;
                }
            });
        }

        SeTable seTable = (SeTable) cachedTables.get(tableName);
        if (seTable == null) {
            throw new NoSuchElementException("Table '" + tableName + "' not found");
        }
        return seTable;
    }

    /**
     * @see ISession#startTransaction()
     */
    public void startTransaction() throws IOException {
        checkActive();
        issue(new Command<Void>() {
            @Override
            public Void execute(final ISession session, final SeConnection connection)
                    throws SeException, IOException {
                connection.setTransactionAutoCommit(0);
                connection.startTransaction();
                transactionInProgress = true;
                return null;
            }
        });
    }

    /**
     * @see ISession#commitTransaction()
     */
    public void commitTransaction() throws IOException {
        checkActive();
        issue(new Command<Void>() {
            @Override
            public Void execute(final ISession session, final SeConnection connection)
                    throws SeException, IOException {
                connection.commitTransaction();
                return null;
            }
        });
        transactionInProgress = false;
    }

    /**
     * @see ISession#isTransactionActive()
     */
    public boolean isTransactionActive() {
        checkActive();
        return transactionInProgress;
    }

    /**
     * @see ISession#rollbackTransaction()
     */
    public void rollbackTransaction() throws IOException {
        checkActive();
        try {
            issue(new Command<Void>() {
                @Override
                public Void execute(final ISession session, final SeConnection connection)
                        throws SeException, IOException {
                    connection.rollbackTransaction();
                    return null;
                }
            });
        } finally {
            transactionInProgress = false;
        }
    }

    /**
     * @see ISession#dispose()
     */
    public void dispose() throws IllegalStateException {
        checkActive();
        if (transactionInProgress) {
            throw new IllegalStateException(
                    "Transaction is in progress, should commit or rollback before closing");
        }

        try {
            this.pool.returnObject(this);
            if (LOGGER.isLoggable(Level.FINER)) {
                LOGGER.finer("<-- " + toString() + " retured to pool. Active: "
                        + pool.getNumActive() + ", idle: " + pool.getNumIdle());
            }
        } catch (Exception e) {
            LOGGER.log(Level.SEVERE, e.getMessage(), e);
        }
    }

    @Override
    public String toString() {
        return "Session[" + sessionId + "]";
    }

    /**
     * Actually closes the connection, called when the session is discarded from the pool
     */
    void destroy() {
        LOGGER.fine("Destroying connection " + toString());
        try {
            issue(new Command<Void>() {
                @Override
                public Void execute(final ISession session, final SeConnection connection)
                        throws SeException, IOException {
                    connection.close();
                    LOGGER.fine(session.toString() + " successfully closed");
                    return null;
                }
            });
        } catch (Exception e) {
            LOGGER.log(Level.FINE, "closing connection " + toString(), e);
        } finally {
            taskExecutor.shutdown();
        }
    }

    /**
     * @see ISession#equals(java.lang.Object)
     */
    @Override
    public boolean equals(Object other) {
        return other == this;
    }

    /**
     * @see ISession#hashCode()
     */
    @Override
    public int hashCode() {
        return 17 ^ this.config.hashCode();
    }

    /**
     * @see ISession#getLayers()
     */
    @SuppressWarnings("unchecked")
    public List<SeLayer> getLayers() throws IOException {
        return issue(new Command<List<SeLayer>>() {
            @Override
            public List<SeLayer> execute(final ISession session, final SeConnection connection)
                    throws SeException, IOException {
                return connection.getLayers();
            }
        });
    }

    /**
     * @see ISession#getUser()
     */
    public String getUser() throws IOException {
        return issue(new Command<String>() {
            @Override
            public String execute(final ISession session, final SeConnection connection)
                    throws SeException, IOException {
                return connection.getUser();
            }
        });
    }

    /**
     * @see ISession#getRelease()
     */
    public SeRelease getRelease() throws IOException {
        return issue(new Command<SeRelease>() {
            @Override
            public SeRelease execute(final ISession session, final SeConnection connection)
                    throws SeException, IOException {
                return connection.getRelease();
            }
        });
    }

    /**
     * @see ISession#getDatabaseName()
     */
    public String getDatabaseName() throws IOException {
        return issue(new Command<String>() {
            @Override
            public String execute(final ISession session, final SeConnection connection)
                    throws SeException, IOException {
                return connection.getDatabaseName();
            }
        });
    }

    public SeDBMSInfo getDBMSInfo() throws IOException {
        return issue(new Command<SeDBMSInfo>() {
            @Override
            public SeDBMSInfo execute(final ISession session, final SeConnection connection)
                    throws SeException, IOException {
                return connection.getDBMSInfo();
            }
        });
    }

    //
    // Factory methods that make use of internal connection
    // Q: How "long" are these objects good for? until the connection closes -
    // or longer...
    //
    /**
     * @see ISession#createSeLayer()
     */
    public SeLayer createSeLayer() throws IOException {
        return issue(new Command<SeLayer>() {
            @Override
            public SeLayer execute(final ISession session, final SeConnection connection)
                    throws SeException, IOException {
                return new SeLayer(connection);
            }
        });
    }

    /**
     * @see ISession#createSeRegistration(java.lang.String)
     */
    public SeRegistration createSeRegistration(final String typeName) throws IOException {
        return issue(new Command<SeRegistration>() {
            @Override
            public SeRegistration execute(final ISession session, final SeConnection connection)
                    throws SeException, IOException {
                return new SeRegistration(connection, typeName);
            }
        });
    }

    /**
     * @see ISession#createSeTable(java.lang.String)
     */
    public SeTable createSeTable(final String qualifiedName) throws IOException {
        return issue(new Command<SeTable>() {
            @Override
            public SeTable execute(final ISession session, final SeConnection connection)
                    throws SeException, IOException {
                return new SeTable(connection, qualifiedName);
            }
        });
    }

    /**
     * @see ISession#createSeInsert()
     */
    public SeInsert createSeInsert() throws IOException {
        return issue(new Command<SeInsert>() {
            @Override
            public SeInsert execute(final ISession session, final SeConnection connection)
                    throws SeException, IOException {
                return new SeInsert(connection);
            }
        });
    }

    /**
     * @see ISession#createSeUpdate()
     */
    public SeUpdate createSeUpdate() throws IOException {
        return issue(new Command<SeUpdate>() {
            @Override
            public SeUpdate execute(final ISession session, final SeConnection connection)
                    throws SeException, IOException {
                return new SeUpdate(connection);
            }
        });
    }

    /**
     * @see ISession#createSeDelete()
     */
    public SeDelete createSeDelete() throws IOException {
        return issue(new Command<SeDelete>() {
            @Override
            public SeDelete execute(final ISession session, final SeConnection connection)
                    throws SeException, IOException {
                return new SeDelete(connection);
            }
        });
    }

    /**
     * @see ISession#createSeRasterColumn()
     */
    public SeRasterColumn createSeRasterColumn() throws IOException {
        return issue(new Command<SeRasterColumn>() {
            @Override
            public SeRasterColumn execute(final ISession session, final SeConnection connection)
                    throws SeException, IOException {
                return new SeRasterColumn(connection);
            }
        });
    }

    /**
     * @see ISession#createSeRasterColumn(com.esri.sde.sdk.client.SeObjectId)
     */
    public SeRasterColumn createSeRasterColumn(final SeObjectId rasterColumnId) throws IOException {
        return issue(new Command<SeRasterColumn>() {
            @Override
            public SeRasterColumn execute(final ISession session, final SeConnection connection)
                    throws SeException, IOException {
                return new SeRasterColumn(connection, rasterColumnId);
            }
        });
    }

    /**
     * @see ISession#describe(java.lang.String)
     */
    public SeColumnDefinition[] describe(final String tableName) throws IOException {
        final SeTable table = getTable(tableName);
        return describe(table);
    }

    /**
     * @see ISession#describe(com.esri.sde.sdk.client.SeTable)
     */
    public SeColumnDefinition[] describe(final SeTable table) throws IOException {
        return issue(new Command<SeColumnDefinition[]>() {
            @Override
            public SeColumnDefinition[] execute(final ISession session,
                    final SeConnection connection) throws SeException, IOException {
                return table.describe();
            }
        });
    }

    /**
     * @see ISession#fetch(com.esri.sde.sdk.client.SeQuery)
     */
    public SdeRow fetch(final SeQuery query) throws IOException {
        return fetch(query, new SdeRow((GeometryFactory) null));
    }

    public SdeRow fetch(final SeQuery query, final SdeRow currentRow) throws IOException {
        return issue(new Command<SdeRow>() {
            @Override
            public SdeRow execute(final ISession session, final SeConnection connection)
                    throws SeException, IOException {
                SeRow row = query.fetch();
                if (row == null) {
                    return null;
                } else {
                    currentRow.setRow(row);
                }
                return currentRow;
            }
        });
    }

    /**
     * @see ISession#close(com.esri.sde.sdk.client.SeState)
     */
    public void close(final SeState state) throws IOException {
        issue(new Command<Void>() {
            @Override
            public Void execute(final ISession session, final SeConnection connection)
                    throws SeException, IOException {
                state.close();
                return null;
            }
        });
    }

    /**
     * @see ISession#close(com.esri.sde.sdk.client.SeStreamOp)
     */
    public void close(final SeStreamOp stream) throws IOException {
        issue(new Command<Void>() {
            @Override
            public Void execute(final ISession session, final SeConnection connection)
                    throws SeException, IOException {
                stream.close();
                return null;
            }
        });
    }

    /**
     * @see ISession#createState(com.esri.sde.sdk.client.SeObjectId)
     */
    public SeState createState(final SeObjectId stateId) throws IOException {
        return issue(new Command<SeState>() {
            @Override
            public SeState execute(final ISession session, final SeConnection connection)
                    throws SeException, IOException {
                return new SeState(connection, stateId);
            }
        });
    }

    /**
     * @see ISession#createSeQuery()
     */
    public SeQuery createSeQuery() throws IOException {
        return issue(new Command<SeQuery>() {
            @Override
            public SeQuery execute(final ISession session, final SeConnection connection)
                    throws SeException, IOException {
                return new SeQuery(connection);
            }
        });
    }

    /**
     * @see ISession#createSeQuery(java.lang.String[], com.esri.sde.sdk.client.SeSqlConstruct)
     */
    public SeQuery createSeQuery(final String[] propertyNames, final SeSqlConstruct sql)
            throws IOException {

        return issue(new Command<SeQuery>() {
            @Override
            public SeQuery execute(final ISession session, final SeConnection connection)
                    throws SeException, IOException {
                return new SeQuery(connection, propertyNames, sql);
            }
        });
    }

    /**
     * @see ISession#createAndExecuteQuery(java.lang.String[],
     *      com.esri.sde.sdk.client.SeSqlConstruct)
     */
    public SeQuery createAndExecuteQuery(final String[] propertyNames, final SeSqlConstruct sql)
            throws IOException {
        return issue(new Command<SeQuery>() {
            @Override
            public SeQuery execute(final ISession session, final SeConnection connection)
                    throws SeException, IOException {
                SeQuery query = new SeQuery(connection, propertyNames, sql);
                query.prepareQuery();
                query.execute();
                return query;
            }
        });
    }

    /**
     * @see ISession#getDefaultVersion()
     */
    public SeVersion getDefaultVersion() throws IOException {
        return issue(new GetVersionCommand(SeVersion.SE_QUALIFIED_DEFAULT_VERSION_NAME));
    }

    /**
     * Creates either a direct child state of parentStateId, or a sibling being an exact copy of
     * parentStatId if either the state can't be closed because its in use or parentStateId does not
     * belong to the current user.
     */
    public SeState createChildState(final long parentStateId) throws IOException {
        return issue(new Command<SeState>() {
            @Override
            public SeState execute(ISession session, SeConnection connection) throws SeException,
                    IOException {
                SeState parentState = new SeState(connection, new SeObjectId(parentStateId));

                SeState realParent = null;

                boolean mergeParentToRealParent = false;

                if (parentState.isOpen()) {
                    // only closed states can have child states
                    try {
                        parentState.close();
                        realParent = parentState;
                    } catch (SeException e) {
                        final int errorCode = e.getSeError().getSdeError();
                        if (SeError.SE_STATE_INUSE == errorCode
                                || SeError.SE_NO_PERMISSIONS == errorCode) {
                            // it's not our state or somebody's editing it so we
                            // need to clone the parent,
                            // starting from the parent of the parent
                            realParent = new SeState(connection, parentState.getParentId());
                            mergeParentToRealParent = true;
                        } else {
                            throw e;
                        }
                    }
                } else {
                    realParent = parentState;
                }

                // create the new state
                SeState newState = new SeState(connection);
                newState.create(realParent.getId());

                if (mergeParentToRealParent) {
                    // a sibling of parentStateId was created instead of a
                    // child, we need to merge the changes
                    // in parentStateId to the new state so they refer to the
                    // same content.
                    // SE_state_merge applies changes to a parent state to
                    // create a new merged state.
                    // The new state is the child of the parent state with the
                    // changes of the second state.
                    // Both input states must have the same parent state.
                    // When a row has been changed in both parent and second
                    // states, the row from the changes state is used.
                    // The parent and changes states must be open or owned by
                    // the current user unless the current user is the ArcSDE
                    // DBA.
                    newState.merge(realParent.getId(), parentState.getId());
                }

                return newState;
            }
        });
    }

    private static final class CreateSessionCommand extends Command<SeConnection> {
        private final ArcSDEConnectionConfig config;

        private CreateSessionCommand(final ArcSDEConnectionConfig config) {
            this.config = config;
        }

        @Override
        public SeConnection execute(final ISession session, final SeConnection connection)
                throws SeException, IOException {
            String serverName = config.getServerName();
            int portNumber = config.getPortNumber().intValue();
            String databaseName = config.getDatabaseName();
            String userName = config.getUserName();
            String userPassword = config.getPassword();

            SeConnection conn;
            try {
                conn = new SeConnection(serverName, portNumber, databaseName, userName,
                        userPassword);
            } catch (SeException e) {
                throw new ArcSdeException("Can't create connection to " + serverName, e);
            } catch (RuntimeException e) {
                throw (IOException) new IOException("Can't create connection to " + serverName)
                        .initCause(e);
            }
            conn.setConcurrency(SeConnection.SE_ONE_THREAD_POLICY);
            return conn;
        }
    }
}
