package org.tecgraf.tdk.cache.query;

import java.util.Iterator;

import org.geotools.data.DefaultQuery;
import org.geotools.data.Query;
import org.geotools.geometry.jts.ReferencedEnvelope;
import org.opengis.filter.And;
import org.opengis.filter.Filter;
import org.opengis.filter.expression.Expression;
import org.opengis.filter.expression.Literal;
import org.opengis.filter.expression.PropertyName;
import org.opengis.filter.spatial.BBOX;
import org.opengis.filter.spatial.BinarySpatialOperator;
import org.opengis.filter.spatial.Contains;
import org.opengis.filter.spatial.Crosses;
import org.opengis.filter.spatial.DWithin;
import org.opengis.filter.spatial.Equals;
import org.opengis.filter.spatial.Intersects;
import org.opengis.filter.spatial.Overlaps;
import org.opengis.filter.spatial.Touches;
import org.opengis.filter.spatial.Within;

import com.vividsolutions.jts.geom.Envelope;
import com.vividsolutions.jts.geom.Geometry;

/**
 * This class represents a simplified {@link QueryAnalyzer} that do not 
 * process residues completely. If the original {@link Query} has {@link Filter#INCLUDE}
 * as filter, any query is a sub-query and residue has {@link Filter#EXCLUDE}
 * as filter. If original {@link Query} filter is not {@link Filter#INCLUDE}, then 
 * all queries are not sub-query and residue is always the given {@link Query} itself.
 * @author fmoura
 *
 */
public class DefaultQueryAnalyzer implements QueryAnalyzer {

	private Query _query;

	public DefaultQueryAnalyzer(Query query) 
	{
		setQuery(query);
	}

	@Override
	public Envelope getEnvelope() {
		Filter filter = _query.getFilter();
		return getEnvelope(filter);
	}
        
    private Envelope getEnvelope(Filter filter)
    {
    	Envelope result = new Envelope();
		
        if (filter instanceof And) 
        {
            ReferencedEnvelope bounds = new ReferencedEnvelope();
            for (Iterator<Filter> iter = ((And) filter).getChildren().iterator(); iter.hasNext();) 
            {
                Filter f = (Filter) iter.next();
                Envelope e = getEnvelope(f);
                bounds.expandToInclude(e);
            }
            result = bounds;
        } 
        else if (filter instanceof BinarySpatialOperator) 
        {
            BinarySpatialOperator gf = (BinarySpatialOperator) filter;
            if (filterIsSupported(gf))
            {
                Expression lg = gf.getExpression1();
                Expression rg = gf.getExpression2();
                if (lg instanceof Literal)
                {
                	Object value = ((Literal) lg).getValue();
                    if (rg instanceof PropertyName)
                    {
                    	result = retrieveEnvelopeFromValue(value);
                    }
                } 
                else if (rg instanceof Literal) 
                {
                	Object value = ((Literal) rg).getValue();
                    if (lg instanceof PropertyName)
                    {
                    	result = retrieveEnvelopeFromValue(value);
                    }
                        
                }
            }
        }
        return result;
    }
    
    private Envelope retrieveEnvelopeFromValue(Object value)
    {
    	if(value instanceof Geometry)
    	{
    		Geometry g = (Geometry) value;
    		return g.getEnvelopeInternal();
    	}
    	else if(value instanceof org.opengis.geometry.Geometry)
    	{
    		org.opengis.geometry.Geometry g = (org.opengis.geometry.Geometry) value;
    		return getEnvelope(g);
    	}
    	return new Envelope();
    }
    

	private Envelope getEnvelope(org.opengis.geometry.Geometry g) {
		Envelope result;
		org.opengis.geometry.Envelope envelope = g.getEnvelope();
		double x1 = envelope.getMinimum(0);
		double x2 = envelope.getMaximum(0);
		double y1 = envelope.getMinimum(1);
		double y2 = envelope.getMaximum(1);
		result = new Envelope(x1,x2,y1,y2);
		return result;
	}
    
    private boolean filterIsSupported(BinarySpatialOperator f)
    {
        if ((f instanceof BBOX)
            || (f instanceof Contains)
            || (f instanceof Crosses)
            || (f instanceof DWithin)
            || (f instanceof Equals)
            || (f instanceof Intersects)
            || (f instanceof Overlaps)
            || (f instanceof Touches)
            || (f instanceof Within)
            )
            return true;
        else
            return false;
    }

	@Override
	public Query getQuery() {
		return _query;
	}

	@Override
	public Query getResidue(Query query) {
		
		if(isSubQuery(query)) return new DefaultQuery(_query.getTypeName(),Filter.EXCLUDE);
		
		return query;
		
	}

	@Override
	public boolean isSubQuery(Query query) {
		if(query == null) throw new IllegalArgumentException("query can't be null");
		
		if(query.equals(_query)) return true;
		
		if(!query.getTypeName().equalsIgnoreCase(_query.getTypeName())) return false;
		
		if(query.getFilter().equals(Filter.EXCLUDE)) return true; // "nothing" is always party of "something"
		if(_query.getFilter().equals(Filter.INCLUDE)) return true;// "anything" is part of "everything"  
		
		return false;
	}

	@Override
	public void setQuery(Query query) {
		if(query == null) throw new IllegalArgumentException("query can't be null");
		_query = query;

	}

}
