package org.geotools.renderer.style.customshape;

import java.awt.Shape;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Collection;

import org.apache.log4j.Logger;


/**
 * This class is the base class for shape creators that have a specification string in the form 'shapeName?paramter1=value1&parameter2=value2&...'.
 * 
 * @author fmoura
 *
 */
public abstract class AbstractShapeCreator implements ShapeCreator {

	private static Logger _logger =  Logger.getLogger(AbstractShapeCreator.class); 
	
	/**
	 * A general parameter class that represents the parameters present in the shape specification string.
	 * @author fmoura
	 *
	 * @param <V>
	 */
	protected static class Parameter<V> {
		private final String name;
		private final V value;
	
		public Parameter(String name, V value) {
			this.name = name;
			this.value = value;
		}
	
		public String getName() {
			return name;
		}
	
		public V getValue() {
			return value;
		}
	}
	
	@SuppressWarnings("unchecked")
    @Override
	public Shape createShape(String shapeSpecification) {
		String name;
		
		final int questionMarkIndex = shapeSpecification.indexOf("?"); 
		
		if (questionMarkIndex >= 0)
        {
        	name = shapeSpecification.substring(0, questionMarkIndex);
        }
        else
        {
        	name = shapeSpecification;
        }
		
		// remove a trailing backslash, if there is one
    	if (name.endsWith("/")) {
    		name = name.substring(0, name.length() - 1);
    	}

		if(!name.equalsIgnoreCase(getShapeName())) return null;
        
		Collection<Parameter<String>> parameters = new ArrayList<Parameter<String>>();
    	
    	parseParameters(shapeSpecification.substring(questionMarkIndex + 1), parameters);
		
        // copy from the collection to an array so we can call 
        // createShape using varargs.
        Parameter<String>[] parametersArray = 
        	new Parameter[parameters.size()];
        int index = 0;
        for (Parameter<String> parameter : parameters) {
        	parametersArray[index] = parameter;
        	++index;
        }
	        
        return createShape(parametersArray);
	}
	
	private void parseParameters(String urlEncodedParameters,
			Collection<Parameter<String>> parameters) {

		// check for an empty parameter list.
		if ("".equals(urlEncodedParameters)) {
			return;
		}

		final String encoding = "UTF-8";
		String[] encodedPairs = urlEncodedParameters.split("&");
	
		for (String encodedPair : encodedPairs) {
			String[] nameAndValue = encodedPair.split("=");
			if (nameAndValue.length == 2) {
				try {
					String name = URLDecoder.decode(nameAndValue[0], encoding);
					String value = URLDecoder.decode(nameAndValue[1], encoding);
					
					parameters.add(new Parameter<String>(name, value));
				} catch (UnsupportedEncodingException e) {
					_logger.warn("Failed to decode a custom shape parameter.", e);
				}
			} else {
				_logger.warn("Malformed custom shape parameter: \"" + encodedPair + "\".");
			}
		}
	}
	
	/**
	 * Creates a shape instance based on the given parameters list. This list is retrieved from the
	 * specification string. 
	 * @param parameters The parameters list read from the specification string
	 * @return The shape instance created based in the given parameters list.
	 */
	protected abstract Shape createShape(Parameter<String>... parameters);
	
	
	/**
	 * Builds a specification string for a shape instance that would be built with the given parameters.
	 * It follows the form 'shapeName?parameter1=value&parameter2=value&...' .
	 * @param parameters The parameters list to be used to complete the specification string.
	 * @return the specification string for the shape instance that would be built with the given parameters
	 */
	protected String buildShapeSpecification(Parameter<String>... parameters)
	{
		String specification = getShapeName();
		if(parameters.length > 0)
		{
			specification += "?";
			boolean first = true;
			for(Parameter<String> parameter : parameters)
			{
				if(!first)
				{
					specification += "&";
				}
				first = false;
				
				specification += parameter.getName()+"="+parameter.getValue();
			}
		}
		
		return specification;
	}

}
