package org.geotools.renderer.style.customshape;

import java.awt.Shape;
import java.awt.geom.GeneralPath;
import java.awt.geom.Rectangle2D;

import org.apache.log4j.Logger;
import org.geotools.renderer.style.shape.ExplicitBoundsShape;

/**
 * This ShapeCreator builds custom shapes to be used in hatch fills. Normal
 * usage generates a single line shape based on an angle parameter.
 * <p>
 * The user can also optionally provide additional x and y repetition
 * parameters, which indicate how many times the line is to be repeated in each
 * direction (i.e., how many tiles). This is interesting for use in
 * PointSymbolizers and especially for more efficient hatch fills, particularly
 * when using vector rendering (e.g., for printing purposes).
 * <p>
 * The shape specification string follows the form
 * 'hatch?angle=0.0&xrepetitions=1&yrepetitions=1', where '0.0' values can be
 * replaced by any double value and '1' values can be replaced by any positive
 * integer value. The angle value is in degrees and the x and y repetitions
 * stand for the number of repeating tiles in each direction.
 * <p>
 * The spacing between each hatch line is actually controlled by the size
 * parameter of the Graphic where this shape mark is used. Within the shape
 * itself, the distance between each line is 1/yrepetitions in the y direction.
 * This distance will then be multiplied by the Graphic's size parameter, thus
 * arriving at a default of 16 pixels distance in the y direction. Notice
 * however that the real separation between the lines (perpendicular to the
 * lines) is dependent on the angle, and is given by y_distance*cos(angle).
 * 
 * @author fmoura
 * @author milton
 */
public class HatchShapeCreator extends AbstractShapeCreator
{

    private static final double MIN_SIN_COS_VALUE = 0.01;
    private static final String SHAPE_NAME = "hatch";
    private static final String ANGLE_PARAMETER_NAME = "angle";
    private static final String XREPETITIONS_PARAMETER_NAME = "xrepetitions";
    private static final String YREPETITIONS_PARAMETER_NAME = "yrepetitions";
    private static Logger _logger = Logger.getLogger(HatchShapeCreator.class);

    @Override
    protected Shape createShape(Parameter<String>... parameters)
    {

        String angleStr = null;
        String xRepetitionsStr = null;
        String yRepetitionsStr = null;

        for (Parameter<String> parameter : parameters)
        {
            if (parameter.getName().equalsIgnoreCase(ANGLE_PARAMETER_NAME))
            {
                angleStr = parameter.getValue();
                continue;
            }
            if (parameter.getName().equalsIgnoreCase(XREPETITIONS_PARAMETER_NAME))
            {
                xRepetitionsStr = parameter.getValue();
            }
            if (parameter.getName().equalsIgnoreCase(YREPETITIONS_PARAMETER_NAME))
            {
                yRepetitionsStr = parameter.getValue();
            }
        }

        if (angleStr == null)
        {
            _logger.warn("Missing required param 'angle'");
            return null;
        }

        double angle = 0.0;
        int xRepetitions = 1;
        int yRepetitions = 1;

        try
        {
            angle = Double.parseDouble(angleStr);
        }
        catch (NumberFormatException e)
        {
            _logger.warn("Parameter 'angle' in invalid format", e);
            return null;
        }
        if (xRepetitionsStr != null)
        {
            try
            {
                xRepetitions = Integer.parseInt(xRepetitionsStr);
            }
            catch (NumberFormatException e)
            {
                _logger.warn("Parameter 'xrepetitions' in invalid format", e);
                return null;
            }
        }
        if (yRepetitionsStr != null)
        {
            try
            {
                yRepetitions = Integer.parseInt(yRepetitionsStr);
                if (yRepetitions <= 0)
                    throw new NumberFormatException("Must be positive (was '" + yRepetitions + "')");
            }
            catch (NumberFormatException e)
            {
                _logger.warn("Parameter 'yrepetitions' in invalid format", e);
                return null;
            }
        }

        return createShape(angle, xRepetitions, yRepetitions);
    }

    /**
     * Creates a hatch basic shape based on the given angle.
     * 
     * @param angle The angle of the hatch lines in degrees.
     * @return The basic {@link Shape} for drawing a hatch with the given angle.
     */
    public Shape createShape(double angle)
    {
        return createShape(angle, 1, 1);
    }

    /**
     * Creates a hatch basic shape based on the given angle and number of
     * repetitions in the x (horizontal) and y (vertical) axes.
     * 
     * @param angle The angle of the hatch lines in degrees.
     * @param xRepetitions The number of horizontal repetitions for the hatch
     *            pattern.
     * @param yRepetitions The number of vertical repetitions for the hatch
     *            pattern.
     * @return The basic {@link Shape} for drawing a hatch with the given angle
     *         and x and y repetitions.
     */
    public Shape createShape(double angle, int xRepetitions, int yRepetitions)
    {
        // checks repetition values: must be positive
        if (xRepetitions <= 0)
        {
            _logger.warn("Parameter 'xrepetitions' must be positive (was '" + xRepetitions + "')");
            return null;
        }
        if (yRepetitions <= 0)
        {
            _logger.warn("Parameter 'yrepetitions' must be positive (was '" + yRepetitions + "')");
            return null;
        }

        double sin = Math.sin(Math.PI * angle / 180.0);
        double cos = Math.cos(Math.PI * angle / 180.0);

        if (Math.abs(sin) <= MIN_SIN_COS_VALUE) // horizontal line
        {
            return createShapeInternalHorizontal(xRepetitions, yRepetitions);
        }
        else if (Math.abs(cos) <= MIN_SIN_COS_VALUE) // vertical line
        {
            return createShapeInternalVertical(xRepetitions, yRepetitions);
        }
        else
        // normal hatch line
        {
            return createShapeInternalAngle(sin, cos, xRepetitions, yRepetitions);
        }
    }

    /**
     * Creates a shape with horizontal lines
     * 
     * @param xRepetitions number of "tiles" in the x direction. The shape
     *            should have one single horizontal line spanning all of those
     *            tiles.
     * @param yRepetitions number of tiles in the y direction. The shape will
     *            have one line for each tile.
     * @return the Shape.
     */
    protected Shape createShapeInternalHorizontal(int xRepetitions, int yRepetitions)
    {
        double width = 1;
        double height = 1;
        double tileWidth = width;
        double tileHeight = height / yRepetitions;

        GeneralPath internalShape = new GeneralPath();
        double y = -height / 2 + tileHeight / 2;
        for (int j = 0; j < yRepetitions; j++)
        {
            internalShape.moveTo(-tileWidth / 2., y);
            internalShape.lineTo(tileWidth / 2., y);
            y += tileHeight;
        }

        ExplicitBoundsShape shape = new ExplicitBoundsShape(internalShape);
        shape.setBounds(new Rectangle2D.Double(-width / 2.0, -height / 2.0, width, height));
        return shape;
    }

    /**
     * Creates a shape with vertical lines
     * 
     * @param xRepetitions number of tiles in the x direction. The shape will
     *            have one line for each tile.
     * @param yRepetitions number of "tiles" in the y direction. The shape
     *            should have one single vertical line spanning all of those
     *            tiles.
     * @return the Shape.
     */
    protected Shape createShapeInternalVertical(int xRepetitions, int yRepetitions)
    {
        double width = 1;
        double height = 1;
        double tileWidth = width / xRepetitions;
        double tileHeight = height;

        GeneralPath internalShape = new GeneralPath();
        double x = -width / 2 + tileWidth / 2;
        for (int i = 0; i < xRepetitions; i++)
        {
            internalShape.moveTo(x, -tileHeight / 2.);
            internalShape.lineTo(x, tileHeight / 2.);
            x += tileWidth;
        }

        ExplicitBoundsShape shape = new ExplicitBoundsShape(internalShape);
        shape.setBounds(new Rectangle2D.Double(-width / 2.0, -height / 2.0, width, height));
        return shape;
    }

    /**
     * Creates a shape for angled lines.
     * 
     * @param xRepetitions number of tiles in the x direction.
     * @param yRepetitions number of tiles in the y direction.
     * @return the Shape.
     */
    protected Shape createShapeInternalAngle(double sin, double cos, int xRepetitions, int yRepetitions)
    {
        double width = Math.abs(cos / sin);
        double height = 1;
        double tileWidth = width / xRepetitions;
        double tileHeight = height / yRepetitions;

        double backslashFactor = 1;
        if (sin * cos < 0)
            backslashFactor = -1;

        GeneralPath internalShape = new GeneralPath();

        double xStart = -width / 2;
        double yStart = -height / 2 * backslashFactor;
        for (int i = 0; i < xRepetitions; i++)
        {
            int tileRepetitions = Math.min((xRepetitions - i), yRepetitions);
            double xEnd = xStart + tileRepetitions * tileWidth;
            double yEnd = yStart + tileRepetitions * tileHeight * backslashFactor;
            internalShape.moveTo(xStart, yStart);
            internalShape.lineTo(xEnd, yEnd);
            xStart += tileWidth;
        }

        xStart = -width / 2;
        yStart = (-height / 2 + tileHeight) * backslashFactor;
        for (int j = 1; j < yRepetitions; j++)
        {
            int tileRepetitions = Math.min(xRepetitions, (yRepetitions - j));
            double xEnd = xStart + tileRepetitions * tileWidth;
            double yEnd = yStart + tileRepetitions * tileHeight * backslashFactor;
            internalShape.moveTo(xStart, yStart);
            internalShape.lineTo(xEnd, yEnd);
            yStart += tileHeight * backslashFactor;
        }

        ExplicitBoundsShape shape = new ExplicitBoundsShape(internalShape);
        shape.setBounds(new Rectangle2D.Double(-width / 2.0, -height / 2.0, width, height));
        return shape;
    }

    @Override
    public String getShapeName()
    {
        return SHAPE_NAME;
    }

    /**
     * This method is a utility to help creating a shape specification string
     * for style description purposes.
     * 
     * @param angle The angle of the hatch lines in degrees.
     * @param xRepetitions The number of horizontal repetitions for the hatch
     *            pattern
     * @param yRepetitions The number of vertical repetitions for the hatch
     *            pattern
     * @return The String that represents this shape specification
     */
    @SuppressWarnings("unchecked")
    public String buildShapeSpecification(double angle, int xRepetitions, int yRepetitions)
    {
        Parameter<String> angleParameter = new Parameter<String>(ANGLE_PARAMETER_NAME, String.valueOf(angle));
        Parameter<String> xRepetitionsParamenter = new Parameter<String>(XREPETITIONS_PARAMETER_NAME, String
                .valueOf(xRepetitions));
        Parameter<String> yRepetitionsParamenter = new Parameter<String>(YREPETITIONS_PARAMETER_NAME, String
                .valueOf(yRepetitions));
        return buildShapeSpecification(angleParameter, xRepetitionsParamenter, yRepetitionsParamenter);
    }

}
