/*
 *    GeoTools - The Open Source Java GIS Toolkit
 *    http://geotools.org
 * 
 *    (C) 2004-2008, Open Source Geospatial Foundation (OSGeo)
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License as published by the Free Software Foundation;
 *    version 2.1 of the License.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 */
package org.geotools.renderer.style.shape;

import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.PathIterator;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;

/**
 * Decorator on top of the {@link Shape}. It extends the Shape interface to
 * include a method 'setBounds' for explicitly defining a bounding box (which is
 * not necessarily associated with the actual shape's bounds).
 * 
 * @author fmoura
 */
public class ExplicitBoundsShape implements Shape {

	private Shape _shape;
	private Rectangle2D _bounds = null;

	/**
	 * The Constructor
	 * 
	 * @param shape
	 *            The actual shape on top of which this decorator will stand.
	 */
	public ExplicitBoundsShape(Shape shape) {
		if (shape == null)
			throw new IllegalArgumentException("Shape can't be null.");
		_shape = shape;
	}

	/**
	 * Sets the explicitly defined bounds for this shape.
	 * 
	 * @param bounds
	 */
	public void setBounds(Rectangle2D bounds) {
		_bounds = bounds;
	}

	public boolean contains(double x, double y, double w, double h) {
		return _shape.contains(x, y, w, h);
	}

	public boolean contains(double x, double y) {
		return _shape.contains(x, y);
	}

	public boolean contains(Point2D p) {
		return _shape.contains((Point2D) p);
	}

	public boolean contains(Rectangle2D r) {
		return _shape.contains((Rectangle2D) r);
	}

	/**
	 * Returns the explicitly defined bounds for this shape. If no bounds were
	 * explicitly set, it delegates the call to the actual shape.
	 * 
	 * @return the Rectangle representing the Shape's bounding box.
	 * @see Shape
	 */
	public Rectangle getBounds() {
		if (_bounds != null)
			return new Rectangle((int) _bounds.getMinX(), (int) _bounds
					.getMinY(), (int) _bounds.getWidth(), (int) _bounds
					.getHeight());
		return _shape.getBounds();
	}

	/**
	 * Returns the explicitly defined bounds for this shape. If no bounds were
	 * explicitly set, it delegates the call to the actual shape.
	 * 
	 * @return the Rectangle2D representing the Shape's bounding box.
	 * @see Shape
	 */
	public Rectangle2D getBounds2D() {
		if (_bounds != null)
			return _bounds;
		return _shape.getBounds2D();
	}

	public PathIterator getPathIterator(AffineTransform at, double flatness) {
		return _shape.getPathIterator(at, flatness);
	}

	public PathIterator getPathIterator(AffineTransform at) {
		return _shape.getPathIterator(at);
	}

	public boolean intersects(double x, double y, double w, double h) {
		return _shape.intersects(x, y, w, h);
	}

	public boolean intersects(Rectangle2D r) {
		return _shape.intersects(r);
	}

	@Override
	public boolean equals(Object obj) {
		if (obj instanceof ExplicitBoundsShape) {
			ExplicitBoundsShape other = (ExplicitBoundsShape) obj;
			boolean result = _shape.equals(other._shape);
			if (_bounds == null)
				return result & (other._bounds == null);
			return result & _bounds.equals(other._bounds);
		} else if (obj instanceof Shape) {
			if (_bounds == null)
				return _shape.equals(obj);
			return false;
		}
		return super.equals(obj);

	}

	@Override
	public int hashCode() {
		int hascode = _shape.hashCode();
		if (_bounds != null)
			hascode += hascode * 37 + _bounds.hashCode();
		return hascode;
	}

}
