/**
 * Tecgraf - GIS development team
 * 
 * Tdk Framework
 * Copyright TecGraf 2008(c).
 * 
 * file: TerralibFeatureTypeAdapter.java
 * created: 2009-08-12
 */
package org.geotools.data.terralib;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.geotools.data.terralib.exception.NullArgumentException;
import org.geotools.data.terralib.util.TerralibAttributeHelper;
import org.geotools.data.terralib.util.TypeAttributeMap;
import org.geotools.feature.simple.SimpleFeatureTypeBuilder;
import org.opengis.feature.Property;
import org.opengis.feature.simple.SimpleFeatureType;
import org.opengis.feature.type.AttributeDescriptor;
import org.opengis.feature.type.AttributeType;
import org.opengis.feature.type.GeometryDescriptor;
import org.opengis.feature.type.Name;
import org.opengis.feature.type.PropertyDescriptor;
import org.opengis.filter.Filter;
import org.opengis.referencing.crs.CoordinateReferenceSystem;
import org.opengis.util.InternationalString;

/**
 * TerralibFeatureTypeAdapter is an adapter for a SimpleFeatureType that guarantees
 * that the SimpleFeatureType is compatible to TerralibProvider.
 * It makes necessary changes to attribute names/types to make it compatible.
 * 
 * 2009-08-12
 * Currently the adapter:
 * 
 * * Checks for invalid attribute names and modifies it. If it is a 
 *   reserved word, add a _ to the end. If it has illegal chars or words, 
 *   remove them leaving the rest of the attribute name intact.
 *   
 * * Changes unsupported attribute types to string (and logs a warning). 
 * 
 * @author fabiomano
 * @since TDK3.0.0
 */
public class TerralibFeatureTypeAdapter implements SimpleFeatureType
{
	private SimpleFeatureType _targetFeatureType;
	
	private static Logger _logger = Logger.getLogger(TerralibFeatureTypeAdapter.class);
	
	public TerralibFeatureTypeAdapter(SimpleFeatureType featureType)
	{
		if (featureType == null)
			throw new NullArgumentException("featureType");
		
		_targetFeatureType = buildCompatibleFeatureType(featureType);
	}

	protected SimpleFeatureType buildCompatibleFeatureType(SimpleFeatureType featureType) 
	{
		SimpleFeatureTypeBuilder builder = new SimpleFeatureTypeBuilder();
		builder.setName(featureType.getName());
		builder.setCRS(featureType.getCoordinateReferenceSystem());
		builder.setDescription(featureType.getDescription());
		
		int attrIndex = 0;
		for (AttributeDescriptor a: featureType.getAttributeDescriptors())
		{
			String name = fixAttributeName(a.getLocalName(), attrIndex);
			
			TypeAttributeMap type = TypeAttributeMap.fromAttributeType(a.getType());
			if (type == null)
			{
				_logger.warn("Attribute " + a.getLocalName() + "'s type is not supported by Terralib (" + a.getType().getBinding()+ "). It was changed to String");
				type = TypeAttributeMap.TA_STRING;
			}
			
			AttributeDescriptor attribute = TerralibAttributeHelper.buildType(name, type, a.getType().isIdentified(),a.isNillable(),a.getMaxOccurs(),a.getType().getRestrictions());
			builder.add(attribute);
			attrIndex++;
		}
		
		if (featureType.getGeometryDescriptor() != null)
			builder.setDefaultGeometry(featureType.getGeometryDescriptor().getLocalName());
		
		SimpleFeatureType newType = builder.buildFeatureType();
		
		//copy the user data
		newType.getUserData().putAll(featureType.getUserData());
		return newType;
	}

	/**
	 * Fix the attribute name if it is a reserved name or if it contains illegal words/chars 
	 * @param attrName The attribute name to fix
	 * @param attrIndex 
	 * @return The correct name
	 */
	private String fixAttributeName(String attrName, int attrIndex) 
	{
		for (String reservedName : TerralibDataStore.getReservedNames())
		{
			if (reservedName.equalsIgnoreCase(attrName))
				attrName = attrName + "_";
		}
		
		for (String illegalWord: TerralibDataStore.getIllegalWords())
		{
			if (attrName.contains(illegalWord))
				attrName = attrName.replace(illegalWord, "");
		}
		if (attrName.equals(""))
			attrName = "field" + (attrIndex + 1);
		
		return attrName;
	}

	/*
	 * (non-Javadoc)
	 * @see org.opengis.feature.simple.SimpleFeatureType#getAttributeCount()
	 */
	@Override
	public int getAttributeCount() 
	{
		return _targetFeatureType.getAttributeCount();
	}

	/*
	 * (non-Javadoc)
	 * @see org.opengis.feature.simple.SimpleFeatureType#getAttributeDescriptors()
	 */
	@Override
	public List<AttributeDescriptor> getAttributeDescriptors() {
		return _targetFeatureType.getAttributeDescriptors();
	}

	/*
	 * (non-Javadoc)
	 * @see org.opengis.feature.simple.SimpleFeatureType#getDescriptor(java.lang.String)
	 */
	@Override
	public AttributeDescriptor getDescriptor(String name) {
		return _targetFeatureType.getDescriptor(name);
	}

	/*
	 * (non-Javadoc)
	 * @see org.opengis.feature.simple.SimpleFeatureType#getDescriptor(org.opengis.feature.type.Name)
	 */
	@Override
	public AttributeDescriptor getDescriptor(Name name) {
		return _targetFeatureType.getDescriptor(name);
	}

	/*
	 * (non-Javadoc)
	 * @see org.opengis.feature.simple.SimpleFeatureType#getDescriptor(int)
	 */
	@Override
	public AttributeDescriptor getDescriptor(int index)
			throws IndexOutOfBoundsException {
		return _targetFeatureType.getDescriptor(index);
	}

	/*
	 * (non-Javadoc)
	 * @see org.opengis.feature.simple.SimpleFeatureType#getType(java.lang.String)
	 */
	@Override
	public AttributeType getType(String name) {
		return _targetFeatureType.getType(name);
	}

	/*
	 * (non-Javadoc)
	 * @see org.opengis.feature.simple.SimpleFeatureType#getType(org.opengis.feature.type.Name)
	 */
	@Override
	public AttributeType getType(Name name) {
		return _targetFeatureType.getType(name);
	}

	/*
	 * (non-Javadoc)
	 * @see org.opengis.feature.simple.SimpleFeatureType#getType(int)
	 */
	@Override
	public AttributeType getType(int index) throws IndexOutOfBoundsException {
		return _targetFeatureType.getType(index);
	}

	/*
	 * (non-Javadoc)
	 * @see org.opengis.feature.simple.SimpleFeatureType#getTypeName()
	 */
	@Override
	public String getTypeName() {
		return _targetFeatureType.getTypeName();
	}

	/*
	 * (non-Javadoc)
	 * @see org.opengis.feature.simple.SimpleFeatureType#getTypes()
	 */
	@Override
	public List<AttributeType> getTypes() {
		return _targetFeatureType.getTypes();
	}

	/*
	 * (non-Javadoc)
	 * @see org.opengis.feature.simple.SimpleFeatureType#indexOf(java.lang.String)
	 */
	@Override
	public int indexOf(String name) {
		return _targetFeatureType.indexOf(name);
	}

	/*
	 * (non-Javadoc)
	 * @see org.opengis.feature.simple.SimpleFeatureType#indexOf(org.opengis.feature.type.Name)
	 */
	@Override
	public int indexOf(Name name) {
		return _targetFeatureType.indexOf(name);
	}

	/*
	 * (non-Javadoc)
	 * @see org.opengis.feature.type.FeatureType#getCoordinateReferenceSystem()
	 */
	@Override
	public CoordinateReferenceSystem getCoordinateReferenceSystem() {
		return _targetFeatureType.getCoordinateReferenceSystem();
	}

	/*
	 * (non-Javadoc)
	 * @see org.opengis.feature.type.FeatureType#getGeometryDescriptor()
	 */
	@Override
	public GeometryDescriptor getGeometryDescriptor() {
		return _targetFeatureType.getGeometryDescriptor();
	}

	/*
	 * (non-Javadoc)
	 * @see org.opengis.feature.type.FeatureType#isIdentified()
	 */
	@Override
	public boolean isIdentified() {
		return _targetFeatureType.isIdentified();
	}

	/*
	 * (non-Javadoc)
	 * @see org.opengis.feature.type.ComplexType#getBinding()
	 */
	@Override
	public Class<Collection<Property>> getBinding() {
		return _targetFeatureType.getBinding();
	}

	/*
	 * (non-Javadoc)
	 * @see org.opengis.feature.type.ComplexType#getDescriptors()
	 */
	@Override
	public Collection<PropertyDescriptor> getDescriptors() {
		return _targetFeatureType.getDescriptors();
	}

	/*
	 * (non-Javadoc)
	 * @see org.opengis.feature.type.ComplexType#isInline()
	 */
	@Override
	public boolean isInline() {
		return _targetFeatureType.isInline();
	}

	/*
	 * (non-Javadoc)
	 * @see org.opengis.feature.type.AttributeType#getSuper()
	 */
	@Override
	public AttributeType getSuper() {
		return _targetFeatureType.getSuper();
	}

	/*
	 * (non-Javadoc)
	 * @see org.opengis.feature.type.PropertyType#getDescription()
	 */
	@Override
	public InternationalString getDescription() {
		return _targetFeatureType.getDescription();
	}

	/*
	 * (non-Javadoc)
	 * @see org.opengis.feature.type.PropertyType#getName()
	 */
	@Override
	public Name getName() {
		return _targetFeatureType.getName();
	}

	/*
	 * (non-Javadoc)
	 * @see org.opengis.feature.type.PropertyType#getRestrictions()
	 */
	@Override
	public List<Filter> getRestrictions() {
		return _targetFeatureType.getRestrictions();
	}

	/*
	 * (non-Javadoc)
	 * @see org.opengis.feature.type.PropertyType#getUserData()
	 */
	@Override
	public Map<Object, Object> getUserData() {
		return _targetFeatureType.getUserData();
	}

	/*
	 * (non-Javadoc)
	 * @see org.opengis.feature.type.PropertyType#isAbstract()
	 */
	@Override
	public boolean isAbstract() {
		return _targetFeatureType.isAbstract();
	}
}
