/**
 * Tecgraf - GIS development team
 * 
 * Tdk Framework
 * Copyright TecGraf 2010(c).
 * 
 * file: TerralibMetadataCache.java
 * created: Apr 19, 2010
 */
package org.geotools.data.terralib;

import java.io.IOException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.geotools.data.terralib.exception.NullArgumentException;
import org.geotools.data.terralib.exception.TerralibProviderRuntimeException;
import org.geotools.data.terralib.util.TypeAttributeMap;
import org.opengis.feature.simple.SimpleFeatureType;


/**
 * TerralibMetadata is responsible for retrieving and caching metadata information
 * about terralib layers and themes.
 * 
 * @author fabiomano
 * @since TDK 3.1
 */
public class TerralibMetadata
{
    private static final String THEME_IDENTIFIER = "@";
    
    private static final String POINTS_TABLE_PREFIX = "Points";
    private static final String LINES_TABLE_PREFIX = "Lines";
    private static final String POLYGONS_TABLE_PREFIX = "Polygons";
    
    private TerralibDataStore _dataStore;
    private Map<String,TypeInfo> _typeInfoMap;
    private Connection _con;
    
    private Object _lock = new Object();    //to sync the methods that fill the cache
    
    //TODO: se tiver AttributePersister...
    
    //prepared statements to retrieve database information
    private PreparedStatement _stmtLayerAttr;
    private PreparedStatement _stmtThemeAttr;
    private PreparedStatement _stmtLayerId;
    private PreparedStatement _stmtThemeId;
    
    public TerralibMetadata (TerralibDataStore dataStore, Connection con)
    {
        if (dataStore == null)
            throw new NullArgumentException("dataStore");
        if (con == null)
            throw new NullArgumentException("con");
        
        _dataStore = dataStore;
        _typeInfoMap = new HashMap<String, TypeInfo>();
        _con = con;
        
        buildPreparedStatements();
    }

    public String getGeometryTable(String typeName) throws IOException
    {
        if (typeName == null)
            throw new NullArgumentException("typeName");
        if (typeName.trim().isEmpty())
            throw new IllegalArgumentException("typeName shouldn't be empty");
        
        if (!_typeInfoMap.containsKey(typeName))
        {
            synchronized (_lock)
            {
                if (!_typeInfoMap.containsKey(typeName))
                {
                    try
                    {
                        fillCache(typeName);
                    }
                    catch (SQLException e)
                    {
                        throw new IOException("Error getting metadata information about typename " + typeName,e);
                    }
                }
            }
        }
        
        return _typeInfoMap.get(typeName).getGeometryTable();
    }
    
    public List<AttributeTableInfo> getAttributeTables(String typeName) throws IOException
    {
        if (typeName == null)
            throw new NullArgumentException("typeName");
        if (typeName.trim().isEmpty())
            throw new IllegalArgumentException("typeName shouldn't be empty");
        
        
        if (!_typeInfoMap.containsKey(typeName))
        {
            synchronized (_lock)
            {
                if (!_typeInfoMap.containsKey(typeName))
                {
                    try
                    {
                        fillCache(typeName);
                    }
                    catch (SQLException e)
                    {
                        throw new IOException("Error getting metadata information about typename " + typeName,e);
                    }
                }
            }            
        }
        
        return _typeInfoMap.get(typeName).getAttributeTables();
    }
    
    /**
     * Builds the prepared statemements to retrieve terralib metadata. 
     */
    private void buildPreparedStatements()
    {
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append("SELECT t.layer_id , lt.attr_table, lt.unique_id ");
        stringBuilder.append(" from te_theme t, te_theme_table tt, te_layer_table lt ");
        stringBuilder.append(" where t.name = ?");
        stringBuilder.append(" and t.view_id = ?");
        stringBuilder.append(" and t.theme_id = tt.theme_id ");
        stringBuilder.append(" and tt.table_id = lt.table_id ");
        try
        {
            _stmtThemeAttr = _con.prepareStatement(stringBuilder.toString());
        }
        catch (SQLException e)
        {
            throw new TerralibProviderRuntimeException("Error creating query to retrieve theme metadata.",e);
        }
        
        stringBuilder = new StringBuilder();
        stringBuilder.append("SELECT L.layer_id ,T.attr_table, T.unique_id");
        stringBuilder.append(" from te_layer L, te_layer_table T ");
        stringBuilder.append(" where L.layer_id = T.layer_id ");
        stringBuilder.append(" and L.name = ?");
        try
        {
            _stmtLayerAttr = _con.prepareStatement(stringBuilder.toString());
        }
        catch (SQLException e)
        {
            throw new TerralibProviderRuntimeException("Error creating query to retrieve layer metadata.",e);
        }        
        
        try
        {
            _stmtThemeId = _con.prepareStatement("SELECT layer_id from te_theme where name = ? and view_id = ?");
        }
        catch (SQLException e)
        {
            throw new TerralibProviderRuntimeException("Error creating query to retrieve layer metadata.",e);
        }        
        
        try
        {
            _stmtLayerId = _con.prepareStatement("SELECT layer_id from te_layer where name = ?");
        }
        catch (SQLException e)
        {
            throw new TerralibProviderRuntimeException("Error creating query to retrieve layer metadata.",e);
        }        
        
    }
    
    private void fillCache(String typeName) throws SQLException, IOException
    {
        
        ResultSet rs;
        
        if (typeName.contains(THEME_IDENTIFIER)) //is a theme
        {
            _stmtThemeAttr.setString(1, getThemeName(typeName));
            _stmtThemeAttr.setInt(2, getViewId(typeName));
            rs = _stmtThemeAttr.executeQuery();
        }
        else
        {
            _stmtLayerAttr.setString(1, typeName);
            rs = _stmtLayerAttr.executeQuery();
        }
        
        int layerId = -1;
        List<AttributeTableInfo> names = new ArrayList<AttributeTableInfo>();
        while (rs.next())
        {
            layerId = rs.getInt("layer_id");
            names.add(new AttributeTableInfo(rs.getString("attr_table"),rs.getString("unique_id")));
        }
        rs.close();
        
        TypeInfo info = new TypeInfo();
        
        info.setAttributeTableInfo(names);        

        if (layerId == -1)
            layerId = retrieveLayerID(typeName);
        
        if (layerId == -1)
        {
//            _typeInfoMap.remove(typeName);
            throw new TerralibProviderRuntimeException("The layer " + typeName + " you are trying to read was not found. If you are using access and has just created the layer, it is an access known bug. Try to wait a little before reading the data.");
        }
        
        //fill geometry table name
        
        SimpleFeatureType schema = _dataStore.getSchema(typeName);
        
        Class<?> binding = schema.getGeometryDescriptor().getType().getBinding();
        
        TypeAttributeMap type = TypeAttributeMap.fromBindingClass(binding);
        if (type == null)
            throw new TerralibProviderRuntimeException("Geometry type " + binding + " not supported");
        
        switch (type)
        {
            case TA_POINT:
            case TA_MULTIPOINT:
                info.setGeometryTable(POINTS_TABLE_PREFIX + layerId);
                break;
            case TA_LINE:
            case TA_MULTILINE:
                info.setGeometryTable(LINES_TABLE_PREFIX + layerId);
                break;
            case TA_POLYGON:
            case TA_MULTIPOLYGON:
                info.setGeometryTable(POLYGONS_TABLE_PREFIX + layerId);
                break;
            default:
                throw new TerralibProviderRuntimeException("Geometry type " + binding + " not supported");
        }
        
        _typeInfoMap.put(typeName, info);            
    }
    
    private int retrieveLayerID(String typeName) throws SQLException
    {
        ResultSet rs;
        if (typeName.contains(THEME_IDENTIFIER)) //is a theme
        {
            _stmtThemeAttr.setString(1, getThemeName(typeName));
            _stmtThemeAttr.setInt(2, getViewId(typeName));
            rs = _stmtThemeAttr.executeQuery();
        }
        else
        {
            _stmtLayerAttr.setString(1, typeName);
            rs = _stmtLayerAttr.executeQuery();
        }
        
        
        if (rs.next())
            return rs.getInt("layer_id");
        
        return -1;
    }
    
    private int getViewId(String typeName)
    {
        return Integer.parseInt(typeName.substring(0,typeName.indexOf(THEME_IDENTIFIER))); 
    }
    
    private String getThemeName(String typeName)
    {
        return typeName.substring(typeName.indexOf(THEME_IDENTIFIER) +1);
    }
    
    private static class TypeInfo
    {
        private String _geometryTable;
        private List<AttributeTableInfo> _attributeTables;

        public TypeInfo()
        {
            _attributeTables = new ArrayList<AttributeTableInfo>();
            
        }
        
        public String getGeometryTable()
        {
            return _geometryTable;
        }
        
        public void setGeometryTable(String geometryTable)
        {
            _geometryTable = geometryTable;
        }
        
        public List<AttributeTableInfo> getAttributeTables()
        {
            return _attributeTables;
        }
        
        public void setAttributeTableInfo(List<AttributeTableInfo> attributeTables)
        {
            _attributeTables = attributeTables;
        }
        
    }
    
    public static class AttributeTableInfo
    {
        private String _tableName;
        private String _uniqueId;

        private AttributeTableInfo(String tableName, String uniqueId)
        {
            _tableName = tableName;
            _uniqueId = uniqueId;
        }
        
        public String getTableName()
        {
            return _tableName;
        }
        
        public String getUniqueId()
        {
            return _uniqueId;
        }
    }
    
    public void dispose()
    {
        try
        {
            _stmtLayerAttr.close();
        }
        catch (SQLException e)
        {
            // ignore
        }
        try
        {
            _stmtThemeAttr.close();
        }
        catch (SQLException e)
        {
            // ignore
        }
        try
        {
            _stmtLayerId.close();
        }
        catch (SQLException e)
        {
            // ignore
        }
        try
        {
            _stmtThemeId.close();
        }
        catch (SQLException e)
        {
            // ignore
        }
        
    }
    
}
