/**
 * Tecgraf - GIS development team
 * 
 * Tdk Framework
 * Copyright TecGraf 2009(c).
 * 
 * file: TerralibFeatureReaderWriter.java
 * created: 06/04/2009
 */
package org.geotools.data.terralib.feature;

import java.io.IOException;
import java.sql.SQLException;
import java.util.List;
import java.util.NoSuchElementException;

import org.apache.log4j.Logger;
import org.geotools.data.DataSourceException;
import org.geotools.data.DataUtilities;
import org.geotools.data.FeatureListenerManager;
import org.geotools.data.FeatureReader;
import org.geotools.data.FeatureWriter;
import org.geotools.data.Transaction;
import org.geotools.data.terralib.TerralibFeatureType;
import org.geotools.data.terralib.TerralibService;
import org.geotools.data.terralib.exception.IllegalStateException;
import org.geotools.data.terralib.exception.NullArgumentException;
import org.geotools.data.terralib.exception.TypeNotFoundException;
import org.geotools.data.terralib.feature.attribute.TerralibAttributePersistenceHandler;
import org.geotools.data.terralib.feature.attribute.TerralibAttributePersistenceHandlerTypeOperation;
import org.geotools.data.terralib.persistence.exception.MissingRequiredGeometryException;
import org.geotools.data.terralib.query.QueryData;
import org.geotools.data.terralib.util.TypeAttributeMap;
import org.geotools.feature.SchemaException;
import org.geotools.feature.simple.SimpleFeatureBuilder;
import org.geotools.feature.simple.SimpleFeatureImpl;
import org.geotools.filter.identity.FeatureIdImpl;
import org.geotools.geometry.jts.ReferencedEnvelope;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;

/**
 * 
 * @author fmoura, fabiomano
 * @since TDK 3.0.0
 */
public class TerralibFeatureReaderWriter implements FeatureReader<SimpleFeatureType, SimpleFeature>,
        FeatureWriter<SimpleFeatureType, SimpleFeature>
{

    protected QueryData _queryData;
    protected TerralibFeatureReader _featureReader;
    protected FeatureListenerManager _listenerManager;
    protected Transaction _transaction;

    private static Logger _logger = Logger.getLogger(TerralibFeatureReaderWriter.class);

    protected SimpleFeature _live = null;       // current Feature read, for FeatureWriter
    protected SimpleFeature _current = null; // copy of live returned to user

    protected TerralibService _service;

    protected int featureIdIndex = -1;

    private TerralibAttributePersistenceHandler _persistenceHandler;

    /**
     * @param listenerManager Retrieved from the datastore.
     * @param service
     * @param queryData
     * @param schema
     * @param transaction
     * @param persistenceHandler
     * @throws SchemaException
     */
    public TerralibFeatureReaderWriter(FeatureListenerManager listenerManager, 
                                       TerralibService service,
                                       QueryData queryData,
                                       TerralibFeatureType schema, 
                                       Transaction transaction,
                                       TerralibAttributePersistenceHandler persistenceHandler)
        throws SchemaException
    {
        if (listenerManager == null)
            throw new NullArgumentException("listenerManager");

        if (service == null)
            throw new NullArgumentException("service");

        if (queryData == null)
            throw new NullArgumentException("queryData");

        if (schema == null)
            throw new NullArgumentException("schema");

        if (transaction == null)
            throw new NullArgumentException("transaction");

        if (persistenceHandler == null)
            throw new NullArgumentException("persistenceHandler");

        if (transaction != Transaction.AUTO_COMMIT)
            _logger.warn("Transaction for the TerraLibFeatureReaderWriter is not set to AUTO_COMMIT."
                    + " For now, TerralibFeatureReaderWriter only implements changes as AUTO_COMMIT."
                    + " Your transaction will be ignored until we implement support for transaction.");

        _listenerManager = listenerManager;
        _queryData = queryData;
        _transaction = transaction;
        _service = service;
        _featureReader = new TerralibFeatureReader(queryData, schema);
        _persistenceHandler = persistenceHandler;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.geotools.data.FeatureReader#getFeatureType()
     */
    @Override
    public TerralibFeatureType getFeatureType()
    {
        return (TerralibFeatureType) _featureReader.getFeatureType();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.geotools.data.FeatureReader#close()
     */
    @Override
    public void close() throws IOException
    {
        if (!_queryData.isClosed())
            _featureReader.close();
        else
            _logger.warn("Feature writer calling close when queryData is already closed");
    }

    /**
     * Checks if the query data is closed.
     * If it is, throw an IllegalStateException.
     */
    private void checkClosedQueryData()
    {
        if (_queryData.isClosed())
            throw new java.lang.IllegalStateException("Terralib query data is closed");
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.geotools.data.FeatureReader#hasNext()
     */
    @Override
    public boolean hasNext() throws IOException
    {
        checkClosedQueryData();
        return _featureReader.hasNext();
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.geotools.data.FeatureReader#next()
     */
    @Override
    public SimpleFeature next() throws IOException, IllegalArgumentException, NoSuchElementException
    {
        checkClosedQueryData();

        SimpleFeatureType featureType = getFeatureType();

        if (hasNext())
        {
            _live = _featureReader.next();
            _current = SimpleFeatureBuilder.copy(_live);
            _logger.trace("Calling next on writer");
        }
        else
        {
            // new content
            _live = null;
            SimpleFeature temp = SimpleFeatureBuilder.template(featureType, null);

            /*
             * Here we create a Feature with a Mutable FID. We use data
             * utilities to create a default set of attributes for the feature
             * and these are copied into the a new MutableFIDFeature. This can
             * probably be improved later, there is also a dependency on
             * DefaultFeatureType here since DefaultFeature depends on it and
             * MutableFIDFeature extends default feature. This may be an issue
             * if someone reimplements the Feature interfaces. It could address
             * by providing a full implementation of Feature in
             * MutableFIDFeature at a later date.
             */
            _current = new TerralibSimpleFeature(temp.getAttributes(), featureType, null);

            _queryData.next();
        }

        return _current;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.geotools.data.FeatureWriter#remove()
     */
    @Override
    public void remove() throws IOException
    {
        checkClosedQueryData();

        if (_live != null)
        {
            _logger.debug("Removing " + _live);

            String featureId = getFeatureId();
            ReferencedEnvelope bounds = ReferencedEnvelope.reference(_live.getBounds());
            _live = null;
            _current = null;

            try
            {
                boolean removeOnlyGeometries = ((_persistenceHandler != null) && _persistenceHandler.canProcess(
                        getFeatureType().getTypeName(), TerralibAttributePersistenceHandlerTypeOperation.REMOVE));

                _service.remove(getFeatureType().getTypeName(), featureId, removeOnlyGeometries);

                if (removeOnlyGeometries)
                {
                    _persistenceHandler.removeFeatureAttributes(featureId);
                }

                fireFeaturesRemoved(bounds, false);
            }
            catch (IllegalStateException e)
            {
                throw new DataSourceException("Error removing row", e);
            }
            catch (TypeNotFoundException e)
            {
                throw new DataSourceException("Error removing row", e);
            }
        }
        else
        {
            throw new IOException("No feature available to be removed.");
        }
    }

    /**
     * @return
     * @throws DataSourceException
     * @throws IOException
     */
    protected String getFeatureId() throws DataSourceException, IOException
    {
        // if (featureIdIndex == -1)
        // {
        // String currentAttributeName = null;
        // for (int i=0; i <= _queryData.getAttributeCount() - 1; i++)
        // {
        // currentAttributeName =
        // _queryData.getAttributeType(i).getName().getLocalPart();
        // if (FEATURE_ID_ATTRIBUTE_NAME.equals(currentAttributeName))
        // {
        // featureIdIndex = i;
        // break;
        // }
        // }
        // if (featureIdIndex == -1)
        // throw new
        // DataSourceException("Couldn't find the id attribute for this feature");
        // }
        //        
        // String featureId = (String)_queryData.read(featureIdIndex);
        String featureId = _queryData.getObjectId();
        return featureId;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.geotools.data.FeatureWriter#write()
     */
    @Override
    public void write() throws IOException
    {
        checkClosedQueryData();

        _logger.debug("write called, live is " + _live + " and current is " + _current);

        if (_live != null) // is updating
        {
            if (_live.equals(_current))
            {
                // no modifications made to current
                _live = null;
                _current = null;
            }
            else
            {
                try
                {
                    doUpdate(_live, _current);

                    ReferencedEnvelope bounds = new ReferencedEnvelope(_live.getType().getCoordinateReferenceSystem());
                    bounds.include(_live.getBounds());
                    bounds.include(_current.getBounds());

                    fireFeaturesChanged(bounds, false);
                }
                catch (DataSourceException dsException)
                {
                    _queryData.close();
                    throw dsException;
                }
                _live = null;
                _current = null;
            }
        }
        else
        {
            _logger.debug("doing insert in terralib featurewriter");

            doInsert((TerralibSimpleFeature) _current);
            fireFeaturesAdded(ReferencedEnvelope.reference(_current.getBounds()), false);
            _current = null;
        }
    }

    /**
     * Gets the listener manager, that can be used to register feature listeners
     * for this object.
     * 
     * @return the ListenerManager for this object
     */
    public FeatureListenerManager getListenerManager()
    {
        return _listenerManager;
    }

    /**
     * Called when a feature is removed.
     */
    protected void fireFeaturesRemoved(ReferencedEnvelope bounds, boolean isCommit)
    {
        String typeName = _featureReader.getFeatureType().getTypeName();
        _listenerManager.fireFeaturesRemoved(typeName, _transaction, bounds, isCommit);
    }

    /**
     * Called when a feature is updated.
     */
    protected void fireFeaturesChanged(ReferencedEnvelope bounds, boolean isCommit)
    {
        String typeName = _featureReader.getFeatureType().getTypeName();
        _listenerManager.fireFeaturesChanged(typeName, _transaction, bounds, isCommit);
    }

    /**
     * Called when a feature is added.
     */
    protected void fireFeaturesAdded(ReferencedEnvelope bounds, boolean isCommit)
    {
        String typeName = _featureReader.getFeatureType().getTypeName();
        _listenerManager.fireFeaturesAdded(typeName, _transaction, bounds, isCommit);
    }

    protected void doUpdate(SimpleFeature live, SimpleFeature current) throws IOException
    {
        for (int i = 0; i < current.getAttributeCount(); i++)
        {
            Object currAtt = current.getAttribute(i);
            Object liveAtt = null;
            
            if (live != null)
                liveAtt = live.getAttribute(i);

            if ((live == null) || !DataUtilities.attributesEqual(liveAtt, currAtt))
            {
                _queryData.write(i, currAtt);
            }
        }

        boolean writeOnlyGeometries = ((_persistenceHandler != null) && (_persistenceHandler.canProcess(current
                .getType().getTypeName(), TerralibAttributePersistenceHandlerTypeOperation.UPDATE)));

        _queryData.flush(writeOnlyGeometries);

        if (writeOnlyGeometries)
        {
            _persistenceHandler.updateFeatureAttributes(current);
        }
    }

    /**
     * Inserts a feature into the database.
     * 
     * <p>
     * This method should both insert a Feature, and update its FID in case the
     * FIDMapper works over database generated ids like autoincrement fields,
     * sequences, and object ids.
     * </p>
     * 
     * @throws SQLException
     */
    protected void doInsert(TerralibSimpleFeature mutable) throws IOException
    {
        // assign the SimpleFeature's attributes to the transfer object, to send
        // to terralib

        boolean writeOnlyGeometries = ((_persistenceHandler != null) && (_persistenceHandler.canProcess(mutable
                .getType().getTypeName(), TerralibAttributePersistenceHandlerTypeOperation.INSERT)));
        
        //the geometry is always the first attribute
    	int insertIndex = 1;
    	int geometryIndex = -1;
    	
        for (int i = 0; i < mutable.getAttributeCount(); i++)
        {
    		Object attributeValue = mutable.getAttribute(i);
    		TypeAttributeMap type = null;

    		if (attributeValue != null)
    		    type = TypeAttributeMap.fromBindingClass(attributeValue.getClass());
    		
        	if ((type != null) && (type.isGeometry()))
        	    geometryIndex = i;  //skips but stores the geometryPosition
        	else if (!writeOnlyGeometries)
        	{
    		    _queryData.write(insertIndex, attributeValue);
                insertIndex ++;
    		}
        }
        
        if (geometryIndex == -1)
            throw new MissingRequiredGeometryException("The geometry attribute is missing");
        
        _queryData.write(0, mutable.getAttribute(geometryIndex));

        // sets queryData's objectId value if the Feature has one specified
        if (mutable.hasSpecifiedID())
            _queryData.setObjectId(mutable.getID());

        _queryData.flush(writeOnlyGeometries);

        String id = getFeatureId();
        mutable.setID(id);
        // mutable.setAttribute(FEATURE_ID_ATTRIBUTE_NAME, id);

        if (writeOnlyGeometries)
        {
            _persistenceHandler.insertFeatureAttributes(mutable);
        }
    }

    public static class TerralibSimpleFeature extends SimpleFeatureImpl
    {
        // variable that stores whether this Feature has a specified ID
        // Features without specified IDs will receive IDs generated by Terralib
        boolean _hasSpecifiedID = false;

        public TerralibSimpleFeature(List<Object> values, SimpleFeatureType ft, String fid)
        {
            super(values, ft, createDefaultFID(fid));

            _hasSpecifiedID = (fid != null);
        }

        private static FeatureIdImpl createDefaultFID(String id)
        {
            if (id == null)
            {
                id = SimpleFeatureBuilder.createDefaultFeatureId();
            }

            return new FeatureIdImpl(id)
            {

                public void setID(String id)
                {
                    if (fid == null)
                    {
                        throw new NullArgumentException("fid must not be null");
                    }
                    if (origionalFid == null)
                    {
                        origionalFid = fid;
                    }
                    fid = id;
                }
            };
        }
        
        public void setID(String fid)
        {
            ((FeatureIdImpl) id).setID(fid);
            _hasSpecifiedID = true;
        }

        public boolean hasSpecifiedID()
        {
            return _hasSpecifiedID;
        }
    }
}
