/**
 * Tecgraf - GIS development team
 * 
 * Tdk Framework
 * Copyright TecGraf 2009(c).
 * 
 * file: TerralibFeatureStore.java
 * created: Aug 20, 2009
 */
package org.geotools.data.terralib.feature;

import java.io.IOException;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import org.geotools.data.AbstractFeatureStore;
import org.geotools.data.DataSourceException;
import org.geotools.data.DataStore;
import org.geotools.data.FeatureListener;
import org.geotools.data.FeatureWriter;
import org.geotools.data.QueryCapabilities;
import org.geotools.data.terralib.TerralibDataStore;
import org.geotools.data.terralib.exception.NullArgumentException;
import org.geotools.data.terralib.feature.TerralibFeatureReaderWriter.TerralibSimpleFeature;
import org.geotools.feature.FeatureCollection;
import org.geotools.feature.FeatureCollections;
import org.geotools.feature.FeatureIterator;
import org.opengis.feature.simple.SimpleFeature;
import org.opengis.feature.simple.SimpleFeatureType;
import org.opengis.filter.Filter;
import org.opengis.filter.identity.FeatureId;

/**
 * Terralib's implementation of a feature store. Takes some cares adding features to avoid locking in case
 * you are copying features from a terralib database to another terralib database (or the same).
 * @author fabiomano
 * @since TDK3.0.0
 */
public class TerralibFeatureStore extends AbstractFeatureStore
{
    private TerralibDataStore _dataStore;
    private final SimpleFeatureType _featureType;
    private TerralibQueryCapabilities _capabilities;

    @SuppressWarnings("unchecked")
    public TerralibFeatureStore(TerralibDataStore dataStore, Set hints, SimpleFeatureType featureType)
    {
        super(hints);
        if (dataStore == null)
            throw new NullArgumentException("dataStore");
        if (featureType == null)
            throw new NullArgumentException("featureType");
        
        _dataStore = dataStore;
        _featureType = featureType;
        _capabilities = new TerralibQueryCapabilities();
    }
    
    /* (non-Javadoc)
     * @see org.geotools.data.AbstractFeatureSource#getQueryCapabilities()
     */
    @Override
    public QueryCapabilities getQueryCapabilities()
    {
        return _capabilities;
    }
    
    /* (non-Javadoc)
     * @see org.geotools.data.AbstractFeatureSource#getDataStore()
     */
    @Override
    public DataStore getDataStore()
    {
        return _dataStore;
    }

    /* (non-Javadoc)
     * @see org.geotools.data.FeatureSource#addFeatureListener(org.geotools.data.FeatureListener)
     */
    @Override
    public void addFeatureListener(FeatureListener listener)
    {
        _dataStore.listenerManager.addFeatureListener(this, listener);
    }

    /* (non-Javadoc)
     * @see org.geotools.data.FeatureSource#getSchema()
     */
    @Override
    public SimpleFeatureType getSchema()
    {
        return _featureType;
    }

    /* (non-Javadoc)
     * @see org.geotools.data.FeatureSource#removeFeatureListener(org.geotools.data.FeatureListener)
     */
    @Override
    public void removeFeatureListener(FeatureListener listener)
    {
        _dataStore.listenerManager.removeFeatureListener(this, listener);
    }
    
    /* (non-Javadoc)
     * @see org.geotools.data.AbstractFeatureStore#addFeatures(org.geotools.feature.FeatureCollection)
     */
    @Override
    public List<FeatureId> addFeatures(FeatureCollection<SimpleFeatureType, SimpleFeature> collection)
            throws IOException
    {
        List<FeatureId> addedFids = new LinkedList<FeatureId>();
        String typeName = getSchema().getTypeName();
        SimpleFeature feature = null;
        SimpleFeature newFeature;
        
        //TODO: this should be removed once we are sure Java reading is 100% correct
        
        //there is a problem when the collection is from terralib, can't open a reader and a writer at the same time. it locks.
        //to solve this we need to copy the features to a feature collection in memory
        FeatureCollection<SimpleFeatureType,SimpleFeature> fcCopy = FeatureCollections.newCollection();
        FeatureIterator<SimpleFeature> iterator = collection.features();
        try
        {
            while (iterator.hasNext())
                fcCopy.add(iterator.next());
        }
        finally 
        {
            collection.close( iterator );
        }

        //now we can copy the features to this feature store
        FeatureWriter<SimpleFeatureType, SimpleFeature> writer = getDataStore().getFeatureWriterAppend(typeName, getTransaction());
        
        iterator = fcCopy.features();
        try 
        {
            while (iterator.hasNext()) 
            {
                feature = (SimpleFeature) iterator.next();
                newFeature = (SimpleFeature)writer.next();
                try 
                {
                    newFeature.setAttributes(feature.getAttributes());
                    if (newFeature instanceof TerralibSimpleFeature)
                        ((TerralibSimpleFeature)newFeature).setID(feature.getID());
                    
                } 
                catch (Exception writeProblem) 
                {
                    throw new DataSourceException("Could not create "
                        + typeName + " out of provided feature: "
                        + feature.getID(), writeProblem);
                }

                writer.write();
                addedFids.add(newFeature.getIdentifier());
            }
        } 
        finally 
        {
            collection.close( iterator );
            writer.close();
        }
        return addedFids;        
    }
    
    
    @Override
    public void removeFeatures(Filter filter) throws IOException {
    	if(Filter.INCLUDE.equals(filter))
    		_dataStore.removeAllFeatures(_featureType.getTypeName());
    	else
    		super.removeFeatures(filter);
    }
    
    //NOT NEEDED NOW BECAUSE ITS IMPOSSIBLE TO GET A READER AND A STORE OF THE SAME TERRALIBDATASTORE (GETS LOCKED)
//    /*
//     * (non-Javadoc)
//     * @see org.geotools.data.AbstractFeatureStore#addFeatures(org.geotools.data.FeatureReader)
//     */
//    @Override
//    public Set<String> addFeatures(FeatureReader <SimpleFeatureType, SimpleFeature> reader) throws IOException 
//    {
//        Set<String> addedFids = new HashSet<String>();
//        String typeName = getSchema().getTypeName();
//        SimpleFeature feature = null;
//        SimpleFeature newFeature;
//        
//        //there is a problem when the collection is from terralib, can't open a reader and a writer at the same time. it locks.
//        //to solve this we need to copy the features to a feature collection in memory
//        FeatureCollection<SimpleFeatureType,SimpleFeature> fcCopy = FeatureCollections.newCollection();
//        
//        try
//        {
//            while (reader.hasNext())
//            {
//                try
//                {
//                    fcCopy.add(reader.next());
//                }
//                catch (Exception e) 
//                {
//                    throw new DataSourceException("Could not add Features, problem with provided reader",e);
//                }                
//            }
//        }
//        finally 
//        {
//            reader.close();
//        }
//        
//        //now we can copy the features to this feature store
//        FeatureIterator<SimpleFeature> iterator = fcCopy.features();
//
//        FeatureWriter<SimpleFeatureType, SimpleFeature> writer = getDataStore()
//        .getFeatureWriterAppend(typeName, getTransaction());
//        
//        try 
//        {
//            while (iterator.hasNext()) 
//            {
//                feature = (SimpleFeature) iterator.next();
//                newFeature = (SimpleFeature)writer.next();
//
//                try 
//                {
//                    newFeature.setAttributes(feature.getAttributes());
//                } catch (Exception writeProblem) {
//                    throw new DataSourceException("Could not create "
//                        + typeName + " out of provided feature: "
//                        + feature.getID(), writeProblem);
//                }
//
//                writer.write();
//                addedFids.add(newFeature.getID());
//            }
//        } 
//        finally 
//        {
//            writer.close();
//        }
//
//        return addedFids;
//    }    
}
