/**
 * Tecgraf - GIS development team
 * 
 * Tdk Framework
 * Copyright TecGraf 2010(c).
 * 
 * file: TerralibFilterToSQL.java
 * created: Apr 13, 2010
 */
package org.geotools.data.terralib.query.filter;

import java.io.IOException;
import java.io.Serializable;
import java.io.StringWriter;
import java.io.Writer;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.geotools.data.jdbc.FilterToSQL;
import org.geotools.data.jdbc.FilterToSQLException;
import org.geotools.data.terralib.exception.NullArgumentException;
import org.geotools.filter.FilterCapabilities;
import org.geotools.filter.FilterFactoryImpl;
import org.geotools.geometry.jts.ReferencedEnvelope;
import org.opengis.feature.simple.SimpleFeatureType;
import org.opengis.feature.type.AttributeDescriptor;
import org.opengis.filter.BinaryComparisonOperator;
import org.opengis.filter.Filter;
import org.opengis.filter.FilterFactory2;
import org.opengis.filter.Id;
import org.opengis.filter.PropertyIsLike;
import org.opengis.filter.PropertyIsNotEqualTo;
import org.opengis.filter.expression.Expression;
import org.opengis.filter.expression.PropertyName;
import org.opengis.filter.identity.Identifier;
import org.opengis.filter.spatial.BBOX;
import org.opengis.filter.spatial.BinarySpatialOperator;
import org.opengis.filter.spatial.Contains;
import org.opengis.filter.spatial.Intersects;
import org.opengis.filter.spatial.Within;

import com.vividsolutions.jts.geom.Geometry;
import com.vividsolutions.jts.geom.LineString;
import com.vividsolutions.jts.geom.MultiLineString;
import com.vividsolutions.jts.geom.MultiPoint;
import com.vividsolutions.jts.geom.MultiPolygon;
import com.vividsolutions.jts.geom.Point;
import com.vividsolutions.jts.geom.Polygon;

/**
 * Encodes a filter into a Terralib compatible SQL WHERE statement.
 * It should hopefully be generic enough that any SQL database will work with it.
 * 
 * @author fabiomano
 * @since TDK 3.1
 */
@SuppressWarnings("deprecation")
public class TerralibFilterToSQL
{
    /**
     * "symbol_not_equal"      String     The symbol to use for ATTRIBUTE != value. Default is !=, but access for i.e. uses <>
     */
    public static final String SYMBOL_NOT_EQUAL = "symbol_not_equal";
    
    private static final String SYMBOL_NOT_EQUAL_DEFAULT = "!=";
    
    private SimpleFeatureType _featureType;
    private Map<String, Serializable> _hints;
    
    private String _geometryTableName;

    /**
     * Constructor without geometry table name. TerralibFilterToSQL will try to make 
     * an query optimization to retrieve the features without the sql IN clause.
     * @param featureType
     * @param geometryTableName
     */
    public TerralibFilterToSQL(SimpleFeatureType featureType)
    {
        if (featureType == null)
            throw new NullArgumentException("featureType");
        
        _featureType = featureType;
        _hints = new HashMap<String, Serializable>();
        _geometryTableName = null;
    }
    
    /**
     * Constructor with geometry table name. When it's not provided, TerralibFilterToSQL tries to make 
     * an query optimization to retrieve the features without the sql IN clause.
     * @param featureType
     * @param geometryTableName
     */
    public TerralibFilterToSQL(SimpleFeatureType featureType, String geometryTableName)
    {
        if (featureType == null)
            throw new NullArgumentException("featureType");
        if (geometryTableName == null)
            throw new NullArgumentException("geometryTableName");
        if (geometryTableName.trim().isEmpty())
            throw new IllegalArgumentException("geometryTableName can't be empty");
        
        _featureType = featureType;
        _hints = new HashMap<String, Serializable>();
        _geometryTableName = geometryTableName;
    }
    
    
    /**
     * Set a hint to the filter translation process.
     * TerralibFilterToSQL has some static fields with the known keys for the hints
     * that are understood by the serializers.
     * i.e. SerializationResources.RELATIVIZE_PATHS_KEY   
     * @param hintKey The hint to be passed to the serialization process
     * @param hintValue The value of the key
     */
    public void setHint(String hintKey, Serializable hintValue)
    {
        _hints.put(hintKey, hintValue);
    }    
    
    /**
     * Clear all filter translation hints.
     */
    public void clearHints()
    {
        _hints.clear();
    }    
    
    /**
     * Encode the given filter to a SQL statement.
     * Writes the result query at the writer
     * @param filter Filter to encode
     * @param writer Writer to store the result query
     * @throws TerralibFilterToSQLException In case the filter can't be translated.
     */
    public void encode(Filter filter, Writer writer) throws TerralibFilterToSQLException 
    {
        if (filter == null)
            throw new NullArgumentException("filter can't be null");
        if (writer == null)
            throw new NullArgumentException("writer can't be null");
        
        InternalFilterToSQL encoder = new InternalFilterToSQL(writer,_hints,_geometryTableName);
        encoder.setFeatureType(_featureType);
        try
        {
            encoder.encode(filter);
        }
        catch (FilterToSQLException e)
        {
            throw new TerralibFilterToSQLException(filter,e.getMessage());
        }
    }
    
    /**
     * A convenience method to encode directly to a string.
     * 
     * Equivalent to:
     * 
     *  StringWriter out = new StringWriter();
     *  new TerralibFilterToSQL().encode(filter,out);
     *  out.getBuffer().toString();
     * 
     * @param filter Filter to encode
     * @return a string representing the filter encoded to SQL.
     * @throws TerralibFilterToSQLException In case the filter can't be translated.
     */
    
    public String encodeToString(Filter filter) throws TerralibFilterToSQLException 
    {
        StringWriter out = new StringWriter();
        this.encode(filter,out);
        return out.getBuffer().toString();
    }    
    
    /**
     * Encode the given expression to a SQL statement.
     * Writes the result Expression at the writer
     * @param expression Expression to encode
     * @param writer Writer to store the result query
     * @throws TerralibFilterToSQLException In case the expression can't be translated.
     */
    public void encode(Expression expression, Writer writer) throws TerralibFilterToSQLException 
    {
        if (expression == null)
            throw new NullArgumentException("expression can't be null");
        if (writer == null)
            throw new NullArgumentException("writer can't be null");
        
        InternalFilterToSQL encoder = new InternalFilterToSQL(writer,_hints,_geometryTableName);
        encoder.setFeatureType(_featureType);
        try
        {
            encoder.encode(expression);
        }
        catch (FilterToSQLException e)
        {
            throw new TerralibFilterToSQLException(expression,e.getMessage());
        }
    }    

    /**
     * A convenience method to encode directly to a string.
     * 
     * Equivalent to:
     * 
     *  StringWriter out = new StringWriter();
     *  new TerralibFilterToSQL().encode(expression,out);
     *  out.getBuffer().toString();
     * 
     * @param expression Expression to encode
     * @return a string representing the filter encoded to SQL.
     * @throws TerralibFilterToSQLException In case the expression can't be translated.
     */
    public String encodeToString(Expression expression) throws TerralibFilterToSQLException
    {
        StringWriter out = new StringWriter();
        this.encode(expression,out);
        return out.getBuffer().toString();
    }    
    
    public boolean supports(Filter filter)
    {
        if (filter == null)
            throw new NullArgumentException("filter");
        
        InternalFilterToSQL encoder = new InternalFilterToSQL(null,_hints,_geometryTableName);
        return encoder.getCapabilities().fullySupports(filter);
    }
    
    private static class InternalFilterToSQL extends FilterToSQL
    {
        private Map<String,Serializable> _hints;
        private String _geometryTableName;
        
        public InternalFilterToSQL(Writer writer, Map<String,Serializable> hints, String geometryTableName)
        {
            super(writer);
            _hints = hints;
            _geometryTableName = geometryTableName;
        }
        
        @Override
        protected FilterCapabilities createFilterCapabilities() 
        {
            FilterCapabilities capabilities = super.createFilterCapabilities();
            
            //adds the type we implemented
            capabilities.addType(Contains.class);
            capabilities.addType(Within.class);
            capabilities.addType(BBOX.class);
            capabilities.addType(Intersects.class);
            capabilities.addType(PropertyIsLike.class);
            return capabilities;
        }        
        
        @Override
        public Object visit(PropertyIsNotEqualTo filter, Object extraData) 
        {
            String differentSymbol;
            if (_hints.containsKey(SYMBOL_NOT_EQUAL))
                differentSymbol = (String)_hints.get(SYMBOL_NOT_EQUAL);
            else
                differentSymbol = SYMBOL_NOT_EQUAL_DEFAULT;
                
            visitBinaryComparisonOperator((BinaryComparisonOperator)filter, differentSymbol);
            return extraData;
        }        
        
        @Override
        public Object visit(Id filter, Object extraData) 
        {
            Set<Identifier> ids = filter.getIdentifiers();
            
            try
            {
                if (ids.size() > 1)
                {
                    boolean first = true;
                    out.write(" object_id IN (");
                    for (Identifier id: ids)
                    {
                        if (!first)
                            out.write(",");
                        else
                            first = false;
                        
                        out.write("'");
                        out.write((String)id.getID());
                        out.write("'");
                    }
                    out.write(")");
                }
                else if (ids.size() == 1)
                    out.write(" object_id = '" + ids.iterator().next().getID() + "'");
            }
            catch (IOException e)
            {
                throw new TerralibFilterToSQLException(filter,"Error writing query to Writer.");
            }
            return extraData;
        }

        /* (non-Javadoc)
         * @see org.geotools.data.jdbc.FilterToSQL#visit(org.opengis.filter.PropertyIsLike, java.lang.Object)
         */
        @Override
        public Object visit(PropertyIsLike filter, Object extraData)
        {
            if (filter == null)
                throw new NullArgumentException("filter");
            if (filter.getLiteral() == null)
                throw new IllegalArgumentException("like filter pattern value can't be null.");
            
            return super.visit(filter, extraData);
        }
        
        @Override
        public Object visit(Contains filter, Object extraData) 
        {
            QueryInfo info = analyzeFilter(filter);
            try
            {
                if (_geometryTableName != null)
                    writeInOpen();
                
                if (isPoint(info.getGeometryBinding()))
                {
                    out.write("(");
                    out.write(" x >= " + info.getBounds().getMinX());
                    out.write(" AND x <= " + info.getBounds().getMaxX());
                    out.write(" AND y >= " + info.getBounds().getMinY());
                    out.write(" AND y <= " + info.getBounds().getMaxY());
                    out.write(")");
                }
                else if (isLine(info.getGeometryBinding()) || (isPolygon(info.getGeometryBinding())))
                {
                    out.write("(");
                    out.write(" lower_x >= " + info.getBounds().getMinX());
                    out.write(" AND upper_x <= " + info.getBounds().getMaxX());
                    out.write(" AND lower_y >= " + info.getBounds().getMinY());
                    out.write(" AND upper_y <= " + info.getBounds().getMaxY());
                    out.write(")");
                }
                else
                    throw new TerralibFilterToSQLException(filter,"Unsupported geometry");
                
                if (_geometryTableName != null)
                    writeInClose();
                
            }
            catch (IOException e)
            {
                throw new TerralibFilterToSQLException(filter,"Error writing query to Writer.");
            }
            
            return extraData;
        }
        
        /* (non-Javadoc)
         * @see org.geotools.data.jdbc.FilterToSQL#visit(org.opengis.filter.spatial.Intersects, java.lang.Object)
         */
        @Override
        public Object visit(Intersects filter, Object extraData)
        {
            return internalBBoxIntersectVisit(filter, extraData);
        }
        
        /**
         * Internal method that creates the query for BBox and Intersects filters.
         * They are the same here, intersects result will be refined at memory. BBox 
         * is complete here. 
         * @param filter The filter to visit
         * @param extraData
         * @return extraData
         */
        private Object internalBBoxIntersectVisit(BinarySpatialOperator filter, Object extraData)
        {
            QueryInfo info = analyzeFilter(filter);
            
            try
            {
                if (_geometryTableName != null)
                    writeInOpen();
                
                if (isPoint(info.getGeometryBinding()))
                {
                    out.write("(");
                    out.write(" x >= " + info.getBounds().getMinX());
                    out.write(" AND x <= " + info.getBounds().getMaxX());
                    out.write(" AND y >= " + info.getBounds().getMinY());
                    out.write(" AND y <= " + info.getBounds().getMaxY());
                    out.write(")");
                }
                else if (isLine(info.getGeometryBinding()) || (isPolygon(info.getGeometryBinding())))
                {
                    out.write("(");
                        out.write("NOT");
                        out.write("(");
                            out.write(" lower_x > " + info.getBounds().getMaxX());
                            out.write(" OR ");
                            out.write(" upper_x < " + info.getBounds().getMinX());
                        out.write(")");
                        out.write(" AND ");
                        out.write("NOT");
                        out.write("(");
                            out.write(" lower_y > " + info.getBounds().getMaxY());
                            out.write(" OR ");
                            out.write(" upper_y < " + info.getBounds().getMinY());
                        out.write(")");
                    out.write(")");
                }                
                else
                    throw new TerralibFilterToSQLException(filter,"Unsupported geometry");

                if (_geometryTableName != null)
                    writeInClose();
            }
            catch (IOException e)
            {
                throw new TerralibFilterToSQLException(filter,"Error writing query to Writer.");
            }
            
            return extraData;  
        }
        
        /* (non-Javadoc)
         * @see org.geotools.data.jdbc.FilterToSQL#visit(org.opengis.filter.spatial.BBOX, java.lang.Object)
         */
        @Override
        public Object visit(BBOX filter, Object extraData)
        {
            return internalBBoxIntersectVisit(filter, extraData);            
        }
        
        @Override
        public Object visit(Within filter, Object extraData) 
        {
            QueryInfo info = analyzeFilter(filter);
            
            try
            {
                if (_geometryTableName != null)
                    writeInOpen();
                
                if (isPoint(info.getGeometryBinding()))
                {
                    out.write("(");
                    out.write(" x > " + info.getBounds().getMinX());
                    out.write(" AND x < " + info.getBounds().getMaxX());
                    out.write(" AND y > " + info.getBounds().getMinY());
                    out.write(" AND y < " + info.getBounds().getMaxY());
                    out.write(")");
                }
                else if (isLine(info.getGeometryBinding()) || (isPolygon(info.getGeometryBinding())))
                {
                    out.write("(");
                    out.write(" lower_x > " + info.getBounds().getMinX());
                    out.write(" AND upper_x < " + info.getBounds().getMaxX());
                    out.write(" AND lower_y > " + info.getBounds().getMinY());
                    out.write(" AND upper_y < " + info.getBounds().getMaxY());
                    out.write(")");
                }                
                else
                    throw new TerralibFilterToSQLException(filter,"Unsupported geometry");

                if (_geometryTableName != null)
                    writeInClose();
                
            }
            catch (IOException e)
            {
                throw new TerralibFilterToSQLException(filter,"Error writing query to Writer.");
            }
            
            return extraData;
        }        
        
        protected void writeInOpen() throws IOException
        {
            out.write(" OBJECT_ID IN ");
            out.write("(");
            out.write(" SELECT DISTINCT OBJECT_ID FROM ");
            out.write(_geometryTableName);
            out.write(" WHERE ");
        }
        
        protected void writeInClose() throws IOException
        {
            out.write(")");
        }
        
        
        protected boolean isPoint(Class<?> bindingClass)
        {
            return ((Point.class.isAssignableFrom(bindingClass))|| (MultiPoint.class.isAssignableFrom(bindingClass)));
        }
        
        protected boolean isLine(Class<?> bindingClass)
        {
            return ((LineString.class.isAssignableFrom(bindingClass))|| (MultiLineString.class.isAssignableFrom(bindingClass)));
        }

        protected boolean isPolygon(Class<?> bindingClass)
        {
            return ((Polygon.class.isAssignableFrom(bindingClass))|| (MultiPolygon.class.isAssignableFrom(bindingClass)));
        }
        
        protected QueryInfo analyzeFilter(BinarySpatialOperator filter)
        {
            Expression property;
            Expression geometry;
            Class<?> geometryBinding;
            
            if (filter.getExpression1() == null)
            {
                FilterFactory2 ff = new FilterFactoryImpl();
                property = ff.property(featureType.getGeometryDescriptor().getLocalName());
                geometry = filter.getExpression2();
            }
            else if (filter.getExpression2() == null)
            {
                FilterFactory2 ff = new FilterFactoryImpl();
                property = ff.property(featureType.getGeometryDescriptor().getLocalName());
                geometry = filter.getExpression1();
            }
            else if (filter.getExpression1() instanceof PropertyName)
            {
                property = filter.getExpression1();
                geometry = filter.getExpression2();
            }
            else if (filter.getExpression2() instanceof PropertyName) 
            {
                geometry = filter.getExpression1();
                property = filter.getExpression2();
            }
            else
                throw new TerralibFilterToSQLException(filter,"Didn't find a expression for the geometry attribute name");
                
            if (geometry == null)
                throw new TerralibFilterToSQLException(filter,"The expression that should contain the geometry is null.");
            
            AttributeDescriptor attType = (AttributeDescriptor)property.evaluate(featureType);
            if (attType != null)
                geometryBinding = attType.getType().getBinding();
            else
                throw new TerralibFilterToSQLException(filter,"Didn't find the geometry binding class");
            
            if (!Geometry.class.isAssignableFrom(geometryBinding))
                throw new TerralibFilterToSQLException(filter,"Didn't find the geometry binding class");
            
            ReferencedEnvelope envelopeBound = null;
            
            Object value = geometry.evaluate(featureType);
            if (value instanceof Polygon)
            {
                Polygon bounds = (Polygon)geometry.evaluate(featureType);
                //we can't retrieve the CRS, will assume it's the feature type's CRS.
                envelopeBound = new ReferencedEnvelope(bounds.getEnvelopeInternal(),featureType.getCoordinateReferenceSystem());
                
            }
            else if (value instanceof ReferencedEnvelope)
                envelopeBound = (ReferencedEnvelope) value;
            else
                throw new TerralibFilterToSQLException(filter,"Unknown geometry evaluate type");
            
            return new QueryInfo(envelopeBound,geometryBinding);
        }
        
        private static class QueryInfo
        {
            private Class<?> _geometryBinding;
            private ReferencedEnvelope _bounds;

            public QueryInfo(ReferencedEnvelope bounds, Class<?> geometryBinding)
            {
                _geometryBinding = geometryBinding;
                _bounds = bounds;
            }
            
            public Class<?> getGeometryBinding()
            {
                return _geometryBinding;
            }
            
            public ReferencedEnvelope getBounds()
            {
                return _bounds;
            }
        }
    }
}
