package org.geotools.data.terralib.query.portal;

import java.io.IOException;

import org.geotools.data.terralib.geometry.MultiTextGeometry;

import com.vividsolutions.jts.geom.MultiLineString;
import com.vividsolutions.jts.geom.MultiPoint;
import com.vividsolutions.jts.geom.MultiPolygon;

/**
 * This interface represents a wrapper on top a ResultSet used to read
 * Terralib attribute values
 * @author fmoura
 *
 */
public interface Portal 
{
	/**
	 * Retrieves a String value
	 * @param fieldName The field name
	 * @return The String value
	 */
	String getString(String fieldName);
	
	/**
	 * Retrieves a integer value
	 * @param fieldName The field name
	 * @return The integer value
	 */
	int getInt(String fieldName);
	
	/**
	 * Retrieves a real number value
	 * @param fieldName The field name
	 * @return
	 */
	double getDouble(String fieldName);
	
	/**
	 * Retrieves a date value
	 * @param fieldName the field name
	 * @return A String representation of the field value in the format "DD/MM/YYYY HH:MM:SS"
	 */
	String getDate(String fieldName);
	
	/**
	 * Closes the result set
	 */
	void close();
	
	/**
	 * Check if the result set is closed
	 * @return
	 */
	boolean isClosed();
	
	/**
	 * Check is there is a next row to be read.
	 * @return
	 */
	boolean hasNext();
	
	/**
	 * Retrieves a MultiPolygon value from the Terralib default multipolygon attribute. 
	 * It also advances to the next row. 
	 * @return The MultiPolygonValue
	 * @throws IOException If an error occurs reading the polygon
	 */
	MultiPolygon fetchPolygon() throws IOException;
	
	/**
	 * Retrieves a MultiLineString value from the Terralib default multiline attribute. 
	 * It also advances to the next row.
	 * @return The MultiLineString value
	 * @throws IOException If an error occurs reading the line
	 */
	MultiLineString fetchLine() throws IOException;
	
	/**
	 * Retrieves a MultiPoint value from the Terralib default multipoint attribute. 
	 * It also advances to the next row.
	 * @return The MultiPoint value
	 * @throws IOException If an error occurs reading the point
	 */
	MultiPoint fetchPoint() throws IOException;
	
	/**
	 * Retrieves a MultiTextGeometry value from the Terralib default multitext attribute. 
	 * It also advances to the next row. 
	 * @return The MultiTextValue
	 * @throws IOException If an error occurs reading the text
	 */
	MultiTextGeometry fetchText() throws IOException;
	
	//TODO: adicionar um mtodo finalizeConnection ou algo assim
	
	/**
	 * Finalizes the portal, free everything necessary.
	 */
	void finalizePortal();
}
