package csbase.client.applications.flowapplication;

import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.geom.AffineTransform;
import java.awt.geom.Rectangle2D;
import java.awt.image.ImageObserver;
import java.text.MessageFormat;

import tecgraf.vix.VO;

/**
 * VO de imagem.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class ImageVO extends VO {

  /**
   * Padding: sul
   */
  private double bottomPadding;

  /**
   * Altura
   */
  private double height;

  /**
   * Imagem
   */
  private Image image;

  /**
   * Padding: esquerda
   */
  private double leftPadding;

  /**
   * Padding: direita
   */
  private double rightPadding;

  /**
   * Padding: norte
   */
  private double topPadding;

  /**
   * Largura
   */
  private double width;

  /**
   * Posio: X
   */
  private double x;

  /**
   * Posio: Y
   */
  private double y;

  /**
   * Construtor
   * 
   * @param image imagem
   */
  public ImageVO(final Image image) {
    setImage(image);
    setWidth(this.image.getWidth(createImageObserver()));
    setHeight(this.image.getWidth(createImageObserver()));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void callbackRepaint(final Graphics2D g) {
    super.callbackRepaint(g);
    final AffineTransform matrix =
      new AffineTransform(1, 0, 0, 1, getX() + this.rightPadding, getY()
        + this.topPadding);
    g.drawImage(this.image, matrix, createImageObserver());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Rectangle2D getBounds2D() {
    return new Rectangle2D.Double(getX(), getY(), getWidth(), getHeight());
  }

  /**
   * Consulta atributo: altura
   * 
   * @return atributo
   */
  public double getHeight() {
    return this.height;
  }

  /**
   * Consulta atributo: largura
   * 
   * @return atributo
   */
  public double getWidth() {
    return this.width;
  }

  /**
   * Consulta atributo: X
   * 
   * @return atributo
   */
  public double getX() {
    return this.x;
  }

  /**
   * Consulta atributo: Y
   * 
   * @return atributo
   */
  public double getY() {
    return this.y;
  }

  /**
   * Ajusta padding: sul.
   * 
   * @param bottomPadding padding.
   */
  public void setBottomPadding(final double bottomPadding) {
    if (bottomPadding <= 0.0) {
      final String errorMessage =
        MessageFormat.format(
          "O parmetro bottomPadding ({0}) tem que ser positivo.",
          bottomPadding);
      throw new IllegalArgumentException(errorMessage);
    }
    this.bottomPadding = bottomPadding;
    this.height += this.bottomPadding;
  }

  /**
   * Ajusta centro em X
   * 
   * @param x centro
   */
  public void setCenterX(final double x) {
    this.x = x - (getWidth() / 2.0);
  }

  /**
   * Ajusta centro em Y
   * 
   * @param y centro
   */
  public void setCenterY(final double y) {
    this.y = y - (getHeight() / 2.0);
  }

  /**
   * Ajusta padding: esquerda.
   * 
   * @param leftPadding padding.
   */
  public void setLeftPadding(final double leftPadding) {
    if (leftPadding <= 0.0) {
      final String errorMessage =
        MessageFormat.format(
          "O parmetro leftPadding ({0}) tem que ser positivo.", leftPadding);
      throw new IllegalArgumentException(errorMessage);
    }
    this.leftPadding = leftPadding;
    this.width += this.leftPadding;
  }

  /**
   * Ajusta padding: direita.
   * 
   * @param rightPadding padding.
   */
  public void setRightPadding(final double rightPadding) {
    if (rightPadding <= 0.0) {
      final String errorMessage =
        MessageFormat.format(
          "O parmetro rightPadding ({0}) tem que ser positivo.", rightPadding);
      throw new IllegalArgumentException(errorMessage);
    }
    this.rightPadding = rightPadding;
    this.width += this.rightPadding;
  }

  /**
   * Ajusta padding: norte.
   * 
   * @param topPadding padding.
   */
  public void setTopPadding(final double topPadding) {
    if (topPadding <= 0.0) {
      final String errorMessage =
        MessageFormat.format(
          "O parmetro topPadding ({0}) tem que ser positivo.", topPadding);
      throw new IllegalArgumentException(errorMessage);
    }
    this.topPadding = topPadding;
    this.height += this.topPadding;
  }

  /**
   * Ajusta posio X
   * 
   * @param x X
   */
  public void setX(final double x) {
    this.x = x;
  }

  /**
   * Ajusta posio Y
   * 
   * @param y Y
   */
  public void setY(final double y) {
    this.y = y;
  }

  /**
   * Cria observador de imagem.
   * 
   * @return observador
   */
  private ImageObserver createImageObserver() {
    return new ImageObserver() {
      @Override
      public boolean imageUpdate(final Image img, final int infoflags,
        final int imgX, final int imgY, final int imgWidth, final int imgHeight) {
        if ((infoflags & HEIGHT) == HEIGHT) {
          setHeight(imgHeight);
          return true;
        }
        if ((infoflags & WIDTH) == WIDTH) {
          setWidth(imgWidth);
          return true;

        }

        if ((infoflags & ALLBITS) == WIDTH) {
          return true;
        }

        return false;
      }

    };
  }

  /**
   * Ajusta altura.
   * 
   * @param height altura
   */
  private void setHeight(final double height) {
    if (height <= 0.0) {
      final String errorMessage =
        MessageFormat.format("O parmetro height ({0}) tem que ser positivo.",
          height);
      throw new IllegalArgumentException(errorMessage);
    }
    this.height = height;
  }

  /**
   * Ajusta imagem.
   * 
   * @param image imagem
   */
  private void setImage(final Image image) {
    if (image == null) {
      throw new IllegalArgumentException("O parmetro image est nulo.");
    }
    this.image = image;
  }

  /**
   * Ajusta largura.
   * 
   * @param width largura
   */
  private void setWidth(final double width) {
    if (width <= 0.0) {
      final String errorMessage =
        MessageFormat.format("O parmetro width ({0}) tem que ser positivo.",
          width);
      throw new IllegalArgumentException(errorMessage);
    }
    this.width = width;
  }
}
