/*
 * FilePropertiesDialog.java
 * 
 * $Author: oikawa $ $Revision: 150777 $ - $Date: 2007-03-12 15:18:17 -0300
 * (Mon, 12 Mar 2007) $
 */
package csbase.client.project;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.MessageFormat;

import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.SwingConstants;

import tecgraf.javautils.core.lng.FormatUtils;
import tecgraf.javautils.core.lng.LNG;
import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.desktop.RemoteTask;
import csbase.client.util.ClientUtilities;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectFileType;
import csbase.logic.User;
import csbase.logic.Utilities;

/**
 * A classe <code>FilePropertiesDialog</code> modela uma janela que exibe as
 * propriedades de um <code>ProjectFile</code>.
 */
public class FilePropertiesDialog {

  /**
   * Dilogo principal.
   */
  private DesktopComponentDialog propertiesDialog = null;

  /**
   * Janela a partir do qual esse dilogo foi chamado.
   */
  private Window owner = null;

  /**
   * O boto default que fecha o dilogo.
   */
  private JButton closeButton;

  /**
   * Projeto onde est o arquivo.
   */
  private CommonClientProject project;

  /**
   * O arquivo cujas propriedades vo ser exibidas.
   */
  private ClientProjectFile file;

  /**
   * rea de descrio do arquivo
   */
  private JTextArea descriptionArea;

  /**
   * Nome do arquivo.
   */
  private String fileName;

  /**
   * Tamanho do arquivo.
   */
  private long fileSize;

  /**
   * Identificador do usurio que criou o arquivo.
   */
  private Object fileOwner;

  /**
   * Data de criao do arquivo.
   */
  private long fileCreationDate;

  /**
   * Data da ltima atualizao do arquivo.
   */
  private long fileLastUpdateDate;

  /**
   * Mostra a janela com as propriedades de um arquivo de projeto.
   * 
   * @param owner A janela a partir do qual esse dilogo foi chamado.
   * @param project O projeto no qual est o arquivo.
   * @param file O arquivo cujas propriedades vo ser exibidas.
   */
  public static void show(Window owner, CommonClientProject project,
    ClientProjectFile file) {
    new FilePropertiesDialog(owner, project, file);
  }

  /**
   * Cria o dilogo.
   * 
   * @return dilogo.
   */
  private DesktopComponentDialog makeDialog() {
    /* Cria um dilogo no modal */
    DesktopComponentDialog dialog =
      new DesktopComponentDialog(owner, getTitle());
    dialog.setModal(false);
    /* Cria o painel principal do dilogo */
    JPanel mainPanel = new JPanel(new BorderLayout());
    mainPanel.add(makePropertiesPanel(dialog.getBackground()));
    mainPanel.add(makeButtonPanel(), BorderLayout.SOUTH);
    dialog.getContentPane().add(mainPanel);
    dialog.pack();
    dialog.center(owner);
    dialog.getRootPane().setDefaultButton(closeButton);
    return dialog;
  }

  /**
   * Cria o painel com as propriedades do arquivo.
   * 
   * @param background A cor de fundo usada pelos componentes.
   * 
   * @return painel de propriedades.
   */
  private JPanel makePropertiesPanel(Color background) {
    ImageIcon icon = ClientProjectFileProperties.getImageIcon(this.file);
    String description;
    try {
      description = this.file.getDescription();
    }
    catch (Exception e) {
      e.printStackTrace();
      description = "";
    }
    String accessMode = ClientProjectFileProperties.getAccessMode(this.file);
    String typeName;
    try {
      ProjectFileType type = ProjectFileType.getFileType(file.getType());
      typeName = type.getDescription();
    }
    catch (Exception e) {
      typeName = " ";
    }
    /* Campo com o nome do arquivo */
    JTextField nameTextField = new JTextField(fileName);
    nameTextField.setEditable(false);
    /* Campo com a localizao */
    String fileLocalization = buildLocalization();
    JTextField pathTextField = new JTextField(fileLocalization);
    pathTextField.setEditable(false);
    /* Tipo do arquivo */
    JLabel typeLabel = new JLabel(icon, SwingConstants.LEFT);
    JTextArea typeTextArea = new JTextArea(" " + typeName);
    typeTextArea.setBackground(background);
    typeTextArea.setEditable(false);
    JPanel typePanel = new JPanel(new FlowLayout(FlowLayout.LEADING, 0, 0));
    typePanel.add(typeLabel);
    typePanel.add(typeTextArea);
    /* Indicao se o arquivo  s leitura ou leitura e escrita */
    JTextArea readOnlyTextArea = new JTextArea(accessMode);
    readOnlyTextArea.setBackground(background);
    readOnlyTextArea.setEditable(false);
    /* Tamanho do arquivo */
    JTextArea sizeTextArea = new JTextArea(FormatUtils.formatSize(fileSize, 2));
    sizeTextArea.setBackground(background);
    sizeTextArea.setEditable(false);
    /* Data de criao do arquivo */
    String creationDate = " ";
    if (fileCreationDate != 0) {
      creationDate = Utilities.getFormattedDate(fileCreationDate);
    }
    JTextArea creationDateTextArea = new JTextArea(creationDate);
    creationDateTextArea.setBackground(background);
    creationDateTextArea.setEditable(false);

    /* Data da ltima alterao do arquivo */
    String lastUpdateDate = " ";
    if (fileLastUpdateDate != 0) {
      lastUpdateDate = Utilities.getFormattedDate(fileLastUpdateDate);
    }
    JTextArea lastUpdateDateTextArea = new JTextArea(lastUpdateDate);
    lastUpdateDateTextArea.setBackground(background);
    lastUpdateDateTextArea.setEditable(false);

    /* Usurio que criou o arquivo */
    String userName;
    try {
      userName = User.getName(fileOwner);
      if (userName == null) {
        userName = " ";
      }
    }
    catch (Exception e) {
      userName = " ";
    }
    JTextArea ownerTextArea = new JTextArea(userName);
    ownerTextArea.setBackground(background);
    ownerTextArea.setEditable(false);
    descriptionArea = new JTextArea(description);
    descriptionArea.setBackground(background);
    descriptionArea.setEditable(false);
    descriptionArea.setFont(new Font("Monospaced", Font.PLAIN, 12));
    descriptionArea.setLineWrap(true);
    descriptionArea.setWrapStyleWord(true);
    JScrollPane textScrollPane = new JScrollPane(descriptionArea);
    Dimension scrDim = new Dimension(400, 200);
    textScrollPane.setSize(scrDim);
    textScrollPane.setPreferredSize(scrDim);
    textScrollPane.setMinimumSize(scrDim);
    textScrollPane.setBorder(BorderFactory.createTitledBorder(LNG
      .get("PRJ_FILE_DESCRIPTION")));
    /* Monta o panel usando grid layout */
    JPanel panel =
      buildPropertiesLayout(nameTextField, pathTextField, typePanel,
        readOnlyTextArea, sizeTextArea, creationDateTextArea,
        lastUpdateDateTextArea, ownerTextArea, textScrollPane);
    return panel;
  }

  /**
   * Monta o layout do painel de propriedades.
   * 
   * @param nameTextField campo de nome.
   * @param pathTextField campo de localizao.
   * @param typePanel campo de tipo.
   * @param readOnlyTextArea campo leitura-somente.
   * @param sizeTextArea campo de tamanho.
   * @param creationDateTextArea campo data de criao.
   * @param lastUpdateDateTextArea campo data da ltima atualizao.
   * @param ownerTextArea campo do autor.
   * @param textScrollPane rea de descrio.
   * 
   * @return painel de propriedades.
   */
  private JPanel buildPropertiesLayout(JTextField nameTextField,
    JTextField pathTextField, JPanel typePanel, JTextArea readOnlyTextArea,
    JTextArea sizeTextArea, JTextArea creationDateTextArea,
    JTextArea lastUpdateDateTextArea, JTextArea ownerTextArea,
    JScrollPane textScrollPane) {
    JPanel panel = new JPanel(new GridBagLayout());
    GridBagConstraints c = new GridBagConstraints();

    // Rtulo para Nome
    c.gridx = 0;
    c.gridy = 0;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 0;
    c.weighty = 0;
    c.insets = new Insets(2, 2, 2, 5); // tlrb
    c.fill = GridBagConstraints.NONE;
    c.anchor = GridBagConstraints.NORTHWEST;
    JLabel nameLabel = new JLabel(LNG.get("PRJ_FILE_NAME"));
    panel.add(nameLabel, c);

    // Campo Nome
    c.gridx = 1;
    c.gridy = 0;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 10;
    c.weighty = 0;
    c.insets = new Insets(2, 2, 2, 5); // tlrb
    c.fill = GridBagConstraints.HORIZONTAL;
    c.anchor = GridBagConstraints.NORTHWEST;
    panel.add(nameTextField, c);

    // Rtulo para Localizao
    c.gridx = 0;
    c.gridy = 1;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 0;
    c.weighty = 0;
    c.insets = new Insets(2, 2, 2, 5); // tlrb
    c.fill = GridBagConstraints.NONE;
    c.anchor = GridBagConstraints.NORTHWEST;
    JLabel pathTextLabel = new JLabel(LNG.get("PRJ_FILE_LOCALIZATION"));
    panel.add(pathTextLabel, c);

    // Campo Localizao
    c.gridx = 1;
    c.gridy = 1;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 0;
    c.weighty = 0;
    c.insets = new Insets(2, 2, 2, 5); // tlrb
    c.fill = GridBagConstraints.HORIZONTAL;
    c.anchor = GridBagConstraints.NORTHWEST;
    panel.add(pathTextField, c);

    // Rtulo para Tipo
    c.gridx = 0;
    c.gridy = 2;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 0;
    c.weighty = 0;
    c.insets = new Insets(2, 2, 2, 5); // tlrb
    c.fill = GridBagConstraints.NONE;
    c.anchor = GridBagConstraints.NORTHWEST;
    JLabel typePanelLabel = new JLabel(LNG.get("PRJ_FILE_TYPE"));
    panel.add(typePanelLabel, c);

    // Campo Tipo
    c.gridx = 1;
    c.gridy = 2;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 0;
    c.weighty = 0;
    c.insets = new Insets(2, 2, 2, 5); // tlrb
    c.fill = GridBagConstraints.HORIZONTAL;
    c.anchor = GridBagConstraints.NORTHWEST;
    panel.add(typePanel, c);

    // Campo Leitura Somente
    c.gridx = 1;
    c.gridy = 3;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 0;
    c.weighty = 0;
    c.insets = new Insets(2, 2, 2, 5); // tlrb
    c.fill = GridBagConstraints.HORIZONTAL;
    c.anchor = GridBagConstraints.NORTHWEST;
    panel.add(readOnlyTextArea, c);

    // Rtulo para Tamanho
    c.gridx = 0;
    c.gridy = 4;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 0;
    c.weighty = 0;
    c.insets = new Insets(2, 2, 2, 5); // tlrb
    c.fill = GridBagConstraints.NONE;
    c.anchor = GridBagConstraints.NORTHWEST;
    JLabel sizeLabel = new JLabel(LNG.get("PRJ_FILE_SIZE"));
    panel.add(sizeLabel, c);

    // Campo Tamanho
    c.gridx = 1;
    c.gridy = 4;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 0;
    c.weighty = 0;
    c.insets = new Insets(2, 2, 2, 5); // tlrb
    c.fill = GridBagConstraints.HORIZONTAL;
    c.anchor = GridBagConstraints.NORTHWEST;
    panel.add(sizeTextArea, c);

    // Rtulo para data de criao
    c.gridx = 0;
    c.gridy = 5;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 0;
    c.weighty = 0;
    c.insets = new Insets(2, 2, 2, 5); // tlrb
    c.fill = GridBagConstraints.NONE;
    c.anchor = GridBagConstraints.NORTHWEST;
    JLabel creationDateLabel = new JLabel(LNG.get("PRJ_FILE_CREATION_DATE"));
    panel.add(creationDateLabel, c);

    // Campo data de criao
    c.gridx = 1;
    c.gridy = 5;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 0;
    c.weighty = 0;
    c.insets = new Insets(2, 2, 2, 5); // tlrb
    c.fill = GridBagConstraints.HORIZONTAL;
    c.anchor = GridBagConstraints.NORTHWEST;
    panel.add(creationDateTextArea, c);

    // Rtulo para data da ltima atualizao
    c.gridx = 0;
    c.gridy = 6;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 0;
    c.weighty = 0;
    c.insets = new Insets(2, 2, 2, 5); // tlrb
    c.fill = GridBagConstraints.NONE;
    c.anchor = GridBagConstraints.NORTHWEST;
    JLabel lastUpdateDateLabel =
      new JLabel(LNG.get("PRJ_FILE_LAST_UPDATE_DATE"));
    panel.add(lastUpdateDateLabel, c);

    // Campo data da ltima atualizao
    c.gridx = 1;
    c.gridy = 6;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 0;
    c.weighty = 0;
    c.insets = new Insets(2, 2, 2, 5); // tlrb
    c.fill = GridBagConstraints.HORIZONTAL;
    c.anchor = GridBagConstraints.NORTHWEST;
    panel.add(lastUpdateDateTextArea, c);

    // Rtulo para autor
    c.gridx = 0;
    c.gridy = 7;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 0;
    c.weighty = 0;
    c.insets = new Insets(2, 2, 2, 5); // tlrb
    c.fill = GridBagConstraints.NONE;
    c.anchor = GridBagConstraints.NORTHWEST;
    JLabel ownerLabel = new JLabel(LNG.get("PRJ_FILE_AUTHOR"));
    panel.add(ownerLabel, c);

    // Campo autor
    c.gridx = 1;
    c.gridy = 7;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 0;
    c.weighty = 0;
    c.insets = new Insets(2, 2, 2, 5); // tlrb
    c.fill = GridBagConstraints.HORIZONTAL;
    c.anchor = GridBagConstraints.NORTHWEST;
    panel.add(ownerTextArea, c);

    // Rtulo para fonte
    c.gridx = 0;
    c.gridy = 8;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 0;
    c.weighty = 0;
    c.insets = new Insets(2, 2, 2, 5); // tlrb
    c.fill = GridBagConstraints.NONE;
    c.anchor = GridBagConstraints.NORTHWEST;
    JLabel sourceLabel = new JLabel(LNG.get("PRJ_FILE_SOURCE"));
    panel.add(sourceLabel, c);

    // rea de descrio
    c.gridx = 0;
    c.gridy = 9;
    c.gridwidth = 2;
    c.gridheight = 8;
    c.weightx = 10;
    c.weighty = 10;
    c.insets = new Insets(8, 2, 2, 5); // tlrb
    c.fill = GridBagConstraints.BOTH;
    c.anchor = GridBagConstraints.NORTHWEST;
    panel.add(textScrollPane, c);
    return panel;
  }

  /**
   * Cria o painel com os botes.
   * 
   * @return painel de botes.
   */
  private JPanel makeButtonPanel() {
    JPanel panel = new JPanel();
    JButton descriptionButton =
      new JButton(LNG.get("PRJ_FILE_ADD_DESCRIPTION"));
    panel.add(descriptionButton);
    descriptionButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        AppendDescriptionDialog appendDialog =
          new AppendDescriptionDialog(FilePropertiesDialog.this);
        appendDialog.setVisible(true);
      }
    });
    closeButton = new JButton(LNG.get("UTIL_CLOSE"));
    panel.add(closeButton);
    closeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        propertiesDialog.close();
      }
    });
    return panel;
  }

  /**
   * Retorna dilogo principal.
   * 
   * @return Dilogo principal.
   */
  public JDialog getDialog() {
    return propertiesDialog;
  }

  /**
   * Informa o ttulo que vai ser utilizado na janela de exibio de
   * propriedades de um arquivo.
   * 
   * @return O ttulo da janela.
   */
  protected String getTitle() {
    String title =
      MessageFormat.format(LNG.get("project.file_see_properties_title"),
        new Object[] { fileName });
    return ClientUtilities.addSystemNameToTitle(title);
  }

  /**
   * Recupera as propriedades do arquivo.
   */
  private void retrieveFileProperties() {
    RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      public void performTask() throws Exception {
        file.updateInfo();
        fileName = file.getName();
        fileSize = file.size();
        fileOwner = file.whoCreated();
        fileCreationDate = file.getCreationDate();
        fileLastUpdateDate = file.getModificationDate();
      }
    };
    task.execute(owner, getTitle(),
      LNG.get("PRJ_WAITING_OBTAIN_FILE_PROPERTIES"));
  }

  /**
   * Constri a representao de onde se encontra um determinado arquivo. A
   * String resultante possui a seguinte forma: [nome-do-projeto]:
   * caminho-dos-diretrios
   * 
   * @return A string que representa a localizao de um arquivo na rvore do
   *         projeto.
   */
  private String buildLocalization() {
    String path = "[" + project.getName() + "]" + ": ";
    String[] filePath = file.getPath();
    for (int i = 1; i < (filePath.length - 1); i++) {
      path += (filePath[i] + "/");
    }
    return path;
  }

  /**
   * Constri o dilogo de exibio de propriedades.
   * 
   * @param owner A janela pai a partir da qual esse dilogo  chamado.
   * @param project O projeto no qual est o arquivo.
   * @param file O arquivo cujas propriedades vo ser exibidas.
   */
  private FilePropertiesDialog(Window owner, CommonClientProject project,
    ClientProjectFile file) {
    this.owner = owner;
    this.project = project;
    this.file = file;
    retrieveFileProperties();
    propertiesDialog = makeDialog();
    propertiesDialog.setVisible(true);
  }

  /**
   * Atualiza a rea de descrio do arquivo exibida no dilogo.
   */
  public void updateDescription() {
    String description;
    try {
      description = this.file.getDescription();
    }
    catch (Exception e) {
      e.printStackTrace();
      description = "";
    }
    descriptionArea.setText(description);
  }

  /**
   * Obtm o arquivo cujas propriedades esto sendo exibidas.
   * 
   * @return arquivo.
   */
  public ClientProjectFile getFile() {
    return file;
  }
}
