package csbase.client.ias;

import java.io.File;
import java.util.Collection;
import java.util.Collections;

import javax.swing.filechooser.FileFilter;

/**
 * A classe <code>Filter</code>  usada pelo JFileChooser para filtrar o
 * conjunto de arquivos exibidos para o usurio.
 */
public class Filter extends FileFilter {
  /**
   * Extenso
   */
  private Collection<String> extensions;

  /**
   * Descrio.
   */
  private String description;

  /**
   * Indica se o filtro aceita arquivos sem extenso.
   */
  protected boolean acceptNoExtension = true;

  /**
   * Cria o filtro para uma determinada extenso.
   * O filtro aceitar tambm arquivos sem extenso.
   * 
   * @param extension a extenso do arquivo que se deseja filtrar.
   * @param description descrio do tipo de arquivo.
   */
  public Filter(String extension, String description) {
    this(extension, description, true);
  }

  /**
   * Cria o filtro com um conjunto de extenses.
   * O filtro aceitar tambm arquivos sem extenso.
   * 
   * @param extensions as extenses de interesse 
   * @param description descrio para o filtro
   */
  public Filter(final Collection<String> extensions, final String description) {
    this(extensions, description, true);
  }

  /**
   * Cria o filtro para uma determinada extenso.
   * 
   * @param extension a extenso do arquivo que se deseja filtrar.
   * @param description descrio do tipo de arquivo.
   * @param acceptNoExtension indica se o filtro aceita arquivos sem extenso
   */
  public Filter(
        final String extension,
        final String description,
        final boolean acceptNoExtension) {
    this(Collections.singletonList(extension), description, acceptNoExtension);
  }

  /**
   * Cria o filtro com um conjunto de extenses.
   * 
   * @param extensions as extenses de interesse 
   * @param description descrio para o filtro
   * @param acceptNoExtension indica se o filtro aceita arquivos sem extenso
   */
  public Filter(
      final Collection<String> extensions,
      final String description,
      final boolean acceptNoExtension) {
    
    this.extensions = extensions;
    this.description = description;
    this.acceptNoExtension = acceptNoExtension;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean accept(File f) {
    if (f.isDirectory()) {
      return true;
    }
    if (isAccept(f)) {
      return true;
    }
    return false;
  }

  /**
   * Verifica se um determinado arquivo  do tipo aceito por este filtro.
   * 
   * @param file o arquivo que se deseja verificar.
   * 
   * @return true, se o arquivo for aceito pelo filtro, false caso contrrio.
   */
  public boolean isAccept(File file) {
    String ext = null;
    String fileName = file.getName();
    int i = fileName.lastIndexOf('.');
    if (i == -1 && acceptNoExtension) {
      return true;
    }
    if (i > 0 && i < fileName.length() - 1) {
      ext = fileName.substring(i + 1).toLowerCase();
      if (extensions.contains(ext)) {
        return true;
      }
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getDescription() {
    return description;
  }
}
