/*
 * $Id: ImportUsersDialog.java 167364 2015-08-11 22:54:49Z pietroguedes $
 */

package csbase.client.ias;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.BorderUtil;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.table.DefaultObjectTableProvider;
import tecgraf.javautils.gui.table.ObjectTableModel;
import tecgraf.javautils.gui.table.ObjectTableProvider;
import tecgraf.javautils.gui.table.SortableTable;
import tecgraf.javautils.parsers.tsv.TsvParser;
import csbase.client.Client;
import csbase.client.ClientLocalFile;
import csbase.client.applications.ApplicationImages;
import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.client.kernel.ClientException;
import csbase.client.project.ClientProjectFileType;
import csbase.client.project.ProjectFileChooser;
import csbase.client.project.ProjectFileChooserOpen;
import csbase.client.project.ProjectTreePath;
import csbase.client.remote.srvproxies.UserGroupProxy;
import csbase.client.util.ClientUtilities;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.ClientFile;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectFileType;
import csbase.logic.User;
import csbase.logic.UserGroup;
import csbase.logic.UserInfo;

/**
 * Tela para importao de usurios a partir de arquivos CSV.
 */
public final class ImportUsersDialog {

  /**
   * Classe responsvel por exibir duas listas com o login de usurios.
   * 
   * @author ntl9
   */
  private class ImportedUserDialog {
    /**
     * Dialogo.
     */
    private DesktopComponentDialog dialog;

    /**
     * Dilogo-pai
     */
    private Window dialogOwner;

    /**
     * Cria o boto responsvel por fechar o dilogo.
     * 
     * @return boto.
     */
    private JButton createCloseButton() {
      final JButton closeButton = new JButton(LNG.get("IAS_CLOSE"));
      closeButton.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(final ActionEvent evento) {
          dialog.close();
        }
      });
      return closeButton;
    }

    /**
     * Criao do dilogo
     * 
     * @param insertedList lista de inseridos
     * @param editedList lista de editados.
     * @param discartedList lista de editados.
     */
    private void createDialog(final List<String> insertedList,
      final List<String> editedList, final List<String> discartedList) {
      // Cria e cofigura o deskComponentDialog
      dialog = new DesktopComponentDialog(this.dialogOwner);
      dialog.setTitle(LNG.get("IAS_IMPORT_RESULT"));
      dialog.setDefaultCloseOperation(DesktopComponentDialog.DISPOSE_ON_CLOSE);

      // Cria as lista que sero exibidas.
      final String insText = LNG.get("IAS_INSERTED");
      final JPanel insertedJList =
        createWidgetFromLoginList(insertedList, insText);
      final String edtText = LNG.get("IAS_EDITED");
      final JPanel editedJList = createWidgetFromLoginList(editedList, edtText);
      final String dscText = LNG.get("IAS_DISCARTED");
      final JPanel discartedJList =
        createWidgetFromLoginList(discartedList, dscText);

      final JPanel panel = new JPanel(new GridBagLayout());
      panel.add(insertedJList, new GBC(0, 0).both().center());
      panel.add(editedJList, new GBC(1, 0).both().center());
      panel.add(discartedJList, new GBC(2, 0).both().center());
      panel.add(createCloseButton(),
        new GBC(0, 1).width(3).center().insets(10, 0, 10, 0));
      dialog.getContentPane().add(panel);
    }

    /**
     * Ajustando a lista para no possuir seleo.
     * 
     * @param logins lista de logins
     * @param title ttulo
     * @return widget
     */
    private JPanel createWidgetFromLoginList(final List<String> logins,
      final String title) {
      final JPanel panel = new JPanel();
      final JLabel label = new JLabel(title);
      final JTextArea textArea = new JTextArea();

      textArea.setFont(new Font(Font.MONOSPACED, Font.PLAIN, 12));
      final StringBuffer buffer = new StringBuffer();
      for (final String login : logins) {
        buffer.append(login + "\n");
      }
      final String text = buffer.toString();
      textArea.setText(text);
      textArea.setCaretPosition(0);

      final JScrollPane jScrollPane = new JScrollPane(textArea);
      textArea.setMinimumSize(new Dimension(100, 50));
      textArea.setPreferredSize(new Dimension(200, 150));
      textArea.setEditable(false);
      textArea.setBackground(panel.getBackground());

      panel.setLayout(new BorderLayout());
      panel.add(label, BorderLayout.NORTH);
      panel.add(jScrollPane, BorderLayout.CENTER);

      final int d = 3;
      panel.setBorder(BorderFactory.createEmptyBorder(d * 2, d, d, d));

      return panel;
    }

    /**
     * Exibe o dilogo.
     */
    public void showDialog() {
      dialog.pack();
      dialog.setLocationRelativeTo(owner);
      dialog.setVisible(true);
    }

    /**
     * Constri um dilogo modal, com duas lista de usurios.
     * 
     * @param owner janela pai do dialogo.
     * @param insertedList lista dos usurios inseridos na importao.
     * @param editedList lista dos usurios alterados na importao.
     * @param discartedList lista de usurios descartados na importao.
     */
    public ImportedUserDialog(final Window owner,
      final List<String> insertedList, final List<String> editedList,
      final List<String> discartedList) {
      if (insertedList != null && editedList != null && discartedList != null) {
        this.dialogOwner = owner;
        createDialog(insertedList, editedList, discartedList);
      }
    }
  }

  /**
   * Resultados da operao de importar os usurios.
   */
  private enum ImportResults {
    /**
     * Todos foram importados.
     */
    ALL_IMPORTED,

    /**
     * Apenas alguns foram importados.
     */
    SOME_IMPORTED,

    /**
     * Nenhum foi importado.
     */
    NONE_IMPORTED
  }

  /**
   * Expresso regular usada para separar os e-mails que viro juntos como um
   * nico elemento de um arquivo CSV.
   */
  private static final String EMAILS_SPLIT_REGEX = "[,;\\s]+";

  /**
   * Classes das colunas da tabela de importao.
   */
  private static final Class<?>[] COL_CLASSES = new Class[] { String.class,
      String.class, String.class, String.class, String.class };

  /**
   * Ttulos das colunas da tabela de importao.
   */
  private static final String[] COL_NAMES = new String[] { LNG.get("IAS_USER"),
      LNG.get("IAS_USER_NAME"), LNG.get("IAS_USER_EMAIL"),
      LNG.get("IAS_USERGROUP"), LNG.get("IAS_USER_CREATION_DATE") };

  /**
   * Janela me
   */
  private final Window owner;

  /**
   * Label para a janela
   */
  private final String windowTitle;

  /**
   * Janela
   */
  private final DesktopComponentDialog window;

  /**
   * Boto: perfil
   */
  private JButton roleButton;

  /**
   * Boto: permisso
   */
  private JButton permissionButton;

  /**
   * Boto: ao
   */
  private JButton actionButton;

  /**
   * Campo: arquivo
   */
  private JTextField filePath;

  /**
   * Container
   */
  private final Container contentPane;

  /**
   * Arquivo corrente
   */
  private ClientFile currentFile;

  /**
   * Tabela
   */
  private SortableTable sortableTable;

  /**
   * Modelo de tabela
   */
  private ObjectTableModel<String[]> tableModel;

  /**
   * Dilogo: perfil
   */
  private final UserRoleDefinitionDialog userRoleDefinitionDialog;

  /**
   * Dilogo: permisso
   */
  private final UserPermissionDefinitionDialog userPermissionDefinitionDialog;

  /**
   * Infomativo do usurio.
   */
  private final UserInfo defaultUserInfo;

  /**
   * Tipo de arquivo de importao.
   */
  private static final String FILE_TYPE = "CSV";

  /**
   * Num. de colunas
   */
  private static final int COLUMN_COUNT = 5;

  /**
   * Coluna: login
   */
  private static final int LOGIN_COLUMN = 0;

  /**
   * Coluna: nome
   */
  private static final int NAME_COLUMN = 1;

  /**
   * Coluna: email
   */
  private static final int E_MAIL_COLUMN = 2;

  /**
   * Coluna: grupo
   */
  private static final int USERGROUP_COLUMN = 3;
  
  /**
   * Coluna: data de criao
   */
  private static final int CREATION_DATE_COLUMN = 4;

  /**
   * Listas dos novos usurios que foram importados.
   */
  private final List<String> insertedUserList = new ArrayList<String>();

  /**
   * Listas dos novos usurios que foram descartados.
   */
  private final List<String> discartedUserList = new ArrayList<String>();

  /**
   * Lista dos usurios que foram alterados com a importao.
   */
  private final List<String> editedUserList = new ArrayList<String>();

  /**
   * Cria o boto que efetivamente importa os usurios lidos do arquivo.
   */
  private void createActionButton() {
    actionButton = new JButton(LNG.get("IAS_IMPORT_USERS_ACTION_BUTTON"));
    actionButton.setEnabled(false);
    actionButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        final boolean useDateFromFile = showCreateDateQuestion();
        
        final RemoteTask<ImportResults> remoteTask =
          new RemoteTask<ImportResults>() {
            @Override
            protected void performTask() throws Exception {
              setResult(importUsers(useDateFromFile));
            }
          };
        remoteTask.execute(window, windowTitle,
          LNG.get("IAS_WAITING_USER_IMPORTING"));
        final ImportResults result = remoteTask.getResult();
        if (result == ImportResults.NONE_IMPORTED) {
          /*
           * nenhum usurio foi importado
           */
          StandardDialogs.showErrorDialog(window,
            LNG.get("IAS_IMPORT_USERS_TITLE"), LNG.get("IAS_IMPORT_ALL_FAILED"));
        }
        else {
          if (result == ImportResults.SOME_IMPORTED) {
            /*
             * avisamos que alguns usurios no foram importados
             */
            StandardDialogs.showWarningDialog(window,
              LNG.get("IAS_IMPORT_USERS_TITLE"),
              LNG.get("IAS_IMPORT_SOME_FAILED"));
          }
          final ImportedUserDialog importedUserDialog =
            new ImportedUserDialog(window, insertedUserList, editedUserList,
              discartedUserList);
          importedUserDialog.showDialog();
        }
        insertedUserList.clear();
        discartedUserList.clear();
        editedUserList.clear();
      }
    });
  }

  /**
   * Exibe um dilogo questionando se o usurio quer usar ou ignorar as datas de
   * criao dos usurios que esto no arquivo.
   * 
   * @return <code>True</code> se o usurio deseja usar as datas do arquivo,
   *          <code>false</code> para usar a data de hoje.
   */
  private boolean showCreateDateQuestion() {
    int answer = StandardDialogs.showYesNoDialog(window,
      LNG.get("IAS_IMPORT_CREATION_DATE_DIALOG_TITLE"),
      LNG.get("IAS_IMPORT_CREATION_DATE_DIALOG_QUESTION"));
    
    if(answer == StandardDialogs.YES_OPTION) {
      return true;
    }
    return false;
  }

  /**
   * Cria painel de botes.
   */
  private void createButtonPanel() {
    createRoleButton();
    createPermissionButton();
    createActionButton();
    final JButton closeButton = createCloseButton();
    ClientUtilities.adjustEqualSizes(roleButton, permissionButton,
      actionButton, closeButton);
    final JPanel buttonsPanel = new JPanel();
    buttonsPanel.add(roleButton);
    buttonsPanel.add(permissionButton);
    buttonsPanel.add(actionButton);
    buttonsPanel.add(closeButton);
    this.contentPane.add(buttonsPanel, BorderLayout.SOUTH);
  }

  /**
   * Cria boto: close
   * 
   * @return boto
   */
  private JButton createCloseButton() {
    final JButton closeButton = new JButton(LNG.get("IAS_CLOSE"));
    closeButton.setEnabled(true);
    closeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        window.close();
      }
    });
    return closeButton;
  }

  /**
   * Cria o painel de escolha do arquivo.
   */
  private void createFileChooserPanel() {
    final JPanel fileChooserPanel = new JPanel(new GridBagLayout());
    final JLabel theLabel = new JLabel(LNG.get("IAS_USERS_FILE") + ':');

    this.filePath = new JTextField(30);
    this.filePath.setEditable(false);

    final JButton findProjButton = createFindProjectButton();
    final JButton findLocalButton = createFindLocalButton();

    fileChooserPanel.add(theLabel, new GBC(0, 0));
    fileChooserPanel.add(filePath, new GBC(1, 0).horizontal()
      .insets(0, 5, 0, 0));
    fileChooserPanel.add(findProjButton, new GBC(2, 0).insets(0, 10, 0, 0));
    fileChooserPanel.add(findLocalButton, new GBC(3, 0).insets(0, 5, 0, 0));
    BorderUtil.setEtchedBorder(fileChooserPanel, null);

    this.contentPane.add(fileChooserPanel, BorderLayout.NORTH);
  }

  /**
   * Cria boto de find.
   * 
   * @return boto
   */
  private JButton createFindLocalButton() {
    final JButton findButton =
      ClientUtilities.createImageButton(ApplicationImages.ICON_OPEN_LOCAL_16);
    findButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent ae) {
        showFileChooser();
      }
    });
    return findButton;
  }

  /**
   * Cria boto de find.
   * 
   * @return boto
   */
  private JButton createFindProjectButton() {
    final JButton findButton =
      ClientUtilities.createImageButton(ApplicationImages.ICON_OPEN_16);
    findButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent ae) {
        showProjectFileChooser();
      }
    });

    final DesktopFrame desktopFrame = DesktopFrame.getInstance();
    if (desktopFrame == null || desktopFrame.getProject() == null) {
      findButton.setEnabled(false);
    }
    return findButton;
  }

  /**
   * Cria boto de permisses
   */
  private void createPermissionButton() {
    this.permissionButton = new JButton(LNG.get("IAS_USER_PERMISSIONS_BUTTON"));
    permissionButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        userPermissionDefinitionDialog.showDialog();
      }
    });
  }

  /**
   * Cria boto: perfil
   */
  private void createRoleButton() {
    this.roleButton = new JButton(LNG.get("IAS_USER_ROLES_BUTTON"));
    this.roleButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        userRoleDefinitionDialog.showDialog();
      }
    });
  }

  /**
   * Cria a tabela.
   */
  private void createTable() {
    final ObjectTableProvider provider = new DefaultObjectTableProvider() {
      /**
       * {@inheritDoc}
       */
      @Override
      public Object[] getCellValues(final Object item) {
        final Object[] line = (Object[]) item;
        
        String date = (String) line[CREATION_DATE_COLUMN];

        return (line == null) ? null
          : new Object[] { (String) line[LOGIN_COLUMN],
              (String) line[NAME_COLUMN], (String) line[E_MAIL_COLUMN],
              (String) line[USERGROUP_COLUMN], date };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public Class<?>[] getColumnClasses() {
        return COL_CLASSES;
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public String[] getColumnNames() {
        return COL_NAMES;
      }
    };
    this.tableModel =
      new ObjectTableModel<String[]>(new ArrayList<String[]>(), provider);
    this.sortableTable = new SortableTable(tableModel);
    this.sortableTable.addMouseListener(new MouseAdapter() {
      @Override
      public void mousePressed(final MouseEvent e) {
        updateActionButton();
      }
    });
  }

  /**
   * Cria o painel de usurios.
   */
  private void createUsersPanel() {
    final JPanel usersPanel = new JPanel();
    usersPanel.setLayout(new BorderLayout());
    createTable();
    final JScrollPane scrollPane = new JScrollPane(sortableTable);
    usersPanel.add(scrollPane, BorderLayout.CENTER);
    BorderUtil.setEtchedBorder(usersPanel, null);
    this.contentPane.add(usersPanel, BorderLayout.CENTER);
  }

  /**
   * Importa um usurio.
   * 
   * @param newUserInfo usurio que ser importado.
   * 
   * @throws Exception em caso de erro na importao.
   */
  private void importUser(final UserInfo newUserInfo) throws Exception {
    final User user = User.getUser(newUserInfo.getLogin());

    // Insere um novo usurio.
    if (user == null) {
      User.createUser(newUserInfo);
      this.insertedUserList.add(newUserInfo.getLogin());
    }

    // Atualiza um usurio j existente.
    else {
      this.updateUser(user, newUserInfo);
      this.editedUserList.add(user.getUserInfo().getLogin());
    }
  }

  /**
   * Importa os usurios selecioandos da lista de disponveis. Os usurios cujos
   * grupos no existam no sero criados.
   * 
   * @param useDateFromFile <code>True</code> se o usurio deseja usar as datas
   *        do arquivo, <code>false</code> para usar a data de hoje.
   * 
   * @return 0 se todos os usurios foram importados com sucesso, 1 se apenas
   *         alguns foram importados, 2 se nenhum foi importado
   * @throws Exception em caso de erro na importao de usurios.
   */
  private ImportResults importUsers(boolean useDateFromFile) throws Exception {
    final int[] selectedRows = this.sortableTable.getSelectedRows();
    final List<UserInfo> usersToImport = new ArrayList<UserInfo>();
    for (final int selectedRow : selectedRows) {
      final String login =
        (String) this.sortableTable.getValueAt(selectedRow,
          ImportUsersDialog.LOGIN_COLUMN);
      final String name =
        (String) this.sortableTable.getValueAt(selectedRow,
          ImportUsersDialog.NAME_COLUMN);
      final String emails =
        (String) this.sortableTable.getValueAt(selectedRow,
          ImportUsersDialog.E_MAIL_COLUMN);
      final String userGroupName =
        (String) this.sortableTable.getValueAt(selectedRow,
          ImportUsersDialog.USERGROUP_COLUMN);
      
      Date creationDate = null;
      if (useDateFromFile) {
        String dateString =
          (String) this.sortableTable.getValueAt(selectedRow,
            ImportUsersDialog.CREATION_DATE_COLUMN);

        if (dateString != null && !dateString.isEmpty()) {
          creationDate = UserManagerPanel.dateFormat.parse(dateString);
        }
      }
      else {
        creationDate = Calendar.getInstance().getTime();
      }
      
      final UserGroup userGroup =
        UserGroupProxy.getUserGroupByName(this.window, this.windowTitle,
          LNG.get("IAS_WAITING_USERGROUP"), userGroupName);
      if (userGroup != null) {
        final UserInfo userInfo =
          new UserInfo(login, name, emails.split(EMAILS_SPLIT_REGEX),
            (Object[]) this.defaultUserInfo.getAttribute(User.ROLE_IDS),
            (Object[]) this.defaultUserInfo.getAttribute(User.PERMISSION_IDS),
            creationDate);
        userInfo.setAttribute(User.PASSWORD, login);
        userInfo.setAttribute(UserGroup.USERGROUP_ID, userGroup.getId());
        usersToImport.add(userInfo);
      }
      else {
        this.discartedUserList.add(login);
      }
    }
    ImportResults result = ImportResults.ALL_IMPORTED;
    if (usersToImport.isEmpty()) {
      /*
       * todos os grupos eram invlidos. nenhum usurio vai ser importado.
       */
      return ImportResults.NONE_IMPORTED;
    }
    else if (usersToImport.size() < selectedRows.length) {
      /*
       * alguns grupos eram invlidos, portanto alguns usurios no sero
       * importados
       */
      result = ImportResults.SOME_IMPORTED;
    }
    for (final UserInfo ui : usersToImport) {
      importUser(ui);
    }
    return result;
  }

  /**
   * L os dados do arquivo e preenche a tabela.
   * 
   * @param file arquivo de entrada
   * @throws IOException se houver algum erro de I/O
   */
  private void loadDataFromFile(final ClientFile file) throws IOException {
    final InputStream inputStream = file.getInputStream();
    final Client client = Client.getInstance();
    final Charset charset = client.getSystemDefaultCharset();
    final TsvParser tsvParser = new TsvParser();

    final RemoteTask<String[][]> remoteTask = new RemoteTask<String[][]>() {
      @Override
      protected void performTask() throws Exception {
        final String[][] fields = tsvParser.parse(inputStream, charset);
        setResult(fields);
      }
    };

    final String tskMsg = LNG.get("IAS_WAITING_USER_IMPORTING");
    final boolean executed = remoteTask.execute(window, windowTitle, tskMsg);
    if (!executed) {
      return;
    }

    final String[][] fields = remoteTask.getResult();
    int linesRead = 0;
    for (final String[] field : fields) {
      
      // -1 porque a data de criao pode estar vazia em usurios antigos.
      if ((field.length != COLUMN_COUNT) && (field.length != COLUMN_COUNT-1)) {
        continue;
      }
      
      // Com data de criao do usurio
      if (field.length == COLUMN_COUNT) {
        // Testa se o valor da data de criao est no formato correto.
        if (field[4] != null && !field[4].isEmpty()) {
          try {
            UserManagerPanel.dateFormat.parse(field[4]);
          }
          catch (ParseException e) {
            continue;
          }
        }

        this.tableModel.add(new String[] { field[0], field[1], field[2],
            field[3], field[4] });
      }
      else { // sem data de criao do usurio
        this.tableModel.add(new String[] { field[0], field[1], field[2],
            field[3], null });
      }
      
      linesRead++;
    }
    
    if (linesRead < (fields.length)) {
      /*
       * nem todas as linhas foram aproveitadas
       */
      StandardDialogs.showWarningDialog(window,
        LNG.get("IAS_IMPORT_USERS_TITLE"),
        LNG.get("IAS_IMPORT_USERS_LOAD_DATA_PARTIAL"));
    }
  }

  /**
   * Faz um merge de dois arrays.
   * 
   * @param array1 array 1
   * @param array2 array 2
   * 
   * @return lista com o resultado do merge.
   */
  private List<Object> merge(final Object[] array1, final Object[] array2) {
    final List<Object> returnList = new ArrayList<Object>();
    for (final Object element : array1) {
      returnList.add(element);
    }
    for (int i = 0; i < array2.length; i++) {
      if (!returnList.contains(array2[i])) {
        returnList.add(array2[i]);
      }
    }
    return returnList;
  }

  /**
   * Exibe o dilogo.
   */
  public void showDialog() {
    createFileChooserPanel();
    createUsersPanel();
    createButtonPanel();
    this.window.pack();
    this.window.center(this.owner);
    this.window.setVisible(true);
  }

  /**
   * Exibe o sistema de arquivo local para escolha do arquivo que ser
   * importado.
   */
  private void showFileChooser() {
    File file = null;
    try {
      final JFileChooser fc = new JFileChooser();
      fc.setAcceptAllFileFilterUsed(true);
      fc.addChoosableFileFilter(new Filter("txt", LNG
        .get("ias.file_type.txt.description")));
      fc.addChoosableFileFilter(new Filter("csv", LNG
        .get("ias.file_type.csv.description")));
      final int returnVal = fc.showOpenDialog(window);
      if (returnVal == JFileChooser.APPROVE_OPTION) {
        file = fc.getSelectedFile();
      }
    }
    catch (final Exception e) {
      StandardErrorDialogs.showErrorDialog(window, windowTitle,
        LNG.get("IAS_IMPORT_USERS_LOAD_DATA_FAILED"));
      return;
    }
    if (file != null) {
      final ClientLocalFile clientLocalFile = new ClientLocalFile(file);
      updateChosenFile(clientLocalFile);
    }
  }

  /**
   * Exibe a arvore de projeto para escolha do arquivo que ser importado.
   */
  private void showProjectFileChooser() {
    ProjectFileChooserOpen chooser;
    try {
      // Se no existe arquivo CSV no sistema, usa o TEXT.
      String fileTypeExt = FILE_TYPE;
      final ClientProjectFileType projFileType =
        ClientProjectFileType.getFileType(FILE_TYPE);
      if (projFileType.getCode().equals(ProjectFileType.UNKNOWN)) {
        fileTypeExt = "TEXT";
      }
      final DesktopFrame desktop = DesktopFrame.getInstance();
      final CommonClientProject project = desktop.getProject();
      final int mode = ProjectFileChooser.FILE_ONLY;
      chooser =
        new ProjectFileChooserOpen(window, project, false, mode, fileTypeExt);
    }
    catch (final ClientException e) {
      StandardErrorDialogs.showErrorDialog(window, e);
      return;
    }
    final ProjectTreePath treePath = chooser.getSelectedPath();
    if (treePath != null) {
      final ClientProjectFile clientProjectFile = treePath.getFile();
      updateChosenFile(clientProjectFile);
    }
  }

  /**
   * Atualiza o boto de ao conforme seleo da tabela.
   */
  private void updateActionButton() {
    boolean canEnableActionButton = false;
    if (this.sortableTable.getSelectedRow() != -1) {
      canEnableActionButton = true;
    }
    this.actionButton.setEnabled(canEnableActionButton);
  }

  /**
   * Atualiza a tabela com os dados do arquivo escolhido.
   * 
   * @param clientFile - arquivo CSV com os usurios a serem importados
   */
  private void updateChosenFile(final ClientFile clientFile) {
    if (clientFile.equals(this.currentFile)) {
      return;
    }
    this.currentFile = clientFile;
    this.tableModel.clear();
    try {
      loadDataFromFile(clientFile);
    }
    catch (final Exception e) {
      StandardErrorDialogs.showErrorDialog(window, windowTitle,
        LNG.get("IAS_IMPORT_USERS_LOAD_DATA_FAILED"));
      e.printStackTrace();
      return;
    }
    this.filePath.setText(this.currentFile.getStringPath());
    updateActionButton();
  }

  /**
   * Atualiza as informaes de um usurio.
   * 
   * @param currentUser usurio que ser atualizado.
   * @param newUserInfo novas informaes do usurio.
   * 
   * @throws Exception em caso de erro na atualizao.
   */
  private void updateUser(final User currentUser, final UserInfo newUserInfo)
    throws Exception {
    List<Object> permissionList = new ArrayList<Object>();
    List<Object> roleList = new ArrayList<Object>();

    // Obtm permisses.
    permissionList =
      merge(
        (Object[]) currentUser.getUserInfo().getAttribute(User.PERMISSION_IDS),
        (Object[]) newUserInfo.getAttribute(User.PERMISSION_IDS));

    // Obtm perfis.
    roleList =
      merge((Object[]) currentUser.getUserInfo().getAttribute(User.ROLE_IDS),
        (Object[]) newUserInfo.getAttribute(User.ROLE_IDS));

    // Atualiza info.
    final UserInfo userInfo = currentUser.getUserInfo();
    userInfo.setAttribute(User.LOGIN, newUserInfo.getAttribute(User.LOGIN));
    userInfo.setAttribute(User.NAME, newUserInfo.getAttribute(User.NAME));
    userInfo.setAttribute(User.EMAILS, newUserInfo.getAttribute(User.EMAILS));
    userInfo.setAttribute(User.PERMISSION_IDS, permissionList.toArray());
    userInfo.setAttribute(User.ROLE_IDS, roleList.toArray());
    userInfo.setAttribute(UserGroup.USERGROUP_ID,
      newUserInfo.getAttribute(UserGroup.USERGROUP_ID));
    userInfo.setAttribute(User.CREATION_DATE,
      newUserInfo.getAttribute(User.CREATION_DATE));
    
    // Atualiza usurio.
    User.modifyUser(currentUser.getId(), userInfo);
  }

  /**
   * Construtor.
   * 
   * @param owner janela-me.
   */
  public ImportUsersDialog(final Window owner) {
    this.owner = owner;
    this.windowTitle = LNG.get("IAS_IMPORT_USERS_TITLE");
    this.window = new DesktopComponentDialog(owner, this.windowTitle);
    this.window.center(owner);
    this.contentPane = this.window.getContentPane();
    this.defaultUserInfo = new UserInfo();
    this.userRoleDefinitionDialog =
      new UserRoleDefinitionDialog(this.window, this.defaultUserInfo, true);
    this.userPermissionDefinitionDialog =
      new UserPermissionDefinitionDialog(this.window, this.defaultUserInfo,
        true);
  }
  
}
