package csbase.client.preferences.types;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import csbase.client.preferences.PreferenceValue;
import csbase.client.preferences.definition.PreferenceDefinition;
import csbase.client.preferences.definition.PreferencePolicy;
import csbase.client.preferences.util.PreferenceBundle;

/**
 * Preferncia que encapsula um objeto do tipo {@link Map}. Este valor deve ser
 * usado da seguinte forma:<br/>
 * 
 * \@Value(type=PVMap.class, defaultValue="chave1=valor1;chave2=valor2;")
 * 
 * Objetos desta classe so restritos a armazenar apenas mapas em que tanto as
 * chaves quanto os valores so do tipo {@link String}. As Strings, tanto chave
 * quanto valor, no podem contr os seguintes caracteres: <br/>
 * - igual (=) <br/>
 * - ponto e virgula (;) <br/>
 * 
 * @see PreferenceValue PreferenceCategory PreferencePolicy
 * 
 * @author Tecgraf
 */
public class PVMap extends PreferenceValue<Map<String, String>> {

  /**
   * Construtor usado na instanciao desta classe por reflexo.
   * 
   * @param name nome da preferncia que possui este valor.
   * @param value valor da preferncia.
   * @param defaultValue valor default da preferncia.
   * @param policy poltica de visibilidade do valor.
   * @param preferenceBundle objeto responsvel pela internacionalizao.
   */
  public PVMap(PreferenceDefinition name, String value, String defaultValue,
    PreferencePolicy policy, PreferenceBundle preferenceBundle) {
    super(name, value, defaultValue, policy, preferenceBundle);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getClassName() {
    return getClass().getName();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder();
    for (Entry<String, String> entry : getValue().entrySet()) {
      builder.append(entry.getKey());
      builder.append("=");
      builder.append(entry.getValue());
      builder.append(";");
    }
    return builder.toString();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, String> toValue(String value) {
    Pattern p = Pattern.compile("(.*?)=(.*?);");

    Map<String, String> result = new HashMap<String, String>();

    Matcher m = p.matcher(value);
    while (m.find()) {
      result.put(m.group(1), m.group(2));
    }

    return result;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public PreferenceValue<Map<String, String>> clone() {
    PVMap clone =
      new PVMap(name, this.toString(), defaultValue, policy, preferenceBundle);

    clone.setPreferenceEditorClass(this.getPreferenceEditorClass());
    return clone;
  }

  /**
   * Adiciona mapeamento de uma chave para um valor.
   * 
   * @param key - chave.
   * @param value - valor.
   */
  public void put(String key, String value) {
    getValue().put(key, value);
  }

  /**
   * Retorna true se existe valor associado a chave. False caso contrrio.
   * 
   * @param key - chave.
   * @return true se existe valor associado a chave. False caso contrrio.
   */
  public boolean containsKey(String key) {
    return getValue().containsKey(key);
  }

  /**
   * Retrona o valor associado a uma chave.
   * 
   * @param key - chave.
   * @return valor associado.
   */
  public String get(String key) {
    return getValue().get(key);
  }
}
