package csbase.client.applications.algorithmsmanager.dialogs;

import java.awt.GridBagLayout;
import java.io.File;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Vector;

import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.SortOrder;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableModel;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.table.SortableTable;
import csbase.client.algorithms.ShowHistoryAction;
import csbase.client.applications.algorithmsmanager.models.AlgorithmListItem;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy;
import csbase.client.util.DateTableCellRenderer;
import csbase.logic.algorithms.HistoryProvider;
import csbase.logic.algorithms.HistoryRecord;

/**
 * Painel com as informaes de histrico do algoritmo selecionado para edio.
 * 
 */
public class AlgorithmHistoryInfoPanel extends CommonInfoEditPanel {
  /** Painel com as informaes estticas do algoritmo */
  private AlgorithmStaticInfoPanel algorithmHeaderPanel;

  /** Objeto provedor do histrico a ser exibido */
  private HistoryProvider historyProvider;

  /** Painel principal */
  private JPanel mainPanel;

  /** Painel com o histrico do algoritmo selecionado */
  private JPanel historyPanel;

  /**
   * Constri o painel com as informaes de categorias do algoritmo.
   * 
   * @param algoPane painel com os dados de atualizao de um algoritmo
   */
  public AlgorithmHistoryInfoPanel(AlgorithmEditDataPanel algoPane) {
    super(algoPane);
    showOperationButtons(false);
  }

  /**
   * Obtm o painel global de edio de dados de algoritmo.
   * 
   * @return o painel global de edio de dados de algoritmo
   */
  private AlgorithmEditDataPanel getAlgorithmEditDataPanel() {
    return (AlgorithmEditDataPanel) getEditPanel();
  }

  /**
   * Obtm o algoritmo selecionado para edio.
   * 
   * @return o algoritmo selecionado para edio
   */
  private AlgorithmListItem getSelectedAlgorithm() {
    return getAlgorithmEditDataPanel().getSelectedAlgorithm();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initializeData() {
    if (getSelectedAlgorithm() == null) {
      return;
    }
    algorithmHeaderPanel.setSelectedAlgorithm(getSelectedAlgorithm());
    updateHistoryPanel();
  }

  /**
   * Altera o painel com a rvore da verso para o algoritmo selecionado.
   * 
   */
  private void updateHistoryPanel() {
    mainPanel.remove(historyPanel);
    historyPanel = getHistoryPanel();
    mainPanel.add(historyPanel, new GBC(0, 1).both().west().insets(0, 0, 0, 0));
    mainPanel.revalidate();
    mainPanel.repaint();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JPanel buildMainInfoPanel() {
    if (mainPanel == null) {
      mainPanel = new JPanel(new GridBagLayout());
      algorithmHeaderPanel = new AlgorithmStaticInfoPanel(getApplication());
      historyPanel = getHistoryPanel();

      mainPanel.add(algorithmHeaderPanel, new GBC(0, 0).horizontal().west()
        .insets(0, 0, 0, 0));
      mainPanel.add(historyPanel, new GBC(0, 1).both().west()
        .insets(0, 0, 0, 0));
    }
    return mainPanel;
  }

  /**
   * Obtm o painel com o histrico do algoritmo.
   * 
   * @return o painel com o histrico do algoritmo
   */
  private JPanel getHistoryPanel() {
    historyPanel = new JPanel(new GridBagLayout());
    historyProvider = new AlgorithmHistory();
    List<?> rows = historyProvider.getHistory();
    if (rows == null) {
      String message =
        getString("AlgorithmHistoryInfoPanel.msg.error.load.history");
      getAlgorithmEditDataPanel().getApplication().showError(message);
      rows = Collections.emptyList();
    }
    historyPanel.add(new JScrollPane(makeTable(rows)), new GBC(0, 0).both()
      .west().insets(5, 5, 5, 5));
    return historyPanel;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void apply() {
    // No faz nada.
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void cancel() {
    // No faz nada.
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setDataChanged() {
    // No faz nada.
  }

  /**
   * Constri a tabela que exibe o histrico.
   * 
   * @param history o histrico.
   * 
   * @return tabela que exibe o histrico.
   */
  private JTable makeTable(List<?> history) {
    TableModel m = new HistoryTableModel(history);

    SortableTable table = new SortableTable(m);
    table.sort(0, SortOrder.DESCENDING);
    table.setDefaultRenderer(Date.class, new DateTableCellRenderer());
    table.adjustColumnWidth();
    return table;
  }

  /**
   * Modelo para a tabela
   */
  private class HistoryTableModel extends AbstractTableModel {
    /** Lista de <code>HistoryRecord</code> */
    private List<?> recordList;

    /** Cabealhos para as colunas */
    private String[] columnNames = {
        LNG.get(ShowHistoryAction.class.getName() + ".dateColumn"),
        LNG.get(ShowHistoryAction.class.getName() + ".pathColumn"),
        LNG.get(ShowHistoryAction.class.getName() + ".loginColumn"),
        LNG.get(ShowHistoryAction.class.getName() + ".nameColumn"),
        LNG.get(ShowHistoryAction.class.getName() + ".descriptionColumn") };

    /** Classes das colunas */
    private Class<?>[] columnClasses = { Date.class, String.class,
        String.class, String.class, String.class };

    /**
     * Cria o modelo.
     * 
     * @param recordList Lista de <code>HistoryRecord</code>.
     */
    public HistoryTableModel(List<?> recordList) {
      this.recordList = recordList;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int getRowCount() {
      return this.recordList.size();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public int getColumnCount() {
      return this.columnNames.length;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Object getValueAt(int rowIndex, int columnIndex) {
      HistoryRecord rec = (HistoryRecord) this.recordList.get(rowIndex);
      switch (columnIndex) {
        case HistoryRecord.DATE_COLUMN_INDEX:
          return rec.getDate();
        case HistoryRecord.PATH_COLUMN_INDEX:
          return FileUtils.joinPath(rec.getDirPath());
        case HistoryRecord.LOGIN_COLUMN_INDEX:
          return rec.getUserLogin();
        case HistoryRecord.NAME_COLUMN_INDEX:
          return rec.getUserName();
        case HistoryRecord.DESCRIPTION_COLUMN_INDEX:
          return rec.getDescription();
        default:
          return null;
      }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getColumnName(int columnIndex) {
      return this.columnNames[columnIndex];
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Class<?> getColumnClass(int columnIndex) {
      return this.columnClasses[columnIndex];
    }
  }

  /**
   * Classe que implementa o provedor de histrico do algoritmo selecionado.
   * 
   */
  private class AlgorithmHistory implements HistoryProvider {

    /**
     * {@inheritDoc}
     */
    @Override
    public List<?> getHistory() {
      if (getSelectedAlgorithm() == null) {
        return new Vector<HistoryRecord>();
      }
      String repositoryName = LNG.get("algomanager.tree.root");
      String[] spath =
        new String[] { repositoryName, getSelectedAlgorithm().getName() };
      return AlgorithmManagementProxy.retrieveHistory(spath);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getName() {
      if (getSelectedAlgorithm() == null) {
        return "";
      }
      String repositoryName =
        getString("AlgorithmHistoryInfoPanel.algorithm.repository.name");
      return repositoryName + File.separator + getSelectedAlgorithm().getName();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getTitle() {
    return getString("AlgorithmHistoryInfoPanel.tab.history");
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean wasModified() {
    return false;
  }
}
