/**
 * $Id: ProjectTreeRenderer.java 150777 2014-03-19 14:16:56Z oikawa $
 */

package csbase.client.project;

import java.awt.Color;
import java.awt.Component;
import java.awt.Font;
import java.awt.GridBagLayout;

import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTree;
import javax.swing.tree.DefaultTreeCellRenderer;

import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.configurationmanager.ConfigurationManagerException;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import csbase.client.applications.ApplicationImages;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectFileType;

/**
 * Responsvel por desenhar os ns da rvore, mudando os cones e os tooltips de
 * acordo com o tipo do n.
 */
class ProjectTreeRenderer extends DefaultTreeCellRenderer {

  /**
   * Tooltip para projetos bloqueados.
   */
  private static final String PRJ_LOCKED_TOOLTIP = LNG
    .get("ProjectTree.locked");
  /**
   * Tooltip para projetos desatualizados em relao ao servidor.
   */
  private static final String PRJ_OUTOFDATE_TOOLTIP = LNG
    .get("ProjectTree.outOfDate");
  /**
   * Valor da propriedade "draw.project.sharing", ou <code>false</code> caso
   * esta no tenha sido definida.
   */
  private static final boolean DRAW_PROJECT_SHARING;

  /*
   * cdigo esttico para inicializao da constante DRAW_PROJECT_SHARING
   */
  static {
    boolean drawSharing;
    try {
      Configuration cfg =
        ConfigurationManager.getInstance().getConfiguration(
          ProjectTreeRenderer.class);
      drawSharing =
        cfg.getOptionalBooleanProperty("draw.project.sharing", false);
    }
    catch (ConfigurationManagerException e) {
      drawSharing = false;
    }
    DRAW_PROJECT_SHARING = drawSharing;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Component getTreeCellRendererComponent(JTree tree, Object value,
    boolean sel, boolean expanded, boolean leaf, int row, boolean isFocused) {
    final JLabel component =
      (JLabel) super.getTreeCellRendererComponent(tree, value, sel, expanded,
        leaf, row, isFocused);
    final ProjectTreeNode projectTreeNode = (ProjectTreeNode) value;
    final ProjectTreeModel model = (ProjectTreeModel) tree.getModel();
    final ClientProjectFile file = projectTreeNode.getClientProjectFile();
    if (value == model.getRoot() && DRAW_PROJECT_SHARING) {
      return buildProjectComponent(component, model, file);
    }

    if (file.isDirectory()) {
      if (file.isUpdated()) {
        component.setToolTipText(null);
      }
      else {
        component.setToolTipText(PRJ_OUTOFDATE_TOOLTIP);
      }
    }
    else {
      try {
        ProjectFileType type = ProjectFileType.getFileType(file.getType());
        component.setToolTipText(type.getDescription());
      }
      catch (Exception e) {
        // ??  pra no fazer nada mesmo?
      }
      if (file.isLocked()) {
        component.setForeground(Color.RED);
        component.setToolTipText(PRJ_LOCKED_TOOLTIP);
      }
    }
    component.setIcon(ClientProjectFileProperties.getImageIcon(file));
    return component;
  }

  /**
   * Montagem de componente especial para n de projeto.
   * 
   * @param component o componente original do label
   * @param model o modelo da rvore
   * @param file o arquivo/diretrio que representa a raiz do projeto.
   * @return o componente a ser colocado na rvore.
   */
  private Component buildProjectComponent(final JLabel component,
    final ProjectTreeModel model, final ClientProjectFile file) {
    final CommonClientProject project = model.getProject();
    if (project == null) {
      return component;
    }
    final JLabel sharedLabel = new JLabel();
    if (project.isPrivate()) {
      sharedLabel.setIcon(ApplicationImages.ICON_PRIVATE_16);
    }
    else if (project.isPublic()) {
      sharedLabel.setIcon(ApplicationImages.ICON_PUBLIC_16);
    }
    else {
      sharedLabel.setIcon(ApplicationImages.ICON_SHARED_16);
    }

    final Font fnt = component.getFont();
    final JPanel panel = new JPanel(new GridBagLayout());
    panel.setOpaque(false);
    component.setIcon(ClientProjectFileProperties.getImageIcon(file));

    final JLabel lb1 = new JLabel("(");
    final JLabel lb2 = new JLabel(")");

    final Color fg = component.getForeground();
    final Color bg = component.getBackground();
    final JLabel labels[] = new JLabel[] { lb1, lb2, sharedLabel };
    for (JLabel label : labels) {
      label.setFont(fnt);
      label.setForeground(fg);
      label.setBackground(bg);
    }

    panel.add(component, new GBC(0, 0).insets(0, 0, 0, 0));
    panel.add(lb1, new GBC(1, 0).insets(0, 0, 0, 0));
    panel.add(sharedLabel, new GBC(2, 0).insets(0, 0, 0, 0));
    panel.add(lb2, new GBC(3, 0).insets(0, 0, 0, 0));
    return panel;
  }

}
