package csbase.client.algorithms.parameters;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import csbase.client.ClientLocalFile;
import csbase.client.desktop.RemoteTask;
import csbase.client.desktop.Task;
import csbase.client.util.StandardErrorDialogs;
import csbase.client.util.filechooser.ClientFileChooser;
import csbase.client.util.filechooser.ClientFileChooserSelectionMode;
import csbase.client.util.filechooser.ClientLocalFileChooserUtil;
import csbase.client.util.filechooser.ClientProjectFileChooserUtil;
import csbase.client.util.filechooser.ClientProjectFileChooserUtil.OperationResult;
import csbase.client.util.filechooser.ClientSGAFileChooserUtil;
import csbase.client.util.filechooser.util.ClientFileChooserUtils;
import csbase.logic.ClientFile;
import csbase.logic.ClientSGAFile;
import csbase.logic.SGASet;
import csbase.logic.algorithms.parameters.FileParameterMode;
import csbase.logic.algorithms.parameters.InputURLListParameter;
import csbase.logic.algorithms.parameters.URLProtocol;
import csbase.logic.algorithms.parameters.FileURLValue;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.SGAServiceInterface;
import tecgraf.javautils.core.lng.LNG;

/**
 * Viso do parmetro lista de URLs de entrada {@link InputURLListParameter}.
 * 
 * @author Tecgraf
 */
public class InputURLListParameterView extends URLListParameterView {

  /**
   * Construtor.
   * 
   * @param parameter parmetro (No aceita {@code null}).
   * @param mode modo de visualizao. No aceita {@code null}.
   */
  public InputURLListParameterView(InputURLListParameter parameter, Mode mode) {
    super(parameter, mode);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public InputURLListParameter getParameter() {
    return (InputURLListParameter) super.getParameter();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected List<FileURLValue> askForFile(URLProtocol fileProtocol, Object... args) {
    String fileType = getParameter().getFileType();
    ClientFileChooserSelectionMode selectionMode = getFileChooserMode();
    String title = getParameter().getLabel();
    switch (fileProtocol) {
      case PROJECT:
        return askForProjectFile(fileType, selectionMode, title);
      case LOCAL:
        return askForLocalFile(fileType, selectionMode, title);
      case SGA:
        String sgaName = (String) args[0];
        return askForSGAFile(fileType, selectionMode, title, sgaName);
    }
    return null;
  }

  /**
   * Abre um navegador de arquivos do projeto.
   * 
   * @param fileType os tipos de arquivo permitido.
   * @param selectionMode o modo de seleo
   *        {@link ClientFileChooserSelectionMode}.
   * @param title o ttulo para a janela do navegador.
   * 
   * @return a URL do arquivo selecionado.
   */
  private List<FileURLValue> askForProjectFile(String fileType,
    ClientFileChooserSelectionMode selectionMode, String title) {
    List<FileURLValue> urls = new ArrayList<FileURLValue>();
    List<OperationResult> selectedFiles =
      ClientProjectFileChooserUtil.browseMultipleInOpenMode(
        getWindow(), title, selectionMode, true, fileType);
    if (selectedFiles == null) {
      return null;
    }
    for (OperationResult selectedFile : selectedFiles) {
      ClientFile file = selectedFile.getClientProjectFile();
      urls.add(getURLFromFile(URLProtocol.PROJECT, file, null));
    }
    return urls;
  }

  /**
   * Abre um navegador de arquivos do cliente.
   * 
   * @param fileType o tipo de arquivo permitido.
   * @param selectionMode o modo de seleo
   *        {@link ClientFileChooserSelectionMode}.
   * @param title o ttulo para a janela do navegador.
   * 
   * @return a URL do arquivo selecionado.
   */
  private List<FileURLValue> askForLocalFile(String fileType,
    ClientFileChooserSelectionMode selectionMode, String title) {
    List<FileURLValue> urls = new ArrayList<FileURLValue>();
    List<String> fileTypes = new LinkedList<String>();
    if (fileType != null) {
      fileTypes.add(fileType);
    }
    switch (selectionMode) {
      case DIRECTORIES_ONLY:
        List<ClientLocalFile> selectedDirs =
          ClientLocalFileChooserUtil.browseMultipleDirectoriesInOpenMode(
            getWindow(), fileTypes, title, true, ClientFileChooserUtils
              .getLocalHomeDirectory());
        if (selectedDirs == null) {
          return null;
        }
        for (ClientLocalFile dir : selectedDirs) {
          urls.add(getURLFromFile(URLProtocol.LOCAL, dir, null));
        }
        break;
      case FILES_ONLY:
        List<ClientLocalFile> selectedFiles =
          ClientLocalFileChooserUtil.browseMultipleFilesInOpenMode(getWindow(),
            fileTypes, title, true, ClientFileChooserUtils
              .getLocalHomeDirectory());
        if (selectedFiles == null) {
          return null;
        }
        for (ClientLocalFile file : selectedFiles) {
          urls.add(getURLFromFile(URLProtocol.LOCAL, file, null));
        }
        break;
      default:
        return null;
    }
    return urls;
  }

  /**
   * Abre um navegador de arquivos do SGA.
   * 
   * @param fileType o tipo de arquivo permitido.
   * @param selectionMode o modo de seleo
   *        {@link ClientFileChooserSelectionMode}.
   * @param title o ttulo para a janela do navegador.
   * @param sgaName nome do SGA.
   * 
   * @return a URL do arquivo selecionado.
   */
  private List<FileURLValue> askForSGAFile(String fileType,
    ClientFileChooserSelectionMode selectionMode,
    String title, final String sgaName) {
    if (sgaName == null) {
      return null;
    }

    final SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
    Task<Boolean> sgaIsAliveTask = new RemoteTask<Boolean>() {
      @Override
      protected void performTask() throws Exception {
        SGASet sga = sgaService.getSGASet(sgaName);
        setResult(sga.getAlive());
      }
    };
    boolean isAlive = false;
    String description =
      LNG.get(URLParameterView.class.getSimpleName() + ".sga_alive_task");
    if (sgaIsAliveTask.execute(getWindow(), title, description)) {
      isAlive = sgaIsAliveTask.getResult();
    }

    if (!isAlive) {
      String errorMessage =
        LNG.get(URLParameterView.class.getSimpleName() + ".sga_not_alive",
          new Object[] { sgaName });
      StandardErrorDialogs.showErrorDialog(getWindow(), errorMessage);
      return null;
    }
    List<String> fileTypes = new LinkedList<String>();
    if (fileType != null) {
      fileTypes.add(fileType);
    }
    ClientSGAFile root = new ClientSGAFile(sgaName, "/");
    List<FileURLValue> urls = new ArrayList<FileURLValue>();
    switch (selectionMode) {
      case DIRECTORIES_ONLY:
        List<ClientSGAFile> selectedDirs =
          ClientSGAFileChooserUtil.browseMultipleDirectoriesInOpenMode(
            getWindow(), fileTypes, title, true, root);
        if (selectedDirs == null) {
          return null;
        }
        for (ClientSGAFile dir : selectedDirs) {
          urls.add(getURLFromFile(URLProtocol.SGA, dir, sgaName));
        }
        break;
      case FILES_ONLY:
        List<ClientSGAFile> selectedFiles =
          ClientSGAFileChooserUtil.browseMultipleFilesInOpenMode(getWindow(),
            fileTypes, title, true, root);
        if (selectedFiles == null) {
          return null;
        }
        for (ClientSGAFile file : selectedFiles) {
          urls.add(getURLFromFile(URLProtocol.SGA, file, sgaName));
        }
        break;
      default:
        return null;
    }
    return urls;
  }

  /**
   * Obtm a URL a partir de um arquivo.
   * 
   * @param fileProtocol o protocolo de acesso ao arquivo.
   * @param file o arquivo.
   * @param host o host da URL.
   * @return URL.
   */
  private FileURLValue getURLFromFile(URLProtocol fileProtocol, ClientFile file,
    String host) {
    if (file == null) {
      return null;
    }
    String type = file.getType();
    return new FileURLValue(file.getStringPath(), type, fileProtocol, host);
  }

  /**
   * Obtm o modo do {@link ClientFileChooser} que ser utilizado para solicitar
   * os arquivos ao usurio.
   * 
   * @return modo arquivo ou diretrio.
   */
  private ClientFileChooserSelectionMode getFileChooserMode() {
    FileParameterMode parameterMode = getParameter().getMode();
    switch (parameterMode) {
      case REGULAR_FILE:
        return ClientFileChooserSelectionMode.FILES_ONLY;
      case DIRECTORY:
        return ClientFileChooserSelectionMode.DIRECTORIES_ONLY;
      default:
        String errorMessage =
          MessageFormat.format("O modo {0} no  vlido.", parameterMode);
        throw new IllegalStateException(errorMessage);
    }
  }

}
