package csbase.client.desktop;

import java.awt.FlowLayout;
import java.awt.GridBagLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.SortOrder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.SwingThreadDispatcher;
import tecgraf.javautils.gui.table.AbstractColumn;
import tecgraf.javautils.gui.table.ColumnsObjectTableModel;
import tecgraf.javautils.gui.table.ObjectTableBuilder;
import tecgraf.javautils.gui.table.ObjectTableBuilder.ColumnsWidthPolicy;
import tecgraf.javautils.gui.table.ObjectTableBuilder.SelectionMode;
import tecgraf.javautils.gui.table.SortableTable;
import csbase.client.ClientServerManager;
import csbase.client.project.action.ProjectCloseAction;
import csbase.client.remote.ClientRemoteMonitor;
import csbase.client.remote.manager.server.ServerInfoManager;
import csbase.client.remote.manager.server.ServerInfoManagerListener;
import csbase.client.util.StandardErrorDialogs;
import csbase.client.util.gui.ComponentUtilities;
import csbase.exception.CSBaseException;
import csbase.logic.LoginInfo;
import csbase.logic.ServerURI;
import csbase.logic.server.ServerInfo;

/**
 * Dilogo que permite ao usurio trocar o servidor CSBase ao qual est
 * conectado.<br>
 * Obs. 1: Ao trocar de servidor, o usurio ser desconectado do servidor
 * antigo. Obs. 2: Se existir um projeto aberto, este ser fechado.
 * 
 * @author Tecgraf
 */
public class ProjectServerDialog extends DesktopComponentDialog {

  /**
   * Tabela que contm os servidores.
   */
  private SortableTable table;
  /**
   * Modelo da tabela de servidores;
   */
  private ColumnsObjectTableModel<ServerInfo> tableModel;

  /**
   * Listener que escuta mudanas nas informaes dos servidores.
   */
  private ServerInfoManagerListener managerListener;

  /**
   * Construtor.
   * 
   * @param owner Janela na qual esse dilogo se originou.
   */
  private ProjectServerDialog(Window owner) {
    super(owner, LNG.get("ProjectServerDialog.title"));
    setDefaultCloseOperation(JDialog.DISPOSE_ON_CLOSE);
    getContentPane().add(buildGUI());

    initializeServersListener();
  }

  /**
   * Cria e apresenta um dilogo permitindo que o usurio troque de servidor.
   * 
   * @param owner Janela que requisitou o dilogo.
   */
  public static void show(Window owner) {
    ProjectServerDialog dialog = new ProjectServerDialog(owner);
    dialog.pack();
    dialog.setVisible(true);
  }

  /**
   * Constroi a interface grfica do dilogo.
   * 
   * @return um painel representando a interface grfica do dilogo.
   */
  @SuppressWarnings("unchecked")
  public JPanel buildGUI() {
    JButton closeButton =
      new JButton(new AbstractAction(
        LNG.get("ProjectServerDialog.action.close.name")) {
        @Override
        public void actionPerformed(ActionEvent e) {
          ProjectServerDialog.this.close();
        }
      });
    final AbstractAction loginAction = new LoginAction();
    loginAction.setEnabled(false);
    JButton loginButton = new JButton(loginAction);

    ObjectTableBuilder<ServerInfo> builder =
      new ObjectTableBuilder<ServerInfo>(new ConnectedColumn(),
        new SuspendedColumn(), new NameColumn(), new URIColumn());
    builder.setColumnsWidthPolicy(ColumnsWidthPolicy.ADJUST_BY_DATA);
    builder.setSelectionMode(SelectionMode.SINGLE_SELECTION);
    builder.setSortOrder(2, SortOrder.DESCENDING);
    List<ServerInfo> servers = getServers();
    this.table = builder.build(servers);
    this.table.getSelectionModel().addListSelectionListener(
      new ListSelectionListener() {
        @Override
        public void valueChanged(ListSelectionEvent e) {
          ServerInfo selected = getSelectedServer();
          if (selected == null) {
            loginAction.setEnabled(false);
          }
          ClientServerManager serverManager = ClientServerManager.getInstance();
          ServerURI currentServer = serverManager.getDefaultURI();
          loginAction.setEnabled(selected != null
            && !currentServer.equals(selected.getURI())
            && !selected.isSuspended());
        }
      });
    this.tableModel = (ColumnsObjectTableModel<ServerInfo>) table.getModel();

    ComponentUtilities.setMaxPreferredSize(loginButton, closeButton);
    JPanel btnPanel = new JPanel(new FlowLayout(FlowLayout.RIGHT));
    btnPanel.add(loginButton);
    btnPanel.add(closeButton);

    JPanel panel = new JPanel(new GridBagLayout());
    GBC gbc = new GBC(0, 0);
    gbc.center().fillxy().insets(12, 12, 11, 11);
    panel.add(new JScrollPane(table), gbc);
    gbc.gridy(1).east().horizontal().insets(0, 12, 11, 11);
    panel.add(btnPanel, gbc);
    return panel;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void close() {
    if (ServerInfoManager.getInstance().isEnabled()) {
      ServerInfoManager.getInstance().removeServerInfoManagerListener(
        this.managerListener);
    }
    super.close();
  }

  /**
   * Cria e inicializa o observador que verifica mudanas nas informaes dos
   * servidores.
   */
  private void initializeServersListener() {
    this.managerListener = new ServerInfoManagerListener() {
      @Override
      public void wasAddedServerInfo(final ServerInfo addedLocalServer) {
        SwingThreadDispatcher.invokeLater(new Runnable() {
          @Override
          public void run() {
            tableModel.add(addedLocalServer);
          }
        });
      }

      @Override
      public void wasRemovedServerInfo(final ServerInfo removedLocalServer) {
        SwingThreadDispatcher.invokeLater(new Runnable() {
          @Override
          public void run() {
            tableModel.remove(removedLocalServer);
          }
        });
      }

      @Override
      public void wasModifiedServerInfo(final ServerInfo oldLocalServer,
        final ServerInfo newLocalServer) {
        SwingThreadDispatcher.invokeLater(new Runnable() {
          @Override
          public void run() {
            tableModel.remove(oldLocalServer);
            tableModel.add(newLocalServer);
          }
        });
      }
    };
    if (ServerInfoManager.getInstance().isEnabled()) {
      ServerInfoManager.getInstance().addServerInfoManagerListener(
        this.managerListener);
    }
  }

  /**
   * Obtm informaes sobre os servidores disponveis.
   * 
   * @return uma lista com informaes sobre cada servidor disponvel.
   */
  private List<ServerInfo> getServers() {
    List<ServerInfo> servers = new ArrayList<ServerInfo>();
    RemoteTask<Set<ServerInfo>> task = new RemoteTask<Set<ServerInfo>>() {
      @Override
      protected void performTask() throws Exception {
        ServerInfoManager manager = ServerInfoManager.getInstance();
        setResult(manager.getServersInfos());
      }
    };
    if (task.execute(ProjectServerDialog.this,
      ProjectServerDialog.this.getTitle(),
      LNG.get("ProjectServerDialog.task.getservers.text"))) {
      servers.addAll(task.getResult());
    }

    return servers;
  }

  /**
   * Obtm o {@link ServerInfo servidor} atualmente selecionado na tabela.
   * 
   * @return o {@link ServerInfo servidor} atualmente selecionado na tabela, ou
   *         null} caso nenhum esteja selecionado.
   */
  private ServerInfo getSelectedServer() {
    int selectedRow = table.getSelectedRow();
    if (selectedRow == -1) {
      return null;
    }
    int rowIndex = table.convertRowIndexToModel(selectedRow);
    return this.tableModel.getRow(rowIndex);
  }

  /**
   * Ao que faz login no servidor selecionado na tabela.
   * 
   * @author Tecgraf
   */
  class LoginAction extends AbstractAction {

    /**
     * Construtor.
     */
    public LoginAction() {
      super(LNG.get("ProjectServerDialog.action.login.name"));
    }

    /**
     * <ol>
     * <li>Se existir algum projeto aberto, fecha ele.</li>
     * <li>Se conecta e monitora o servidor selecionado na tabela.</li>
     * <li>Substitui o servidor corrente - ou antigo servidor - pelo novo
     * servidor como o servidor padro para a obteno de servios.</li>
     * <li>Se desloga do antigo servidor.</li>
     * </ol>
     * {@inheritDoc}
     */
    @Override
    public void actionPerformed(final ActionEvent e) {
      final ServerInfo selected = getSelectedServer();
      final ClientServerManager serverManager =
        ClientServerManager.getInstance();
      final ServerURI oldServer = serverManager.getDefaultURI();
      if (oldServer.equals(selected.getURI())) {
        return;
      }

      // Se conecta no novo servidor.
      if (!login(selected)) {
        return;
      }
      // Fecha o projeto corrente, caso exista.
      if (!new ProjectCloseAction(DesktopFrame.getInstance().getTree()).close()) {
        return;
      }
      /*
       * Substitui o servidor corrente pelo novo servidor conectado e desconecta
       * do antigo servidor.
       */
      if (changeServer(selected)) {
        List<ServerInfo> servers = getServers();
        tableModel.setRows(servers);
      }
    }
  }

  /**
   * Se conecta a um servidor que esteja vivo e comea a sua monitorao.
   * 
   * @param server O servidor ao qual se deseja conectar.
   * 
   * @return <tt>true</tt> se o servidor dado est sendo monitorado e est vivo.
   */
  private boolean login(final ServerInfo server) {
    final ClientServerManager serverManager = ClientServerManager.getInstance();
    RemoteTask<Integer> loginTask = new RemoteTask<Integer>() {
      @Override
      protected void performTask() throws Exception {
        if (!serverManager.isMonitored(server.getURI())) {
          LoginInfo loginInfo =
            ClientRemoteMonitor.getInstance().getLoginInfo();
          if (!serverManager.loginWithUserPassword(server.getURI(), loginInfo)) {
            setResult(1);
            return;
          }
        }
        if (!serverManager.isAlive(server.getURI())) {
          // Se o servidor no estiver vivo, desconecta dele.
          serverManager.logout(server.getURI(), true);
          setResult(2);
        }
        else {
          setResult(0);
        }
      }

      @Override
      protected void handleError(Exception error) {
        if (CSBaseException.class.isAssignableFrom(error.getClass())) {
          StandardErrorDialogs.showErrorDialog(ProjectServerDialog.this,
            ProjectServerDialog.this.getTitle(), LNG.get(
              "ProjectServerDialog.task.login.error.3",
              new Object[] { server.getName() }));
        }
        else {
          super.handleError(error);
        }
      }
    };
    if (loginTask.execute(
      ProjectServerDialog.this,
      ProjectServerDialog.this.getTitle(),
      LNG.get("ProjectServerDialog.task.login.text",
        new Object[] { server.getName() }))) {

      Integer responseCode = loginTask.getResult();
      if (responseCode == 0) {
        return true;
      }
      StandardErrorDialogs.showErrorDialog(ProjectServerDialog.this,
        ProjectServerDialog.this.getTitle(), LNG.get(
          "ProjectServerDialog.task.login.error." + responseCode,
          new Object[] { server.getName() }));
      return false;

    }
    return false;
  }

  /**
   * Troca o servidor padro e interrompe a monitorao do antigo servidor.
   * 
   * @param connectedServer o novo servidor padro. Este servidor deve estar
   *        sendo monitorado.
   * 
   * @return <tt>true</tt> se foi possvel fazer a troca do servidor padro e se
   *         a monitorao do antigo servidor foi interrompida com sucesso.
   */
  private boolean changeServer(final ServerInfo connectedServer) {
    final ClientServerManager serverManager = ClientServerManager.getInstance();
    final ServerURI oldServerURI = serverManager.getDefaultURI();

    RemoteTask<Void> login = new RemoteTask<Void>() {
      @Override
      protected void performTask() {
        serverManager.setDefaultServer(connectedServer.getURI());
        serverManager.logout(oldServerURI, true);
      }
    };
    return login.execute(ProjectServerDialog.this,
      ProjectServerDialog.this.getTitle(),
      LNG.get("ProjectServerDialog.task.changeserver.text"));
  }

  /**
   * Modelo de coluna da tabela que indica se o servidor est suspenso.
   * 
   * @author Tecgraf
   */
  class SuspendedColumn extends AbstractColumn<ServerInfo> {

    /**
     * Construtor.
     */
    protected SuspendedColumn() {
      super(Boolean.class);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getColumnName() {
      return LNG.get("ProjectServerDialog.column.suspended.name");
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Object getValue(ServerInfo obj) {
      return obj.isSuspended();
    }
  }

  /**
   * Modelo de coluna da tabela que indica em que servidor voc est conectado.
   * 
   * @author Tecgraf
   */
  class ConnectedColumn extends AbstractColumn<ServerInfo> {

    /**
     * Construtor.
     */
    protected ConnectedColumn() {
      super(Boolean.class);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getColumnName() {
      return LNG.get("ProjectServerDialog.column.connected.name");
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Object getValue(ServerInfo obj) {
      ClientServerManager serverManager = ClientServerManager.getInstance();
      ServerURI currentServerURI = serverManager.getDefaultURI();

      return currentServerURI.equals(obj.getURI());
    }
  }

  /**
   * Modelo de coluna da tabela que indica o nome do servidor.
   * 
   * @author Tecgraf
   */
  class NameColumn extends AbstractColumn<ServerInfo> {

    /**
     * Construtor.
     */
    protected NameColumn() {
      super(String.class);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getColumnName() {
      return LNG.get("ProjectServerDialog.column.name.name");
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Object getValue(ServerInfo obj) {
      return obj.getName();
    }
  }

  /**
   * Modelo de coluna da tabela que indica a uri do servidor.
   * 
   * @author Tecgraf
   */
  class URIColumn extends AbstractColumn<ServerInfo> {

    /**
     * Construtor.
     */
    protected URIColumn() {
      super(String.class);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String getColumnName() {
      return LNG.get("ProjectServerDialog.column.uri.name");
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Object getValue(ServerInfo obj) {
      return obj.getURI().toString();
    }
  }
}
