/*
 * $Id: CreateFileTask.java 169707 2015-11-12 10:52:44Z lmoreira $
 */
package csbase.client.project.tasks;

import java.awt.Window;
import java.rmi.RemoteException;

import csbase.exception.CSBaseRuntimeException;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;

/**
 * A classe <code>CreateFileTask</code> implementa uma tarefa que cria um novo
 * arquivo em um projeto. O tratamento de RemoteException  delegado  super-
 * classe. O resultado da execuo pode ser verificado atravs do mtodo
 * {@link #getErrorCode()}.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class CreateFileTask extends CoreTask<ClientProjectFile> {

  /**
   * Cdigos de erro.
   */
  public static enum ErrorCode {
    /**
     * Operao bem sucedida
     */
    NO_ERROR,
    /**
     * Taks no executada
     */
    NOT_EXECUTED,
    /**
     * Falha de RMI
     */
    RMI_FAILURE,

    /**
     * Estado desconhecido.
     */
    UNKNOWN,
  }

  /**
   * Projeto da forma como ele  visto pelo cliente.
   */
  private CommonClientProject clientProject;
  /**
   * Diretrio onde ser criado o arquivo.
   */
  private ClientProjectFile parent;

  /**
   * Nome do novo arquivo.
   */
  private String fileName;

  /**
   * Tipo do novo arquivo.
   */
  private String fileType;

  /**
   * Cdigo de erro associado  execuo.
   */
  private ErrorCode errorCode = ErrorCode.NOT_EXECUTED;

  /**
   * Construtor.
   * 
   * @param clientProject - projeto (viso do cliente)
   * @param parent - diretrio onde ser criado o arquivo
   * @param fileName - nome do novo arquivo
   * @param fileType - tipo do novo arquivo
   */
  public CreateFileTask(CommonClientProject clientProject,
    ClientProjectFile parent, String fileName, String fileType) {
    this.clientProject = clientProject;
    this.parent = parent;
    this.fileName = fileName;
    this.fileType = fileType;
  }

  /**
   * Est provisoriamente chamando o handle de RemoteTask apenas para
   * RemoteException e CSBaseRuntimeException. Verificar se  mesmo necessrio
   * continuar fazendo outros tratamentos. {@inheritDoc}
   */
  @Override
  protected void handleError(Exception error) {
    if (error instanceof RemoteException) {
      super.handleError(error);
      errorCode = ErrorCode.RMI_FAILURE;
    }
    else if (error instanceof CSBaseRuntimeException) {
      super.handleError(error);
      errorCode = ErrorCode.UNKNOWN;
    }
    else {
      errorCode = ErrorCode.UNKNOWN;
    }
    setResult(null);
  }

  /**
   * Retorna o cdigo de erro associado  execuo.
   * 
   * @return cdigo de erro associado  execuo
   */
  public ErrorCode getErrorCode() {
    return errorCode;
  }

  /**
   * Cria o arquivo.
   */
  @Override
  protected void performTask() throws Exception {
    ClientProjectFile newFile =
      clientProject.createFile(parent, fileName, fileType);
    setResult(newFile);
    errorCode = ErrorCode.NO_ERROR;
  }

  /**
   * Mtodo de convenincia para chamada automtica da task (janela-me ser o
   * desktop)
   * 
   * @param clientProject projeto
   * @param parent diretrio de criao
   * @param fileName nome do arquivo
   * @param fileType tipo do arquivo.
   * 
   * @return o arquivo ou null
   */
  static final public ClientProjectFile runTask(
    CommonClientProject clientProject, ClientProjectFile parent,
    String fileName, String fileType) {
    final ClientProjectFile file =
      runTask(null, clientProject, parent, fileName, fileType);
    return file;
  }

  /**
   * Mtodo de convenincia para chamada automtica da task.
   * 
   * @param window janela-me par aa task.
   * @param clientProject projeto
   * @param parent diretrio de criao
   * @param fileName nome do arquivo
   * @param fileType tipo do arquivo.
   * 
   * @return o arquivo ou null
   */
  static final public ClientProjectFile runTask(Window window,
    CommonClientProject clientProject, ClientProjectFile parent,
    String fileName, String fileType) {
    final CreateFileTask task =
      new CreateFileTask(clientProject, parent, fileName, fileType);
    final Object[] args = new Object[] { fileName };
    final boolean hasResult = task.execute(window, args);
    if (!hasResult) {
      return null;
    }

    final ClientProjectFile file = task.getResult();
    return file;
  }
}
