/*
 * $Id: LightEffect.java 146656 2013-11-07 17:13:39Z mjulia $
 */
package csbase.client.applications.imageviewer.effects;

import java.awt.image.BufferedImage;
import java.awt.image.BufferedImageOp;
import java.awt.image.LookupOp;
import java.awt.image.ShortLookupTable;
import java.util.Hashtable;

import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import tecgraf.javautils.gui.GUIUtils;
import csbase.client.applications.imageviewer.ImageViewer;

/**
 * Efeitos de luz.
 * 
 * @author Tecgraf/PUC-Rio
 */
public final class LightEffect extends AbstractEffect {

  /**
   * Slider
   */
  final JSlider offsetSlider = new JSlider(JSlider.HORIZONTAL);

  /**
   * Slider
   */
  final JSlider gammaSlider = new JSlider(JSlider.HORIZONTAL);

  /**
   * Slider
   */
  final JSlider contrastSlider = new JSlider(JSlider.HORIZONTAL);

  /**
   * Gamma default
   */
  private final int DEFAULT_GAMMA = 100;

  /**
   * Offset default
   */
  private final int DEFAULT_OFFSET = 0;

  /**
   * Constrast default
   */
  private final int DEFAULT_CONTRAST = 100;

  /**
   * Construtor
   * 
   * @param application aplicao.
   */
  public LightEffect(ImageViewer application) {
    super(application);
    offsetSlider.addChangeListener(new ChangeListener() {
      @Override
      public void stateChanged(ChangeEvent e) {
        LightEffect.this.updatePreview();
      }
    });

    gammaSlider.addChangeListener(new ChangeListener() {
      @Override
      public void stateChanged(ChangeEvent e) {
        LightEffect.this.updatePreview();
      }
    });

    contrastSlider.addChangeListener(new ChangeListener() {
      @Override
      public void stateChanged(ChangeEvent e) {
        LightEffect.this.updatePreview();
      }
    });

    final int oMin = -200;
    final int oMax = 200;
    final int oStep = 50;
    offsetSlider.setMinimum(oMin);
    offsetSlider.setMaximum(oMax);
    offsetSlider.setValue(DEFAULT_OFFSET);
    offsetSlider.setMajorTickSpacing(oStep);
    offsetSlider.setPaintTicks(true);
    offsetSlider.setPaintLabels(true);
    Hashtable<Integer, JLabel> offsetTable = new Hashtable<Integer, JLabel>();
    final JLabel oMinLabel = new JLabel(oMin + "%");
    offsetTable.put(new Integer(oMin), oMinLabel);
    final JLabel oMaxLabel = new JLabel(oMax + "%");
    offsetTable.put(new Integer(oMax), oMaxLabel);
    final JLabel oDefLabel = new JLabel(DEFAULT_OFFSET + "%");
    offsetTable.put(new Integer(DEFAULT_OFFSET), oDefLabel);
    offsetSlider.setLabelTable(offsetTable);

    final int cMin = 0;
    final int cMax = 500;
    contrastSlider.setMinimum(cMin);
    contrastSlider.setMaximum(cMax);
    contrastSlider.setValue(DEFAULT_CONTRAST);
    contrastSlider.setMajorTickSpacing(DEFAULT_CONTRAST);
    contrastSlider.setPaintTicks(true);
    contrastSlider.setPaintLabels(true);

    Hashtable<Integer, JLabel> contrastTable = new Hashtable<Integer, JLabel>();
    final JLabel cMinLabel = new JLabel(cMin + "%");
    contrastTable.put(new Integer(cMin), cMinLabel);
    final JLabel cMaxLabel = new JLabel(cMax + "%");
    contrastTable.put(new Integer(cMax), cMaxLabel);
    final JLabel cDefLabel = new JLabel(DEFAULT_CONTRAST + "%");
    contrastTable.put(new Integer(DEFAULT_CONTRAST), cDefLabel);
    contrastSlider.setLabelTable(contrastTable);

    final int gMin = 0;
    final int gMax = 300;
    gammaSlider.setMinimum(gMin);
    gammaSlider.setMaximum(gMax);
    gammaSlider.setValue(DEFAULT_GAMMA);
    gammaSlider.setMajorTickSpacing(50);
    gammaSlider.setPaintTicks(true);
    gammaSlider.setPaintLabels(true);

    Hashtable<Integer, JLabel> gammaTable = new Hashtable<Integer, JLabel>();
    final JLabel gMinLabel = new JLabel(gMin + "%");
    gammaTable.put(new Integer(gMin), gMinLabel);
    final JLabel gMaxLabel = new JLabel(gMax + "%");
    gammaTable.put(new Integer(gMax), gMaxLabel);
    final JLabel gDefLabel = new JLabel(DEFAULT_GAMMA + "%");
    gammaTable.put(new Integer(DEFAULT_GAMMA), gDefLabel);
    gammaSlider.setLabelTable(gammaTable);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public JPanel getParameterPanel() {
    final JLabel offsetLabel =
      new JLabel(getString("LightEffect.offset.label"));
    final JLabel contrastLabel =
      new JLabel(getString("LightEffect.contrast.label"));
    final JLabel gammaLabel = new JLabel(getString("LightEffect.gamma.label"));
    final JComponent[] oRow = new JComponent[] { offsetLabel, offsetSlider };
    final JComponent[] cRow =
      new JComponent[] { contrastLabel, contrastSlider };
    final JComponent[] gRow = new JComponent[] { gammaLabel, gammaSlider };
    final JComponent[][] rows = new JComponent[][] { oRow, cRow, gRow };
    return GUIUtils.createBasicGridPanel(rows);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public BufferedImage transformImage(BufferedImage image) {
    float offset = offsetSlider.getValue();
    float contrast = contrastSlider.getValue() / 100.F;
    float gamma = gammaSlider.getValue() / 100.F;

    short[] lookup = new short[256];
    for (int i = 0; i < 256; i++) {
      final double d = contrast * Math.pow(i + offset, gamma);
      short v = (short) Math.round(d);
      if (v < 0) {
        v = 0;
      }
      if (v > 255) {
        v = 255;
      }
      lookup[i] = v;
    }
    final ShortLookupTable lookupTable = new ShortLookupTable(0, lookup);
    BufferedImageOp lightOp = new LookupOp(lookupTable, null);
    BufferedImage newImage = lightOp.filter(image, null);
    return newImage;

    //    final RescaleOp lightOp = new RescaleOp(contrast, offset, null);
    //    BufferedImage newImage = lightOp.filter(image, null);
    //    return newImage;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void resetParameters() {
    offsetSlider.setValue(DEFAULT_OFFSET);
    contrastSlider.setValue(DEFAULT_CONTRAST);
    gammaSlider.setValue(DEFAULT_GAMMA);
  }
}
