package csbase.client.algorithms.parameters;

import java.awt.Window;
import java.util.Map;
import java.util.Set;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.Client;
import csbase.client.desktop.CommandNotificationHandler;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.RemoteTask;
import csbase.client.kernel.ClientException;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.CommandInfo;
import csbase.logic.CommandSubmission;
import csbase.logic.CommonClientProject;
import csbase.logic.Priority;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmVersionId;
import csbase.logic.algorithms.ExecutionType;
import csbase.logic.algorithms.parameters.ParameterLoader;
import csbase.remote.ClientRemoteLocator;

/**
 * Tarefa responsvel por executar o algoritmo extrator de parmetros.
 *
 * XXX - Internacionalizar.
 */
public class ParameterLoaderCommandExecutionTask extends RemoteTask<Boolean> {
  /** A janela da aplicao [para dialogs de erro] */
  private final Window window;
  /** A viso do componente do carregagor */
  private final ParameterLoaderView parameterView;
  /** Flag que indica se o extrator est sendo executado */
  private boolean isExtractingParameters;
  /** Flag que indica se o extrator foi executado com sucesso. */
  private boolean success;
  /** Identificador do comando em execuo */
  private String cmdId;

  /**
   * Cria a tarefa.
   *
   * @param window a janela que solicitou a tarefa.
   * @param parameterView a viso do carregador de parmetros que est
   *        utilizando a tarefa.
   */
  public ParameterLoaderCommandExecutionTask(Window window,
    ParameterLoaderView parameterView) {
    this.window = window;
    this.parameterView = parameterView;
    this.isExtractingParameters = false;
  }

  /**
   * Cancela a tarefa. Interrompe a execuo do comando extrator.
   */
  @Override
  protected void cancelTask() {
    super.cancelTask();
    RemoteTask<Object> cancelTask = new RemoteTask<Object>() {
      @Override
      protected void performTask() throws Exception {
        if (!ClientRemoteLocator.schedulerService.removeCommand(cmdId)) {
          ClientRemoteLocator.sgaService.killCommand(cmdId);
        }
      }
    };
    cancelTask.execute(window, LNG
      .get("ParameterLoaderCommandExecutionTask.title"), LNG
      .get("ParameterLoaderCommandExecutionTask.cancel.msg"));
  }

  /**
   * Submete o comando que far o carregamento do parmetro.
   *
   * @throws Exception em caso de erro durante a submisso.
   */
  @Override
  protected void performTask() throws Exception {
    ParameterLoader parameterLoader = parameterView.getParameterLoader();
    Map<String, String> extractParametersValues =
      parameterView.getExtractParametersValues();
    isExtractingParameters = true;
    success = false;

    // Define um nome para o arquivo de sada temporrio, gerado
    // pelo algoritmo extrator.
    extractParametersValues.put(parameterLoader.getOutputParameterName(),
      parameterView.getOutputExtractFileName());
    // Obtm o configurador do algoritmo extrator e preenche os
    // valores de seus parmetros.
    String algoName = parameterLoader.getAlgorithmName();
    AlgorithmVersionId versionId = parameterLoader.getAlgorithmVersionId();
    AlgorithmConfigurator extractConfigurator =
      ClientRemoteLocator.algorithmService.createAlgorithmConfigurator(
        algoName, versionId);
    extractConfigurator.setParameterValuesByName(extractParametersValues);
    // Submisso do comando para o escalonador.
    CommonClientProject project = DesktopFrame.getInstance().getProject();
    String clientName = Client.getInstance().getClientName();
    CommandSubmission submission =
      new CommandSubmission(extractConfigurator, project.getId(), clientName);
    submission.setPriority(Priority.ROOT);
    submission.setDescription("Extrator de Parmetros");

    ExecutionType type = extractConfigurator.getExecutionType();
    if (type != ExecutionType.SIMPLE) {
      throw new ClientException(String.format(LNG
        .get("ParameterLoaderCommandExecutionTask.error.execution.type"), type));
    }
    Set<CommandInfo> commands =
      ClientRemoteLocator.schedulerService.submitCommand(submission);
    if (commands == null) {
      throw new ClientException(LNG
        .get("ParameterLoaderCommandExecutionTask.error.submit"));
    }
    CommandInfo command = commands.iterator().next();
    cmdId = command.getId();
    // Define o listener que vai esperar a resposta do extrator.
    // O listener  quem vai setar a varivel isExtractingParameters
    // indicando se a execuo terminou.
    CommandNotificationHandler.getInstance().addListener(
      new ParameterLoaderCommandListener(this, cmdId));

    // Para evitar busywaiting na espera pela notificao, faz
    // sleep.
    while (isExtractingParameters) {
      Thread.sleep(5 * 1000);
    }
    setResult(success);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void handleError(Exception error) {
    if (error instanceof ClientException) {
      StandardErrorDialogs.showErrorDialog(window, LNG
        .get("ParameterLoaderCommandExecutionTask.title"), error.getMessage());
    }
    else {
      super.handleError(error);
    }
  }

  /**
   * Atualiza os flags que indicam o final da execuo do Extrator. Este mtodo
   *  chamado pelo listener aps a execuo do Extrator.
   *
   * @param wasSuccessful indica se o comando foi terminado com sucesso.
   */
  public void executionCompleted(boolean wasSuccessful) {
    isExtractingParameters = false;
    this.success = wasSuccessful;
  }
}
