/*
 * FilePropertiesDialog.java
 * 
 * $Author: analodi $ $Revision: 170434 $ - $Date: 2007-03-12 15:18:17 -0300
 * (Mon, 12 Mar 2007) $
 */
package csbase.client.project;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.GridBagLayout;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.text.MessageFormat;

import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.SwingConstants;

import csbase.client.ClientSmartFile;
import csbase.client.ClientSmartFileFactory;
import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.desktop.RemoteTask;
import csbase.client.util.ClientUtilities;
import csbase.logic.ClientOptimizationMode;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectFileType;
import csbase.logic.User;
import csbase.logic.Utilities;
import csbase.logic.filters.ProjectFileTrueFilter;
import tecgraf.javautils.core.lng.FormatUtils;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;

/**
 * A classe <code>FilePropertiesDialog</code> modela uma janela que exibe as
 * propriedades de um <code>ProjectFile</code>.
 */
public class FilePropertiesDialog {

  /**
   * Dilogo principal.
   */
  private DesktopComponentDialog propertiesDialog = null;

  /**
   * Janela a partir do qual esse dilogo foi chamado.
   */
  private Window owner = null;

  /**
   * O boto default que fecha o dilogo.
   */
  private JButton closeButton;

  /**
   * Projeto onde est o arquivo.
   */
  private CommonClientProject project;

  /**
   * O arquivo cujas propriedades vo ser exibidas.
   */
  private ClientProjectFile file;

  /**
   * rea de descrio do arquivo
   */
  private JTextArea descriptionArea;

  /**
   * Label de estado do otimizao do arquivo
   */
  private JLabel optStatusLabel;

  /**
   * Nome do arquivo.
   */
  private String fileName;

  /**
   * Tamanho do arquivo.
   */
  private long fileSize;

  /**
   * Identificador do usurio que criou o arquivo.
   */
  private Object fileOwner;

  /**
   * Data de criao do arquivo.
   */
  private long fileCreationDate;

  /**
   * Data da ltima atualizao do arquivo.
   */
  private long fileLastUpdateDate;

  /**
   * Modo de otimizao
   */
  private ClientOptimizationMode fileOptimizationMode;

  /**
   * Caminho local do arquivo (caso esteja usando a otimizao)
   */
  private String fileOptimizationPath;

  /**
   * Mostra a janela com as propriedades de um arquivo de projeto.
   * 
   * @param owner A janela a partir do qual esse dilogo foi chamado.
   * @param project O projeto no qual est o arquivo.
   * @param file O arquivo cujas propriedades vo ser exibidas.
   */
  public static void show(Window owner, CommonClientProject project,
    ClientProjectFile file) {
    new FilePropertiesDialog(owner, project, file);
  }

  /**
   * Cria o dilogo.
   * 
   * @return dilogo.
   */
  private DesktopComponentDialog makeDialog() {
    /* Cria um dilogo no modal */
    DesktopComponentDialog dialog =
      new DesktopComponentDialog(owner, getTitle());
    dialog.setModal(false);
    /* Cria o painel principal do dilogo */
    JPanel mainPanel = new JPanel(new GridBagLayout());
    mainPanel.add(makePropertiesPanel(dialog.getBackground()),
      new GBC(0, 0).fillxy().weights(1, 1));
    mainPanel.add(makeButtonPanel(), new GBC(0, 1));
    dialog.getContentPane().add(mainPanel);
    dialog.pack();
    dialog.center(owner);
    dialog.getRootPane().setDefaultButton(closeButton);
    return dialog;
  }

  /**
   * Cria o painel com as propriedades do arquivo.
   * 
   * @param background A cor de fundo usada pelos componentes.
   * 
   * @return painel de propriedades.
   */
  private JPanel makePropertiesPanel(final Color background) {
    ImageIcon icon = ClientProjectFileProperties.getImageIcon(this.file);
    String accessMode = ClientProjectFileProperties.getAccessMode(this.file);
    ProjectFileType type = ProjectFileType.getFileType(file.getType());
    /* Campo com o nome do arquivo */
    JTextField nameTextField = new JTextField(fileName);
    nameTextField.setEditable(false);
    /* Campo com a localizao */
    String fileLocalization = buildLocalization();
    JTextField pathTextField = new JTextField(fileLocalization);
    pathTextField.setEditable(false);
    /* Tipo do arquivo */
    JLabel typeLabel = new JLabel(icon, SwingConstants.LEFT);
    JTextArea typeTextArea = new JTextArea(" " + type.getDescription());
    typeTextArea.setBackground(background);
    typeTextArea.setEditable(false);
    JPanel typePanel = new JPanel(new FlowLayout(FlowLayout.LEADING, 0, 0));
    typePanel.add(typeLabel);
    typePanel.add(typeTextArea);
    /* Indicao se o arquivo  s leitura ou leitura e escrita */
    JTextArea readOnlyTextArea = new JTextArea(accessMode);
    readOnlyTextArea.setBackground(background);
    readOnlyTextArea.setEditable(false);
    /* Tamanho do arquivo */
    JTextArea sizeTextArea = new JTextArea(FormatUtils.formatSize(fileSize, 2));
    sizeTextArea.setBackground(background);
    sizeTextArea.setEditable(false);
    /* Data de criao do arquivo */
    String creationDate = " ";
    if (fileCreationDate != 0) {
      creationDate = Utilities.getFormattedDate(fileCreationDate);
    }
    JTextArea creationDateTextArea = new JTextArea(creationDate);
    creationDateTextArea.setBackground(background);
    creationDateTextArea.setEditable(false);

    /* Data da ltima alterao do arquivo */
    String lastUpdateDate = " ";
    if (fileLastUpdateDate != 0) {
      lastUpdateDate = Utilities.getFormattedDate(fileLastUpdateDate);
    }
    JTextArea lastUpdateDateTextArea = new JTextArea(lastUpdateDate);
    lastUpdateDateTextArea.setBackground(background);
    lastUpdateDateTextArea.setEditable(false);

    /* Usurio que criou o arquivo */
    String userName;
    try {
      userName = User.getName(fileOwner);
      if (userName == null) {
        userName = " ";
      }
    }
    catch (Exception e) {
      userName = " ";
    }
    JTextArea ownerTextArea = new JTextArea(userName);
    ownerTextArea.setBackground(background);
    ownerTextArea.setEditable(false);

    JTextField optModeTextField =
      new JTextField(getOptimizationModeText(fileOptimizationMode));
    optModeTextField.setBackground(background);
    optModeTextField.setEditable(false);
    optModeTextField.setBorder(null);

    /* Caixa de texto que pode ser "percorrida com cursor" mas no editada */
    final JTextField optPathTextField =
      new JTextField(fileOptimizationPath, 20);
    optPathTextField.setBackground(background);
    optPathTextField.addFocusListener(new FocusListener() {

      public void focusLost(FocusEvent fe) {
        optPathTextField.setEditable(true);
        optPathTextField.setBackground(background);
      }

      public void focusGained(FocusEvent fe) {
        optPathTextField.setEditable(false);
      }
    });
    optPathTextField.setCaretPosition(0);

    /* Estado de otimizao do arquivo */
    optStatusLabel = new JLabel();
    optStatusLabel.setPreferredSize(new Dimension(110, 25));
    JButton optStatusButton =
      new JButton(LNG.get("PRJ_FILE_OPTIMIZATION_STATUS_BUTTON"));
    optStatusButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        boolean status = getOptimizationStatus(file);
        setOptStatusTextField(status);
      }
    });
    optStatusButton.setPreferredSize(new Dimension(100, 25));
    boolean status = false;
    status = getOptimizationStatus(file);
    setOptStatusTextField(status);
    JPanel optStatusPanel = new JPanel(new GridBagLayout());
    optStatusPanel.add(optStatusLabel, new GBC(0, 0).west().weightx(1));
    optStatusPanel.add(optStatusButton, new GBC(1, 0));

    JPanel descrPanel = makeDescriptionPanel(background);

    /* Monta o panel usando grid layout */
    JPanel panel = buildPropertiesLayout(nameTextField, pathTextField,
      typePanel, readOnlyTextArea, sizeTextArea, creationDateTextArea,
      lastUpdateDateTextArea, ownerTextArea, optModeTextField, optPathTextField,
      optStatusPanel, descrPanel);
    return panel;
  }

  /**
   * 
   * @param background Cor de fundo padro
   * @return O Painel de descrio
   */
  private JPanel makeDescriptionPanel(final Color background) {
    String description = getFileDescription();
    descriptionArea = new JTextArea(description);
    descriptionArea.setBackground(background);
    descriptionArea.setEditable(false);
    descriptionArea.setFont(new Font("Monospaced", Font.PLAIN, 12));
    descriptionArea.setLineWrap(true);
    descriptionArea.setWrapStyleWord(true);
    JScrollPane textScrollPane = new JScrollPane(descriptionArea);
    Dimension scrDim = new Dimension(400, 200);
    textScrollPane.setSize(scrDim);
    textScrollPane.setPreferredSize(scrDim);
    textScrollPane.setMinimumSize(scrDim);

    JButton descriptionButton = new JButton(LNG.get("PRJ_FILE_ADD"));
    descriptionButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        AppendDescriptionDialog appendDialog =
          new AppendDescriptionDialog(FilePropertiesDialog.this);
        appendDialog.setVisible(true);
      }
    });

    JPanel descrPanel = new JPanel(new GridBagLayout());
    descrPanel.add(textScrollPane,
      new GBC(0, 0).fillxy().weightx(1).top(5).left(5).bottom(5).right(5));
    descrPanel.add(descriptionButton, new GBC(0, 1).top(5).bottom(5));
    descrPanel.setBorder(
      BorderFactory.createTitledBorder(LNG.get("PRJ_FILE_DESCRIPTION")));
    return descrPanel;
  }

  /**
   * Configura o label de acordo como o estado da otimizao
   * 
   * @param status estado da otimizao
   */
  private void setOptStatusTextField(boolean status) {
    optStatusLabel.setForeground(status ? new Color(35, 142, 35) : Color.RED);
    optStatusLabel.setText(status ? LNG.get("PRJ_FILE_OPTIMIZATION_STATUS_OK")
      : LNG.get("PRJ_FILE_OPTIMIZATION_STATUS_FAILED"));
  }

  /**
   * 
   * @param cpf Arquivo de projeto
   * @return true, se o arquivo pode utilizar a otimizao agora. Teste
   *         imediato. False, caso contrrio.
   */
  private boolean getOptimizationStatus(ClientProjectFile cpf) {

    final ClientSmartFile csf =
      ClientSmartFileFactory.getInstance().create(cpf);
    RemoteTask<Boolean> task = new RemoteTask<Boolean>() {
      @Override
      protected void performTask() throws Exception {
        setResult(csf.testOptimization());
      }
    };
    task.execute(owner, getTitle(), LNG.get("PRJ_FILE_OPTIMIZATION_TASK_MSG"));
    if (task.getStatus()) {
      return task.getResult();
    }
    return false;
  }

  /**
   * 
   * @param mode Modo de otimizao
   * @return Texto que deve ser mostrado para indicar o modo de otimizao
   */
  private String getOptimizationModeText(ClientOptimizationMode mode) {
    String key = "PRJ_FILE_OPTIMIZATION." + mode.name();
    return LNG.get(key);
  }

  /**
   * 
   * @return true, se o arquivo utiliza alguma otimizao
   */
  private boolean isFileOptimized() {
    return !fileOptimizationMode.equals(ClientOptimizationMode.NONE);
  }

  /**
   * Monta o layout do painel de propriedades.
   * 
   * @param nameTextField campo de nome.
   * @param pathTextField campo de localizao.
   * @param typePanel campo de tipo.
   * @param readOnlyTextArea campo leitura-somente.
   * @param sizeTextArea campo de tamanho.
   * @param creationDateTextArea campo data de criao.
   * @param lastUpdateDateTextArea campo data da ltima atualizao.
   * @param ownerTextArea campo do autor.
   * @param optModeTextField Modo de otimizao
   * @param optPathTextField Path do arquivo no modo de otimizao
   * @param optStatusPanel Panel com o boto de status
   * @param descriptionPanel rea de descrio.
   * 
   * @return painel de propriedades.
   */
  private JPanel buildPropertiesLayout(JTextField nameTextField,
    JTextField pathTextField, JPanel typePanel, JTextArea readOnlyTextArea,
    JTextArea sizeTextArea, JTextArea creationDateTextArea,
    JTextArea lastUpdateDateTextArea, JTextArea ownerTextArea,
    JTextField optModeTextField, JTextField optPathTextField,
    JPanel optStatusPanel, JPanel descriptionPanel) {

    JComponent[][] rowsPanel1 =
      new JComponent[][] {
          { new JLabel(LNG.get("PRJ_FILE_NAME")), nameTextField },
          { new JLabel(LNG.get("PRJ_FILE_LOCALIZATION")), pathTextField },
          { new JLabel(LNG.get("PRJ_FILE_TYPE")), typePanel },
          { null,
              readOnlyTextArea },
          { new JLabel(LNG.get("PRJ_FILE_SIZE")), sizeTextArea },
          { new JLabel(LNG.get("PRJ_FILE_CREATION_DATE")),
              creationDateTextArea },
          { new JLabel(LNG.get("PRJ_FILE_LAST_UPDATE_DATE")),
              lastUpdateDateTextArea },
        { new JLabel(LNG.get("PRJ_FILE_AUTHOR")), ownerTextArea },
        { new JLabel(LNG.get("PRJ_FILE_SOURCE")), null },
        { new JLabel(LNG.get("PRJ_FILE_OPTIMIZATION")), optModeTextField },
        { isFileOptimized() ? new JLabel(LNG.get("PRJ_FILE_OPTIMIZATION_PATH"))
          : null,
            isFileOptimized() ? optPathTextField : null },
          { isFileOptimized()
            ? new JLabel(LNG.get("PRJ_FILE_OPTIMIZATION_STATUS")) : null,
              isFileOptimized() ? optStatusPanel : null } };

    JPanel panel1 = GUIUtils.createBasicGridPanel(rowsPanel1);
    JPanel panel2 = GUIUtils
      .createBasicGridPanel(new JComponent[][] { { descriptionPanel } });

    JPanel panel = new JPanel(new GridBagLayout());
    panel.add(panel1, new GBC(0, 0).northwest().fillx());
    panel.add(panel2, new GBC(0, 1).northwest().fillx());

    return panel;
  }

  /**
   * Cria o painel com os botes.
   * 
   * @return painel de botes.
   */
  private JPanel makeButtonPanel() {
    JPanel panel = new JPanel();
    closeButton = new JButton(LNG.get("UTIL_CLOSE"));
    panel.add(closeButton);
    closeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        propertiesDialog.close();
      }
    });
    return panel;
  }

  /**
   * Retorna dilogo principal.
   * 
   * @return Dilogo principal.
   */
  public JDialog getDialog() {
    return propertiesDialog;
  }

  /**
   * Informa o ttulo que vai ser utilizado na janela de exibio de
   * propriedades de um arquivo.
   * 
   * @return O ttulo da janela.
   */
  protected String getTitle() {
    String title = MessageFormat.format(
      LNG.get("project.file_see_properties_title"), new Object[] { fileName });
    return ClientUtilities.addSystemNameToTitle(title);
  }

  /**
   * Recupera as propriedades do arquivo.
   */
  private void retrieveFileProperties() {
    RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      public void performTask() throws Exception {
        file.updateInfo();
        fileName = file.getName();
        fileSize = file.getTotalSize(ProjectFileTrueFilter.getInstance());
        fileOwner = file.whoCreated();
        fileCreationDate = file.getCreationDate();
        fileLastUpdateDate = file.getModificationDate();
      }
    };
    task.execute(owner, getTitle(),
      LNG.get("PRJ_WAITING_OBTAIN_FILE_PROPERTIES"));

    final ClientSmartFile smartFile =
      ClientSmartFileFactory.getInstance().create(file);
    fileOptimizationMode = smartFile.getOptimizationMode();
    fileOptimizationPath = smartFile.getLocalPath();
  }

  /**
   * 
   * @return A descrio do arquivo (consulta remota)
   */
  private String getFileDescription() {
    RemoteTask<String> task = new RemoteTask<String>() {
      @Override
      protected void performTask() throws Exception {
        setResult(file.getDescription());
      }
    };
    task.execute(owner, getTitle(),
      LNG.get("PRJ_WAITING_OBTAIN_FILE_PROPERTIES"));
    if (task.getStatus()) {
      return task.getResult();
    }
    return "";
  }

  /**
   * Constri a representao de onde se encontra um determinado arquivo. A
   * String resultante possui a seguinte forma: [nome-do-projeto]:
   * caminho-dos-diretrios
   * 
   * @return A string que representa a localizao de um arquivo na rvore do
   *         projeto.
   */
  private String buildLocalization() {
    String path = "[" + project.getName() + "]" + ": ";
    String[] filePath = file.getPath();
    for (int i = 1; i < (filePath.length - 1); i++) {
      path += (filePath[i] + "/");
    }
    return path;
  }

  /**
   * Constri o dilogo de exibio de propriedades.
   * 
   * @param owner A janela pai a partir da qual esse dilogo  chamado.
   * @param project O projeto no qual est o arquivo.
   * @param file O arquivo cujas propriedades vo ser exibidas.
   */
  private FilePropertiesDialog(Window owner, CommonClientProject project,
    ClientProjectFile file) {
    this.owner = owner;
    this.project = project;
    this.file = file;
    retrieveFileProperties();
    propertiesDialog = makeDialog();
    propertiesDialog.setVisible(true);
  }

  /**
   * Atualiza a rea de descrio do arquivo exibida no dilogo.
   */
  public void updateDescription() {
    String description = getFileDescription();
    descriptionArea.setText(description);
  }

  /**
   * Obtm o arquivo cujas propriedades esto sendo exibidas.
   * 
   * @return arquivo.
   */
  public ClientProjectFile getFile() {
    return file;
  }
}
