/*
 * $Id$
 */
package csbase.client.algorithms.commands.cache.events;

import csbase.client.algorithms.commands.cache.events.CommandUpdatedEvent.Type;
import csbase.client.util.event.EventListener;
import csbase.logic.CommandInfo;

/**
 * @author Tecgraf / PUC-Rio
 * 
 * Representa o contrato de um observador de eventos do tipo
 * {@link CommandUpdatedEvent}
 * 
 * @see CommandUpdatedEvent
 */
public abstract class AbstractCommandUpdatedEventListener implements
  EventListener<CommandUpdatedEvent> {

  /**
   * O identificador do projeto do comando sendo observado.
   */
  private final Object projectId;  
  /**
   * O identificador do comando sendo observado.
   */
  private final String commandId;

  /**
   * Construtor.
   * 
   * @param projectId Identificador do projeto do comando sendo observado.
   * @param commandId Identificador do comando que deseja observar.
   */
  public AbstractCommandUpdatedEventListener(Object projectId, String commandId) {
    this.projectId =projectId; 
    this.commandId = commandId;
  }

  @Override
  public final void eventFired(CommandUpdatedEvent event) {
    if (Type.exception == event.getType()) {
      eventInterrupted(event.getException(), event.getExceptionDescription());
    }
    else {
      if (commandId.equals(event.getCommand().getId())) {
        eventFired(event.getType(), event.getCommand());
      }
    }
  }
  
  /**
   * Obtm o identificador do projeto do comando sendo observado.
   * 
   * @return o identificador do projeto do comando sendo observado.
   */
  public Object getProjectId() {
    return projectId;
  }

  /**
   * Obtm o identificador do comando sendo observado.
   * 
   * @return o identificador do comando sendo observado.
   */
  public final String getCommandId() {
    return commandId;
  }  
  
  /**
   * Indica que houve uma alterao no comando observado.
   * 
   * @param type tipo de evento.
   * @param command comando sendo observado.
   */
  protected abstract void eventFired(Type type, CommandInfo command);

  /**
   * Indica que o processo de atualizao da cache foi interrompido com uma
   * exceo.
   * 
   * @param exception Exceo que interrompeu o processo de atualizao da
   *        cache.
   * @param description Descrio da exceo (pode ser <tt>null</tt>).
   */
  protected void eventInterrupted(Exception exception, String description) {
  }
}
