/*
 * $Id: DesktopFrame.java 172010 2016-03-14 17:42:19Z clinio $
 */
package csbase.client.desktop;

import java.awt.AWTException;
import java.awt.CheckboxMenuItem;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.Image;
import java.awt.MenuItem;
import java.awt.PopupMenu;
import java.awt.SystemTray;
import java.awt.TrayIcon;
import java.awt.TrayIcon.MessageType;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.net.URL;
import java.rmi.RemoteException;
import java.util.*;

import javax.swing.AbstractAction;
import javax.swing.ButtonGroup;
import javax.swing.ImageIcon;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JSplitPane;
import javax.swing.LookAndFeel;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.UIManager.LookAndFeelInfo;

import csbase.client.Client;
import csbase.client.ClientServerManager;
import csbase.client.ClientUI;
import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applicationmanager.ApplicationManager;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.executor.ExecutorFrame;
import csbase.client.desktop.dircontents.DirectoryContentsPanel;
import csbase.client.ias.AdminFrame;
import csbase.client.kernel.ClientException;
import csbase.client.login.InitialContext;
import csbase.client.preferences.PreferenceCategory;
import csbase.client.preferences.PreferenceManager;
import csbase.client.preferences.types.PVBoolean;
import csbase.client.preferences.types.PVDimension;
import csbase.client.preferences.types.PVInteger;
import csbase.client.project.ProjectTree;
import csbase.client.project.ProjectTreeAdapter;
import csbase.client.project.RecentProjectsManager;
import csbase.client.project.action.OneProjectOpenAction;
import csbase.client.project.action.ProjectCloseAction;
import csbase.client.project.action.ShowTreeFilterAction;
import csbase.client.project.dialogs.ProjectBuildingDialog;
import csbase.client.project.tasks.OpenProjectOnTreeRemoteTask;
import csbase.client.remote.AdministrationObserver;
import csbase.client.remote.AlgorithmManagementObserver;
import csbase.client.remote.ClientRemoteMonitor;
import csbase.client.remote.ClientRemoteMonitorListener;
import csbase.client.remote.ProjectAdminObserver;
import csbase.client.remote.manager.server.ServerInfoManager;
import csbase.client.remote.srvproxies.EventLogProxy;
import csbase.client.remote.srvproxies.NotificationProxy;
import csbase.client.util.StandardErrorDialogs;
import csbase.exception.ConfigurationException;
import csbase.logic.*;
import csbase.logic.applicationservice.ApplicationRegistry;
import csbase.logic.filetypefinder.FileTypeFinder;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.ProjectServiceInterface;
import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.configurationmanager.ConfigurationManagerException;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;

/**
 * A classe <code>DesktopFrame</code> implementa a interface principal do
 * sistema no lado cliente.
 *
 * @author Tecgraf/PUC-Rio
 */
public class DesktopFrame implements ClientUI {

  /**
   * Atributo da sesso que representa o nome de um projeto a ser aberto ao
   * iniciar o desktop.
   */
  private static final String PROJECT_NAME_ATTRIBUTE = "project_name";

  /**
   * Atributo da sesso que representa o nome de um algoritmo a ser executado ao
   * iniciar o desktop.
   */
  private static final String ALGORITHM_NAME_ATTRIBUTE = "algorithm_name";

  /**
   * Atributo da sesso que representa os parmetros de um algoritmo a ser
   * executado ao iniciar o desktop.
   */
  private static final String ALGORITHM_PARAMETERS_ATTRIBUTE =
    "algorithm_parameters";

  /**
   * Nmero mximo de projetos recentes mostrados no menu. O valor mximo a ser
   * considerado  ProjectsHistory.MAX_PROJECTS,
   */
  private static final int MAX_RECENT_PROJECTS = 10;

  /**
   * Fila de log de aes do cliente
   */
  private static String[] desktopQueue = new String[] { "Desktop" };

  /**
   * Componente responsvel pela rea de aplicaes.
   */
  private static DesktopFrame instance = null;

  /**
   * Componente que indica como se a atualizao da rvore de projetos deve ser
   * automtica.
   */
  private static JCheckBoxMenuItem projectAutoUpdateItem;

  /**
   * Componente que indica a rvore deve exibir ou no um filtro dos seus ns.
   */
  private static JCheckBoxMenuItem showTreeFilterItem;

  /**
   * A janela de administrao do sistema.
   */
  protected AdminFrame adminFrame;

  /**
   * Painel que exibe o contedo do diretrio selecionado na rvore
   */
  protected DirectoryContentsPanel dirContentsPanel;

  /**
   * Painel de notificaes.  protected porque sua inicializao pode ser
   * sobrescrita pelas subclasses.
   */
  protected NotificationPanel notifPanel;

  /**
   * Painel da rvore de projetos.
   */
  protected JComponent projectTreePanel;

  /**
   * JSplitPane que separa rvore de projetos e a tabela com os arquivos.
   */
  protected JSplitPane splitPaneBetweenTreeAndTable;

  /**
   * JSplitPane que separa rvore de projetos e os frames das aplicaes.
   */
  protected JSplitPane splitPaneBetweenTreeAndApp;

  /**
   * JSplitPane que separa os frames das aplicaes e o painel de notificaes.
   */
  protected JSplitPane splitPaneBetweenAppAndNotif;

  /**
   * A rvore de diretrios do projeto aberto
   */
  protected ProjectTree projectTree;

  /**
   * Item de menu default para encerramentodo desktop.
   */
  private MenuItem defaultExitTrayMenuItem;

  /**
   * Item de menu default para visibilidade do desktop.
   */
  private CheckboxMenuItem defaultVisibilityTrayCheckMenuItem;

  /**
   * Vetor de janelas
   */
  private Vector<DesktopWindowInterface> dependentWindows = null;

  /**
   * A janela onde essa interface  apresentada.
   */
  private DesktopComponentFrame mainFrame = null;

  /**
   * Este Listener recebe as notificaes quando a conexo com o servidor
   * perdida ou restabelecida. Exibe mensagens nessas situaes.
   */
  protected MonitoredServerListener connMonitorListener;

  /**
   * Imagem de erro
   */
  private Image errorImage = null;

  /**
   * Gerente de configuraes. Usado para obteno de propriedades desta classe.
   */
  private Configuration configuration;

  /**
   * Painel de filtro para a rvore de projetos.
   */
  protected TreeFilterPanel treeFilterPanel;

  /**
   * Indica se arquivos ocultos (que comeam com ".") devem ser exibidos.
   */
  private Boolean showHiddenFiles;

  /**
   * Tray Icon do desktop (pode estar null se o S.O no der suporte a este
   * recurso).
   */
  final private TrayIcon trayIcon;

  /**
   * Tratador padro de eventos do desktop CSBASE.
   */
  private DesktopFrameOpenBusEventHandler eventHandler;

  // ----------- Atributos relativos a preferncias do usurio --------------

  /**
   * Encerra o cliente fechando todas as janelas dependentes do tipo
   * DesktopComponentFrame.
   */
  private void shutdownDependents() {
    int base = 0;
    DesktopWindowInterface lastWindow = null;
    while (dependentWindows.size() > 0) {
      DesktopWindowInterface window = dependentWindows.get(base);
      if (window == lastWindow) {
        base++;
        continue;
      }
      window.close();
      lastWindow = window;
    }
  }

  /**
   * Mtodo para ajustar o ttulo das janelas dependentes do tipo
   * DesktopComponentFrame.
   *
   * @param title ttulo
   * @return o novo ttulo.
   */
  public String adjustTitle(String title) {
    return title;
  }

  /**
   * Mtodo para atribuir o ttulo do desktop.
   *
   * @param title ttulo
   */
  protected final void setTitle(String title) {
    this.mainFrame.setTitle(title);
  }

  /**
   * Mtodo para consultar o ttulo do desktop.
   *
   * @return ttulo
   */
  public String getTitle() {
    return this.mainFrame.getTitle();
  }

  /**
   * Mtodo para atribuir a imagem do desktop frame.
   *
   * @param image a imagem da janela do desktop.
   */
  public void setFrameImage(Image image) {
    this.mainFrame.setIconImage(image);
  }

  /**
   * Obtm imagem de erro.
   *
   * @return a imagem
   */
  public Image getDesktopErrorImage() {
    return this.errorImage;
  }

  /**
   * Configura a imagem de erro.
   *
   * @param image a imagem
   */
  public void setDesktopErrorImage(Image image) {
    this.errorImage = image;
  }

  /**
   * Mtodo para consultar a imagem do desktop frame.
   *
   * @return a imagem.
   */
  public Image getFrameImage() {
    return this.mainFrame.getIconImage();
  }

  /**
   * Mtodo para atribuir o menu bar da janela.
   *
   * @param menuBar o menu
   */
  public void setMenuBar(JMenuBar menuBar) {
    this.mainFrame.setJMenuBar(menuBar);
  }

  /**
   * Mtodo para consultar o menu bar da janela.
   *
   * @return menubar o menu
   */
  public JMenuBar getMenuBar() {
    final JMenuBar menuBar = this.mainFrame.getJMenuBar();
    return menuBar;
  }

  /**
   * Encerra a aplicao cliente principal.
   */
  public void shutdownDesktop() {
    if (eventHandler != null) {
      eventHandler.shutdown();
    }

    final ClientRemoteMonitor clientRemoteMonitor =
      ClientRemoteMonitor.getInstance();

    clientRemoteMonitor.deleteListener(connMonitorListener);

    /* Termina todas as aplicaes iniciadas dentro do Desktop */
    EventLogProxy.addClientInformation(desktopQueue, "logout");
    String[] queue = { "Idiom" };
    String[] msg = { "Textos sem chave em qualquer idioma:\n",
      Client.getInstance().getTextNotFoundInAnyIdiom() };
    EventLogProxy.addClientInformation(queue, msg);
    msg =
      new String[] { "Textos sem chave no idioma " + LNG.getLocale() + ":\n",
        Client.getInstance().getTextNotFoundInSelectedIdiom() };
    EventLogProxy.addClientInformation(queue, msg);

    final ApplicationManager appManager = ApplicationManager.getInstance();
    appManager.finishAllApplications();

    shutdownDependents();
    stopServicesObservers();

    // Guarda as configuraes do desktop do usurio
    if (clientRemoteMonitor.isAlive()) {
      PreferenceCategory dp = getDesktopPreferences();
      if (dp.getPreferenceAsBoolean(DesktopPref.SAVE_STATE)) {
        saveDesktopPreferences();
      }
      ClientServerManager.getInstance().shutdown();
    }

    mainFrame.setVisible(false);
    mainFrame.close();

    synchronized (instance) {
      instance = null;
    }

    final Client client = Client.getInstance();
    client.shutdown();
  }

  /**
   * Ajusta o desktop de acordo com as preferncias.
   */
  protected void configureDesktop() {
    PreferenceCategory dp = getDesktopPreferences();

    PVDimension size = (PVDimension) dp.getPreference(DesktopPref.SIZE);
    setDesktopSize(size.getValue());

    if (dp.getPreferenceAsBoolean(DesktopPref.MAXIMIZED)) {
      maximizeDesktop();
    }

    int dividerAppNotif = dp.getPreferenceAsInt(DesktopPref.DIV_APP_NOTIF);
    int dividerTreeTable = dp.getPreferenceAsInt(DesktopPref.DIV_TREE_TABLE);
    int dividerTreeApp = dp.getPreferenceAsInt(DesktopPref.DIV_TREE_APP);

    splitPaneBetweenAppAndNotif.setDividerLocation(dividerAppNotif);
    splitPaneBetweenTreeAndTable.setDividerLocation(dividerTreeTable);
    splitPaneBetweenTreeAndApp.setDividerLocation(dividerTreeApp);
  }

  /**
   * Salva as configuraes do desktop do usurio.
   */
  protected void saveDesktopPreferences() {
    PreferenceCategory dp = getDesktopPreferences();

    PVDimension size = (PVDimension) dp.getPreference(DesktopPref.SIZE);
    size.setValue(
      new Dimension(mainFrame.getSize().width, mainFrame.getSize().height));

    PVBoolean maximized = dp.getPVBoolean(DesktopPref.MAXIMIZED);
    maximized.setValue(mainFrame.getExtendedState() == JFrame.MAXIMIZED_BOTH);

    PVInteger dividerAppNotif = dp.getPVInteger(DesktopPref.DIV_APP_NOTIF);
    dividerAppNotif.setValue(splitPaneBetweenAppAndNotif.getDividerLocation());

    PVInteger dividerTreeTable = dp.getPVInteger(DesktopPref.DIV_TREE_TABLE);
    dividerTreeTable
      .setValue(splitPaneBetweenTreeAndTable.getDividerLocation());

    PVInteger dividerTreeApp = dp.getPVInteger(DesktopPref.DIV_TREE_APP);
    dividerTreeApp.setValue(splitPaneBetweenTreeAndApp.getDividerLocation());

    PreferenceManager.getInstance().savePreferences();
  }

  /**
   * Obtm as preferncias do usurio para configurao do desktop.
   *
   * @return preferncias do usurio para configurao do desktop.
   */
  protected PreferenceCategory getDesktopPreferences() {
    PreferenceManager pm = PreferenceManager.getInstance();
    PreferenceCategory allPrefs = pm.loadPreferences();

    return allPrefs.getCategory(DesktopPref.class);
  }

  /**
   * Indica se a rvore de projetos deve ser autualizada automaticamente.
   *
   * @return true se a rvore deve ser atualizada.
   */
  public boolean mustUpdateProjectTree() {
    PreferenceCategory dp = getDesktopPreferences();
    return dp.getPreferenceAsBoolean(DesktopPref.AUTO_UPDATE);
  }

  /**
   * Registra uma janela cliente na janela principal. Ao fechar a janela
   * pricipal, todas as janelas registradas tambm so fechadas.
   *
   * @param win Uma nova janela registrada
   */
  public void addWindow(DesktopWindowInterface win) {
    dependentWindows.add(win);
  }

  /**
   * Descadastra uma janela da cliente janela principal.
   *
   * @param win A janela que deve ser removida da janela principal.
   */
  public void removeWindow(DesktopWindowInterface win) {
    dependentWindows.remove(win);
  }

  /**
   * Consulta  janela Java (JFrame)
   *
   * @return .
   */
  public DesktopComponentFrame getDesktopFrame() {
    return mainFrame;
  }

  /**
   * Verifica se o usuario logado possui projetos
   *
   * @return <code>true</code> se o usurio possui projetos
   */
  public boolean userHasProject() {
    RemoteTask<Boolean> task = new RemoteTask<Boolean>() {
      @Override
      protected void performTask() throws Exception {
        setResult(CommonClientProject
          .userHasHisOwnProjects(User.getLoggedUser().getId()));
      }
    };

    if (!task.execute(getDesktopFrame(), getTitle(),
      LNG.get("DesktopFrame.info.checkIfUserHasProject"))) {
      // o erro j foi exibido
      return false;
    }
    return task.getResult();
  }

  /**
   * Verifica se o usuario logado participa de projetos de outros usurios
   *
   * @return <code>true</code> se o usurio participa de projetos de outros
   * usurios
   */
  public boolean userHasOthersProject() {
    try {
      final User loggedUser = User.getLoggedUser();
      return CommonClientProject
        .userParticipatesOnSharedProjects(loggedUser.getId());
    }
    catch (Exception e) {
      StandardErrorDialogs.showErrorDialog(getDesktopFrame(), getTitle(), e);
    }
    return false;
  }

  /**
   * Pendura um listener na rvore de projetos, que ir verificar, a cada vez
   * que um projeto for removido, se o usurio ainda possui projetos. Se o
   * projeto removido for o ltimo do usurio, o item especificado ser
   * removido.
   *
   * @param item componente da interface, geralmente um menu ou um boto.
   */
  protected void enableIfProject(final JComponent item) {
    projectTree.addProjectTreeListener(new ProjectTreeAdapter() {
      @Override
      public void projectRemoved() {
        item.setEnabled(userHasProject());
      }
    });
  }

  /**
   * Consulta o file type finder da rvore de projetos
   *
   * @return buscador de tipo de arquivo.
   */
  public final FileTypeFinder getFileTypeFinder() {
    return projectTree.getFileTypeFinder();
  }

  /**
   * Pendura um listener para na rvore de projetos que ser notificado a cada
   * vez que um projeto for aberto ou fechado. Se for um evento de abertura de
   * projeto, o item especificado ser habilitado.
   *
   * @param item item componente da interface, geralmente um menu ou um boto.
   */
  protected void enableIfProjectOpened(final JComponent item) {
    projectTree.addProjectTreeListener(new ProjectTreeAdapter() {
      @Override
      public void projectChanged(CommonClientProject project) {
        item.setEnabled(project != null);
      }
    });
  }

  /**
   * Habilita o gerencimento do projeto se o usurio logado for dono do mesmo ou
   * se for o administrador.
   *
   * @param item .
   */
  protected void enableIfAdminProject(final JMenuItem item) {
    projectTree.addProjectTreeListener(new ProjectTreeAdapter() {
      @Override
      public void projectChanged(CommonClientProject project) {
        item.setEnabled((project != null)
          && (project.getUserId().equals(User.getLoggedUser().getId())
          || User.getLoggedUser().isAdmin()));
      }
    });
  }

  /**
   * Consulta ao desktop criado na aplicao cliente.
   *
   * @return .
   */
  public static DesktopFrame getInstance() {
    return instance;
  }

  /**
   * Constri e apresenta os componentes da interface principal do cliente.
   */
  protected void showDesktop() {
    configureDesktop();

    // Exibio condicional ao parmetro.
    final Client client = Client.getInstance();
    final boolean visibleDesktop = client.shouldDesktopStartVisible();
    mainFrame.setVisible(visibleDesktop);
    if (!visibleDesktop) {
      displayTrayInfoMessage(client.getSystemName());
    }
    updateVisibilityTrayCheckMenuItem();
    EventLogProxy.addClientInformation(desktopQueue, "login");

    /* Abrindo o ltimo projeto aberto pelo usurio (testa preferncia) */
    openLastProject();

    // Faz disparo de aplicao sugerida (na URL, por exemplo) aps
    // tentar abrir um projeto (last project). Com isso, alguns aplicativos
    // que dependem de projeto podem ser abertos.
    runStartApplication(client);
  }

  /**
   * Se a preferncia <code>DesktopPref.OPEN_LAST_PROJECT</code> estiver com
   * valor true, abre automaticamente o ltimo projeto aberto pelo usurio. O
   * valor default  false.
   */
  private void openLastProject() {
    PreferenceCategory dp = getDesktopPreferences();
    if (dp.getPreferenceAsBoolean(DesktopPref.OPEN_LAST_PROJECT)) {
      Object lastProjectId = getLastOpenedProjectId();
      if (lastProjectId == null) {
        return;
      }
      if (!checkOpenableProject(lastProjectId)) {
        return;
      }
      openProject(lastProjectId);
    }
  }

  /**
   * @return O Id do ltimo projeto aberto pelo usurio. Retorna null, caso no
   * exista registro de projetos abertos.
   */
  private Object getLastOpenedProjectId() {
    Collection<ProjectBasicInfo> recentProjects = getRecentProjects();
    if (recentProjects.size() == 0) {
      return null;
    }
    ProjectBasicInfo lastProjectInfo =
      (ProjectBasicInfo) ((LinkedList<ProjectBasicInfo>) recentProjects).peek();
    if (lastProjectInfo == null) {
      return null;
    }
    return lastProjectInfo.getProjectId();
  }

  /**
   * Lanamento da aplicao, se definida pela URL de lanamento do cliente (ver
   * {@link Client#getStartApplicationId()})
   *
   * @param client cliente
   */
  private void runStartApplication(final Client client) {
    final String appId = client.getStartApplicationId();
    if (appId == null) {
      return;
    }
    final ApplicationManager appManager = ApplicationManager.getInstance();
    final ApplicationRegistry reg = appManager.getApplicationRegistry(appId);
    if (reg == null) {
      final String tag = "DesktopFrame.undefined.start.application.failure";
      final String fmt = LNG.get(tag);
      final String err = String.format(fmt, appId);
      StandardErrorDialogs.showErrorDialog(this.getDesktopFrame(), err);
      return;
    }

    final Locale locale = LNG.getLocale();
    try {
      final CommonClientProject project = getProject();
      if (reg.requireProject() && project == null) {
        final String tag = "DesktopFrame.project.start.application.failure";
        final String fmt = LNG.get(tag);
        final String err = String.format(fmt, reg.getApplicationName(locale));
        StandardErrorDialogs.showErrorDialog(this.getDesktopFrame(), err);
        return;
      }
      appManager.runApplication(appId);
    }
    catch (ApplicationException e) {
      final String fmt = LNG.get("DesktopFrame.start.application.failure");
      final String err = String.format(fmt, reg.getApplicationName(locale));
      StandardErrorDialogs.showErrorDialog(this.getDesktopFrame(), err, e);
    }
  }

  /**
   * Seta o tamanho da janela. Este mtodo deve ser sobrescrito em projetos com
   * requisitos especficos quanto ao tamanho da janela.
   *
   * @param dimension Dimenso da janela
   */
  private void setDesktopSize(Dimension dimension) {
    mainFrame.pack();
    mainFrame.setExtendedState(Frame.NORMAL);
    mainFrame.setSize(dimension);
    mainFrame.validate();
  }

  /**
   * Maximiza o desktop.
   */
  private void maximizeDesktop() {
    mainFrame.pack();
    mainFrame.setExtendedState(Frame.MAXIMIZED_BOTH);
    // usamos validate() para garantir que mudanas no tamanho (p.ex.
    // maximize) sejam consideradas antes do frame ser exibido
    mainFrame.validate();
  }

  /**
   * Obtm o projeto que est aberto na janela principal do desktop.
   *
   * @return O projeto corrente selecionado na janela principal do desktop.
   */
  public CommonClientProject getProject() {
    return this.projectTree == null ? null : this.projectTree.getProject();
  }

  /**
   * Obtm a rvore do projeto aberto.
   *
   * @return A rvore do projeto aberto.
   */
  public ProjectTree getTree() {
    return projectTree;
  }

  /**
   * Inicializa os observadores do usurio, dos projetos do usurio e dos
   * algoritmos
   *
   * @throws ClientException Caso ocorra falha na inicializao dos observadores
   */
  protected void startServicesObservers() throws ClientException {
    try {
      //      SchedulerObserver.start();
      //      CommandObserver.start();
      AdministrationObserver.start();
      ProjectAdminObserver.start();
      AlgorithmManagementObserver.start();
      if (ServerInfoManager.getInstance().isEnabled()) {
        ServerInfoManager.getInstance().start();
      }
    }
    catch (Exception e) {
      throw new ClientException(e);
    }
  }

  /**
   * Remove os observadores do usurio, dos projetos do usurio e dos algoritmos
   */
  protected void stopServicesObservers() {
    try {
      if (ServerInfoManager.getInstance().isEnabled()) {
        ServerInfoManager.getInstance().stop();
      }
    }
    catch (Exception e) {
      System.out.println("Falha ao remover os observadores: " + e.getMessage());
    }
  }

  /**
   * Cria a imagem configurada para o Desktop com base na configurao do
   * cliente.
   *
   * @return a imagem
   */
  protected Image getDesktopFrameIcon() {
    final String iconPath = configuration.getOptionalProperty("iconPath");
    if (iconPath == null) {
      return null;
    }
    final URL url = getClass().getResource(iconPath);
    final ImageIcon imageIcon = new ImageIcon(url);
    final Image image = imageIcon.getImage();
    return image;
  }

  /**
   * Constri a interface principal do cliente.
   *
   * @throws ClientException Caso ocorra falha na construo do cliente
   * @throws ConfigurationException Caso ocorra falha na obteno do gerente de
   * configurao.
   */
  public DesktopFrame() throws ClientException {
    if (instance != null) {
      throw new ClientException("Double Desktop!");
    }

    final Locale locale = LNG.getLocale();
    final RemoteTask<Void> initTask = new RemoteTask<Void>() {
      @Override
      protected void performTask() throws RemoteException {
        ApplicationManager.setInstance(locale);
      }
    };

    final DesktopComponentFrame dskFrame = this.getDesktopFrame();
    final String initMessage = LNG.get("DesktopFrame.init.message");
    final String initTitle = LNG.get("DesktopFrame.init.title");
    final boolean initExecuted =
      initTask.execute(dskFrame, initTitle, initMessage);
    if (!initExecuted) {
      final Exception excpt = initTask.getError();
      final String err = LNG.get("DesktopFrame.init.error");
      throw new ClientException(err, excpt);
    }

    // a instanciao do PreferenceManager s pode ser feita
    // aps a instanciao do ApplicationManager 
    PreferenceManager prefManager = PreferenceManager.getInstance();
    prefManager.loadDefinition(DesktopPref.class);

    try {
      final ConfigurationManager manager = ConfigurationManager.getInstance();
      this.configuration = manager.getConfiguration(this.getClass());
    }
    catch (ConfigurationManagerException e) {
      throw new ConfigurationException(e);
    }
    dependentWindows = new Vector<DesktopWindowInterface>();
    mainFrame = new DesktopComponentFrame();
    mainFrame.setPreferredSize(new Dimension(800, 600));
    mainFrame.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
    instance = this;
    startServicesObservers();

    // Coloca o cone do desktop
    final Image desktopImage = getDesktopFrameIcon();
    if (desktopImage != null) {
      setFrameImage(desktopImage);
    }

    trayIcon = mountTrayIcon();
  }

  /**
   * Monta o tray icon do desktop, conforme definio do sistema.
   *
   * @return tray icon
   * @throws ClientException em caso de erro.
   */
  private TrayIcon mountTrayIcon() throws ClientException {
    if (!SystemTray.isSupported()) {
      return null;
    }
    final String tag = "desktop.use.tray.icon";
    final Boolean useTray =
      configuration.getOptionalBooleanProperty(tag, false);
    final TrayIcon tr = useTray ? createTrayIcon() : null;

    if (tr == null) {
      return null;
    }

    final SystemTray tray = SystemTray.getSystemTray();
    if (tray == null) {
      return null;
    }
    try {
      tray.add(tr);
      return tr;
    }
    catch (AWTException e) {
      e.printStackTrace();
      return null;
    }
  }

  /**
   * Ajuste de visibilidade do desktop
   *
   * @param shown ajuste a ser feito.
   */
  final public void setDesktopVisible(final boolean shown) {
    final JFrame frm = DesktopFrame.this.getDesktopFrame();
    frm.setVisible(shown);
    if (eventHandler != null) {
      eventHandler.fireDesktopVisibilityInfo(shown);
    }
  }

  /**
   * Consulta  visibilidade do desktop
   *
   * @return indicativo
   */
  final public boolean isDesktopVisible() {
    final JFrame frm = DesktopFrame.this.getDesktopFrame();
    return frm.isVisible();
  }

  /**
   * Retorna: item de menu (tray) para visibilidade do desktop; conforme
   * atributo {@link #defaultVisibilityTrayCheckMenuItem} .
   *
   * @return o valor
   */
  protected final CheckboxMenuItem getDefaultVisibilityTrayCheckMenuItem() {
    return defaultVisibilityTrayCheckMenuItem;
  }

  /**
   * Retorna: item de menu (tray) para finalizao do desktop; conforme atributo
   * {@link #defaultExitTrayMenuItem}.
   *
   * @return o valor
   */
  protected final MenuItem getDefaultExitTrayMenuItem() {
    return defaultExitTrayMenuItem;
  }

  /**
   * Cria o tray default do sistema.
   *
   * @return o tray icon do desktop.
   */
  protected TrayIcon createTrayIcon() {
    final DesktopFrame desktop = this;
    defaultExitTrayMenuItem = new MenuItem();
    defaultExitTrayMenuItem.addActionListener(new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent ae) {
        desktop.shutdownDesktop();
      }
    });
    final String exitText = LNG.get("DesktopFrame.tray.exit.item");
    defaultExitTrayMenuItem.setLabel(exitText);

    defaultVisibilityTrayCheckMenuItem = new CheckboxMenuItem();
    defaultVisibilityTrayCheckMenuItem.addItemListener(new ItemListener() {
      @Override
      public void itemStateChanged(ItemEvent e) {
        final boolean visible = DesktopFrame.this.isDesktopVisible();
        desktop.setDesktopVisible(!visible);
      }
    });
    final String showText = LNG.get("DesktopFrame.tray.visible.item");
    defaultVisibilityTrayCheckMenuItem.setLabel(showText);

    final PopupMenu popup = new PopupMenu();
    popup.add(defaultVisibilityTrayCheckMenuItem);
    popup.addSeparator();
    popup.add(defaultExitTrayMenuItem);

    final String tooltip = getTitle();
    final Image desktopFrameIcon = getDesktopFrameIcon();
    final Image errImage = ApplicationImages.ICON_ERROR_16.getImage();
    final Image image =
      (desktopFrameIcon == null ? errImage : desktopFrameIcon);
    final TrayIcon tr = new TrayIcon(image, tooltip, popup);
    tr.setImageAutoSize(true);
    tr.setToolTip(LNG.get("systemName"));

    final JFrame frm = DesktopFrame.this.getDesktopFrame();
    frm.addComponentListener(new ComponentAdapter() {
      @Override
      public void componentHidden(ComponentEvent e) {
        updateVisibilityTrayCheckMenuItem();
      }

      @Override
      public void componentShown(ComponentEvent e) {
        updateVisibilityTrayCheckMenuItem();
      }
    });
    return tr;
  }

  /**
   * Atualiza o estado do item de menu de visibilidade padro (para tray icon
   * default).
   */
  private void updateVisibilityTrayCheckMenuItem() {
    final boolean visible = isDesktopVisible();
    if (defaultVisibilityTrayCheckMenuItem != null) {
      defaultVisibilityTrayCheckMenuItem.setState(visible);
    }
  }

  /**
   * {@inheritDoc} Cadastra o listener que observa a conexo do cliente com o
   * servidor no <code>ClientRemoteMonitor</code>.
   */
  @Override
  public void preInitialization(InitialContext initialContext) {
    connMonitorListener = new ClientRemoteMonitorListener(mainFrame);
    ClientServerManager.getInstance().addCommonListener(connMonitorListener);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void showUI() {
    this.showDesktop();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void postInitialization() {
    final Client client = Client.getInstance();
    if (client.isOpenBusNeeded()) {
      eventHandler = new DesktopFrameOpenBusEventHandler(this);
    }

    if (this.loadProject()) {
      this.executeAlgorithm();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public JFrame getView() {
    return this.getDesktopFrame();
  }

  /**
   * Executa um algoritmo que est definido na sesso do usurio.
   */
  @SuppressWarnings("unchecked")
  private void executeAlgorithm() {
    final ClientRemoteMonitor clientRemoteMonitor =
      ClientRemoteMonitor.getInstance();
    final Session session = clientRemoteMonitor.getSession();
    final String algorithmName =
      (String) session.getAttribute(ALGORITHM_NAME_ATTRIBUTE);
    if (algorithmName == null) {
      return;
    }

    Map<String, String> algorithmParameters = (Map<String, String>) session
      .getAttribute(ALGORITHM_PARAMETERS_ATTRIBUTE);
    try {
      final ApplicationManager applicationManager =
        ApplicationManager.getInstance();
      final ExecutorFrame executorFrame =
        (ExecutorFrame) applicationManager.runApplication("executor");
      executorFrame.showConfigurator(algorithmName, algorithmParameters);
    }
    catch (ApplicationException e) {
      StandardErrorDialogs.showErrorDialog(this.getDesktopFrame(), e);
    }
  }

  /**
   * Carrega um projeto que est definido na sesso do usurio.
   *
   * @return true, caso o projeto tenha sido carregado, ou false, caso
   * contrrio.
   */
  private boolean loadProject() {
    String projectName = (String) ClientRemoteMonitor.getInstance().getSession()
      .getAttribute(PROJECT_NAME_ATTRIBUTE);
    if (projectName == null) {
      return false;
    }
    this.openProject(projectName);
    return true;
  }

  /**
   * Verifica se projeto pode ser aberto (no est aguardando alocao de rea
   * solicitada).
   *
   * @param projectId id do projeto a ser verificado.
   * @return flag indicando se projeto pode ser aberto (no est aguardando
   * alocao de rea solicitada).
   */
  public boolean checkOpenableProject(final Object projectId) {
    Task<Boolean> task = new RemoteTask<Boolean>() {
      @Override
      protected void performTask() throws Exception {
        ProjectServiceInterface psi = ClientRemoteLocator.projectService;
        if (psi.existsProject(projectId)) {
          ProjectAdminInfo projectAdminInfo = psi.getProjectAdminInfo(projectId);
          if (projectAdminInfo == null) {
            setResult(true);
          }
          else {
            setResult(projectAdminInfo.isUnlockedWithAreaAllocated());
          }
        }
        else {
          setResult(false);
        }
      }

      @Override
      protected void handleError(Exception error) {
        error.printStackTrace();
      }
    };

    task.execute(getView(),
      LNG.get("DesktopFrame.checking.project.blocked.title"),
      LNG.get("DesktopFrame.checking.project.blocked.msg"));

    if (task.wasCancelled()) {
      StandardDialogs.showErrorDialog(getView(),
        LNG.get("DesktopFrame.cancelled.task.title"),
        LNG.get("DesktopFrame.cancelled.task.msg"));
      return false;
    }

    if (task.getStatus() != true) {
      StandardDialogs.showErrorDialog(getView(),
        LNG.get("DesktopFrame.locked.project.failure.title"),
        LNG.get("DesktopFrame.locked.project.failure.msg"));
      return false;
    }

    boolean openable = task.getResult();
    if (!openable) {
      StandardDialogs.showErrorDialog(getView(),
        LNG.get("DesktopFrame.locked.project.title"),
        LNG.get("DesktopFrame.locked.project.msg"));
    }
    return openable;
  }

  /**
   * Abre um projeto.
   *
   * @param projectId Identificador do projeto.
   * @return Objeto que representa o projeto aberto.
   */
  public CommonClientProject openProject(Object projectId) {

    CommonClientProject project = this.projectTree.getProject();
    if (project != null) {
      ProjectCloseAction closeAction = new ProjectCloseAction(this.projectTree);
      if (!closeAction.close()) {
        return null;
      }
    }

    final OpenProjectOnTreeRemoteTask openProjectTask =
      new OpenProjectOnTreeRemoteTask(projectId, this.projectTree);
    if (openProjectTask.execute(getDesktopFrame(),
      LNG.get("PRJ_WAITING_OPEN_PROJECT"),
      LNG.get("PRJ_WAITING_OPEN_PROJECT"))) {
      final CommonClientProject result = openProjectTask.getResult();
      return result;
    }
    return null;
  }

  /**
   * Obtem os <code>ProjectBasicInfo</code> de projetos que foram recetemente
   * abertos pelo usurio
   *
   * @return Coleo de <code>ProjectBasicInfo</code> de projetos que foram
   * recetemente abertos pelo usurio
   */
  public Collection<ProjectBasicInfo> getRecentProjects() {
    RemoteTask<Collection<ProjectBasicInfo>> task =
      new RemoteTask<Collection<ProjectBasicInfo>>() {
        @Override
        protected void performTask() throws Exception {
          final RecentProjectsManager manager = new RecentProjectsManager();
          Collection<ProjectBasicInfo> infos =
            manager.getProjectsInfosFromHistory();
          for (Iterator<ProjectBasicInfo> iterator = infos.iterator(); iterator
            .hasNext(); ) {
            ProjectBasicInfo info = iterator.next();
            String userLogin = User.getUser(info.getUserId()).getLogin();
            info.setUserLogin(userLogin);
          }
          setResult(infos);
        }
      };
    task.execute(mainFrame, getTitle(),
      LNG.get("DesktopFrame.task.get.projects.history.msg"));
    if (task.getStatus()) {
      return task.getResult();
    }
    return null;
  }

  /**
   * @return Coleo de aes para abrir um projeto especfico. Cada item se
   * refere a um projeto aberto recentemente pelo usurio. O nmero de
   * itens est limitado a MAX_RECENT_PROJECTS e no inclui o projeto
   * aberto no momento.
   */
  public AbstractAction[] getRecentProjectOpenActions() {

    Collection<ProjectBasicInfo> projectsInfos = getRecentProjects();
    ArrayList<AbstractAction> collection = new ArrayList<AbstractAction>();
    int i = 0;
    for (Iterator<ProjectBasicInfo> iterator =
         projectsInfos.iterator(); iterator.hasNext()
           && i < MAX_RECENT_PROJECTS; ) {
      ProjectBasicInfo projectBasicInfo = iterator.next();
      if (getProject() == null
        || !getProject().getId().equals(projectBasicInfo.getProjectId())) {
        OneProjectOpenAction action =
          new OneProjectOpenAction(projectBasicInfo);
        collection.add(action);
        i++;
      }
    }
    return collection.toArray(new AbstractAction[0]);
  }

  /**
   * Constri o item de menu para restaurar as configuraes do desktop.
   *
   * @return o item de menu para restaurar as configuraes do desktop
   */
  protected JMenuItem createRestoreDesktopItem() {
    final String restoreText = LNG.get("DesktopFrame.menu.restore.preferences");
    final JMenuItem restoreItem = new JMenuItem(restoreText);
    restoreItem.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        configureDesktop();
      }
    });
    return restoreItem;
  }

  /**
   * Constri o submenu para alterao de L&F do desktop.
   *
   * @return o item de menu para restaurar as configuraes do desktop
   */
  protected JMenu createLookAndFeelDesktopSubmenu() {
    final JMenu lafMenu = new JMenu();
    final UIManager.LookAndFeelInfo[] lafs =
      UIManager.getInstalledLookAndFeels();
    final LookAndFeel currLaF = UIManager.getLookAndFeel();
    final ButtonGroup grp = new ButtonGroup();
    for (final LookAndFeelInfo laf : lafs) {
      final AbstractAction action = new AbstractAction(laf.getName()) {
        @Override
        public void actionPerformed(ActionEvent e) {
          updateLaF(laf);
        }
      };
      JCheckBoxMenuItem item = new JCheckBoxMenuItem(action);
      lafMenu.add(item);
      grp.add(item);
      if (currLaF.getName().equals(laf.getName())) {
        item.setSelected(true);
      }
    }

    String lafText = LNG.get("DesktopFrame.submenu.laf.preferences");
    lafMenu.setText(lafText);
    return lafMenu;
  }

  /**
   * Atualizao de LaF (em teste)
   *
   * @param laf o look and feel desejado.
   */
  private void updateLaF(UIManager.LookAndFeelInfo laf) {
    final DesktopComponentFrame frame = getDesktopFrame();
    try {
      UIManager.setLookAndFeel(laf.getClassName());
      Container ancestor = frame.getRootPane().getTopLevelAncestor();
      SwingUtilities.updateComponentTreeUI(ancestor);
    }
    catch (Exception e) {
      final String err = LNG.get("DesktopFrame.laf.error.message");
      final String msg = err + "\n" + e.getMessage();
      StandardDialogs.showErrorDialog(frame, frame.getTitle(), msg);
    }
  }

  /**
   * Constri o item de menu para configurar a atualizao automtica do
   * projeto.
   *
   * @return item
   */
  protected JMenuItem createProjectAutoUpdateItem() {
    projectAutoUpdateItem =
      new JCheckBoxMenuItem(LNG.get("DesktopFrame.menu.project.preferences"),
        mustUpdateProjectTree());
    projectAutoUpdateItem.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        if (projectAutoUpdateItem.isSelected() && getProject() != null) {
          RemoteTask<Void> task = new RemoteTask<Void>() {
            @Override
            protected void performTask() throws Exception {
              DesktopFrame.this.getProject().refreshTree();
            }
          };
          task.execute(mainFrame, getTitle(),
            LNG.get("DesktopFrame.refresh.project.tree"));
        }
      }
    });
    projectAutoUpdateItem
      .setToolTipText(LNG.get("DesktopFrame.menu.project.preferences.tooltip"));
    return projectAutoUpdateItem;
  }

  /**
   * Constri o item de menu para configurar a exibio de um filtro dos ns da
   * rvore de projetos
   *
   * @return o tem de menu que permite habilitar ou desabilitar a exibio do
   * filtro
   */
  protected JMenuItem createShowTreeFilterItem() {
    /*
     * precisamos do lock esttico sobre a classe porque se neste ponto
     * showTreeFilter = null o synchronized() abaixo lanaria NPE
     */
    synchronized (DesktopFrame.class) {
      showTreeFilterItem =
        new JCheckBoxMenuItem(new ShowTreeFilterAction(treeFilterPanel));
      boolean isSelected =
        getDesktopPreferences().getPreferenceAsBoolean(DesktopPref.TREE_FILTER);
      showTreeFilterItem.setSelected(isSelected);
      treeFilterPanel.setVisible(isSelected);
    }
    treeFilterPanel.addListener(new TreeFilterPanelListener() {
      @Override
      public void visibilityChanged(boolean panelVisible) {
        synchronized (showTreeFilterItem) {
          showTreeFilterItem.setSelected(panelVisible);
        }
      }
    });
    return showTreeFilterItem;
  }

  /**
   * Cria o painel de notificaes.
   *
   * @return o painel de notificaes
   */
  protected JComponent createNotificationPanel() {
    try {
      NotificationProxy.resetObservers();
    }
    catch (Exception e) {
      StandardErrorDialogs.showErrorDialog(getDesktopFrame(), getTitle(), e);
    }
    this.notifPanel = new BasicNotificationPanel();
    notifPanel.setUser(User.getLoggedUser());
    //TODO Substituir os handlers por notificaes
    notifPanel.addNotificationHandler(CommandNotificationHandler.getInstance());
    notifPanel.addNotificationHandler(
      CommandPersistenceCommandNotificationHandler.getInstance());
    return notifPanel.getPanel();
  }

  /**
   * Cria o painel de detalhes dos arquivos de um diretrio.
   *
   * @return o painel de detalhes
   */
  protected JComponent createDirectoryContentsPanel() {
    dirContentsPanel = new DirectoryContentsPanel(projectTree);
    return dirContentsPanel;
  }

  /**
   * Cria o painel de filtro para a rvore de projetos.
   *
   * @return painel de filtro para a rvore de projetos.
   */
  protected JComponent createTreeFilterPanel() {
    treeFilterPanel =
      TreeFilterPanel.getInstance(projectTree, dirContentsPanel);
    treeFilterPanel.setVisible(false);
    return treeFilterPanel;
  }

  /**
   * Obtm a configurao da classe.
   *
   * @return a configurao da classe
   */
  protected final Configuration getConfiguration() {
    return this.configuration;
  }

  /**
   * Define se os arquivos ocultos devem ser exibidos.
   *
   * @param showHiddenFiles - true se os arquivos devem ser exibidos
   */
  public void setShowHiddenFiles(boolean showHiddenFiles) {
    this.showHiddenFiles = showHiddenFiles;
  }

  /**
   * @return true se os arquivos ocultos devem ser exibidos
   */
  public boolean shouldShowHiddenFiles() {
    if (showHiddenFiles == null) {
      showHiddenFiles = getDesktopPreferences()
        .getPreferenceAsBoolean(DesktopPref.SHOW_HIDDEN_FILES);
    }
    return showHiddenFiles;
  }

  /**
   * Mtodo que retorna o tray icon do sistema; se no tiver, retorna
   * {@code null} (sem tray). Ao redefinir o mtodo {@link #createTrayIcon()}, o
   * sistema indica se usar tray ou no (e como esse  montado).
   *
   * O desktop CSBASE verifica se o S.O. possui tal funcionalidade (mtodo
   * {@link SystemTray#isSupported()})
   *
   * @return o tray icon (ou {@code null}) se o desktop no tiver tray icon.
   */
  final public TrayIcon getTrayIcon() {
    return trayIcon;
  }

  /**
   * @param message mensagem
   */
  final public void displayTrayInfoMessage(final String message) {
    displayTrayMessage(message, MessageType.INFO);
  }

  /**
   * @param message mensagem
   */
  final public void displayTrayWarningMessage(final String message) {
    displayTrayMessage(message, MessageType.WARNING);
  }

  /**
   * @param message mensagem
   */
  final public void displayTrayErrorMessage(final String message) {
    displayTrayMessage(message, MessageType.ERROR);
  }

  /**
   * Exibe uma mensagem de tray
   *
   * @param message mensagem
   * @param type tipo
   */
  private void displayTrayMessage(final String message, MessageType type) {
    if (trayIcon == null) {
      return;
    }
    Client client = Client.getInstance();
    final String systemName = client.getSystemName();
    trayIcon.displayMessage(systemName, message, type);
  }

  /**
   * Abre o dilogo de criao de projeto.
   *
   * @param window Janela pai do dilogo.
   * @return O projeto criado.
   */
  public CommonClientProject openProjectCreationDialog(final Window window) {
    return ProjectBuildingDialog.createProject(window, null);
  }

  /**
   * Abre o dilogo de update de projeto.
   *
   * @param window Janela pai do dilogo.
   * @param project O projeto a ser alterado.
   * @return O projeto criado.
   */
  public CommonClientProject openProjectUpdateDialog(final Window window,
    final CommonClientProject project) {
    return ProjectBuildingDialog.updateProject(window, project, null);
  }

  /**
   * Consulta um id do desktop conforme {@link Client#getClientInstanceId()}.
   *
   * @return o id do desktop.
   */
  final public String getClientInstanceId() {
    final Client client = Client.getInstance();
    return client.getClientInstanceId();
  }

  /**
   * Consulta um id do dektop originrio da execuo deste cliente conforme
   * {@link Client#getFatherClientInstanceId()}.
   *
   * @return o id do desktop.
   */
  final public String getFatherClientInstanceId() {
    final Client client = Client.getInstance();
    return client.getFatherClientInstanceId();
  }

  /**
   * Ajusta o projeto corrente do desktop.
   *
   * @param prj projeto a ser ajustado como corrente
   */
  final public void setCurrentProject(CommonClientProject prj) {
    if (projectTree != null) {
      projectTree.setProject(prj);
    }
  }
}
