package csbase.client.util.csvpanel.table;

import java.util.Vector;

import javax.swing.event.TableModelEvent;
import javax.swing.event.TableModelListener;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableModel;
import javax.swing.table.TableModel;

/**
 * O modelo da tabela que apresenta o contedo do CSV.
 */
public class CSVTableModel extends AbstractTableModel implements
  TableModelListener {

  /** O modelo original da tabela. */
  private DefaultTableModel realModel;

  /** O modelo de mapeamento da tabela. */
  private TableModel wrapperModel;

  /**
   * Indica se a tabela dever aceitar edies.
   */
  private boolean editable;

  /**
   * Indica o nmero de linhas no incio do arquivo CSV que devem ser
   * consideradas como cabealho da tabela.
   */
  private int numRowsAsHeader;

  /**
   * Construtor.
   * 
   * @param data os dados da tabela.
   * @param editable Indica se a tabela dever aceitar edies.
   * @param numRowsAsHeader Indica o nmero de linhas no incio do arquivo CSV
   *        que devem ser consideradas como cabealho da tabela.
   */
  public CSVTableModel(String[][] data, boolean editable, int numRowsAsHeader) {
    this.numRowsAsHeader = numRowsAsHeader;
    this.editable = editable;
    this.realModel = new DefaultTableModel();
    this.realModel.addTableModelListener(this);
    updateWrapperModel();
    setData(data);
  }

  /**
   * Atualiza o modelo de acordo com as configuraes da tabela.
   */
  private void updateWrapperModel() {
    this.wrapperModel = realModel;
    if (numRowsAsHeader > 0) {
      this.wrapperModel =
        new MultiHeaderTableModel(wrapperModel, numRowsAsHeader);
    }
    fireTableStructureChanged();
  }

  /**
   * Obtm a tabela de dados do modelo.
   * 
   * @return a tabela de dados.
   */
  public String[][] getData() {
    return convertToMatrix(realModel.getDataVector());
  }

  /**
   * Atribui uma nova tabela de dados ao modelo.
   * 
   * @param tableData a nova tabela de dados.
   */
  protected void setData(String[][] tableData) {
    Object[] columns = null;
    if (tableData != null && tableData.length > 0) {
      columns = new Object[tableData[0].length];
    }
    realModel.setDataVector(tableData, columns);
    fireTableStructureChanged();
  }

  /**
   * Returns a vector that contains the same objects as the array.
   * 
   * @param vector o vector a ser converido
   * @return o array; se <code>vector</code> for <code>null</code>, retorna
   *         <code>null</code>
   */
  protected static String[] convertToArray(Vector<?> vector) {
    if (vector == null) {
      return null;
    }
    return vector.toArray(new String[vector.size()]);
  }

  /**
   * Returns a vector of vectors that contains the same objects as the array.
   * 
   * @param vector the double array to be converted
   * @return the new vector of vectors; if <code>anArray</code> is
   *         <code>null</code>, returns <code>null</code>
   */
  protected static String[][] convertToMatrix(Vector<?> vector) {
    if (vector == null) {
      return null;
    }
    String[][] array = new String[vector.size()][];
    for (int i = 0; i < array.length; i++) {
      array[i] = convertToArray((Vector<?>) vector.get(i));
    }
    return array;
  }

  /**
   * Determina se a tabela deve aceitar edies.
   * 
   * @param editable verdadeiro se a tabela deve aceitar edies ou falso, caso
   *        contrrio.
   */
  public void setEditable(boolean editable) {
    this.editable = editable;
  }

  /**
   * Indica se a tabela aceita edies.
   * 
   * @return verdadeiro se a tabela aceita edies ou falso, caso contrrio.
   */
  public boolean isEditable() {
    return editable;
  }

  /**
   * Determina o nmero de linhas no incio do arquivo CSV que devem ser
   * consideradas como cabealho da tabela.
   * 
   * @param numRowsAsHeader nmero de linhas que devem ser consideradas como
   *        cabealho. Se igual a zero, no usa nenhuma das linhas no cabealho.
   */
  public void setNumRowsAsHeader(int numRowsAsHeader) {
    if (this.numRowsAsHeader != numRowsAsHeader) {
      if (numRowsAsHeader >= 0 && numRowsAsHeader <= realModel.getRowCount()) {
        this.numRowsAsHeader = numRowsAsHeader;
      }
      else {
        this.numRowsAsHeader = 0;
      }
      updateWrapperModel();
    }
  }

  /**
   * Indica o nmero de linhas no incio do arquivo CSV que devem ser
   * consideradas como cabealho da tabela.
   * 
   * @return nmero de linhas que devem ser consideradas como cabealho. Se
   *         igual a zero, no usa nenhuma das linhas no cabealho.
   */
  public int getNumRowsAsHeader() {
    return numRowsAsHeader;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getRowCount() {
    return wrapperModel.getRowCount();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getColumnCount() {
    return wrapperModel.getColumnCount();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getColumnName(int columnIndex) {
    return wrapperModel.getColumnName(columnIndex);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Class<?> getColumnClass(int columnIndex) {
    return wrapperModel.getColumnClass(columnIndex);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Object getValueAt(int rowIndex, int columnIndex) {
    return wrapperModel.getValueAt(rowIndex, columnIndex);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setValueAt(Object aValue, int rowIndex, int columnIndex) {
    wrapperModel.setValueAt(aValue, rowIndex, columnIndex);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isCellEditable(int rowIndex, int columnIndex) {
    if (!editable) {
      return false;
    }
    return wrapperModel.isCellEditable(rowIndex, columnIndex);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void tableChanged(TableModelEvent e) {
    int firstRow = convertRealToWrapperModelIndex(e.getFirstRow());
    int lastRow = convertRealToWrapperModelIndex(e.getLastRow());
    TableModelEvent event =
      new TableModelEvent(this, firstRow, lastRow, e.getColumn(), e.getType());
    fireTableChanged(event);
  }

  /**
   * Adiciona uma coluna na tabela.
   * 
   * @param col ndice da coluna a adicionar.
   */
  public void addColumn(int col) {
    int index = col;
    Vector<?> dataVector = realModel.getDataVector();
    for (int i = 0; i < dataVector.size(); i++) {
      Vector<?> row = (Vector<?>) dataVector.get(i);
      row.add(index, null);
    }
    //Recarrega os dados para recriar a tabela.
    String[][] data = getData();
    setData(data);
  }

  /**
   * Remove uma coluna da tabela.
   * 
   * @param col ndice da coluna a remover.
   */
  public void removeColumn(int col) {
    int index = col;
    Vector<?> dataVector = realModel.getDataVector();
    for (int i = 0; i < dataVector.size(); i++) {
      Vector<?> row = (Vector<?>) dataVector.get(i);
      row.remove(index);
    }
    //Recarrega os dados para recriar a tabela.
    String[][] data = getData();
    setData(data);
  }

  /**
   * Remove uma linha da tabela.
   * 
   * @param row ndice da linha a remover.
   */
  public void removeRow(int row) {
    int index = convertWrapperToRealModelIndex(row);
    realModel.removeRow(index);
  }

  /**
   * Adiciona uma linha na tabela.
   * 
   * @param row ndice da linha a adicionar.
   */
  public void addRow(int row) {
    int index = convertWrapperToRealModelIndex(row);
    realModel.insertRow(index, (Object[]) null);
  }

  /**
   * Duplica uma linha da tabela.
   * 
   * @param row o ndice da linha.
   */
  public void duplicateRow(int row) {
    int index = convertWrapperToRealModelIndex(row);
    Vector<?> dataRow = (Vector<?>) realModel.getDataVector().get(index);
    realModel.insertRow(index, dataRow.toArray(new Object[dataRow.size()]));
  }

  /**
   * Converte o ndice da linha para o ndice no modelo real.
   * 
   * @param row o ndice da linha.
   * @return o ndice no modelo real.
   */
  private int convertWrapperToRealModelIndex(int row) {
    if (row >= 0) {
      return row + numRowsAsHeader;
    }
    else {
      return row;
    }
  }

  /**
   * Converte o ndice da linha para o ndice no modelo de mapeamento.
   * 
   * @param row o ndice da linha.
   * @return o ndice na viso.
   */
  private int convertRealToWrapperModelIndex(int row) {
    if (row >= 0) {
      return Math.max(row - numRowsAsHeader, 0);
    }
    else {
      return row;
    }
  }
}