package csbase.client.algorithms.parameters;

import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.rmi.RemoteException;
import java.util.LinkedList;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JTextField;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import csbase.client.applications.ApplicationImages;
import csbase.client.desktop.RemoteTask;
import csbase.client.util.filechooser.ClientSGAFileChooserUtil;
import csbase.logic.ClientSGAFile;
import csbase.logic.algorithms.parameters.ClientSGAFileParameter;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.SGAServiceInterface;

/**
 * View do parmetro que permite a escolha de um path em um SGA.
 * 
 * @author Tecgraf
 */
public class ClientSGAFileParameterView extends
  SimpleParameterView<ClientSGAFile> {

  /**
   * Construtor padro.
   * 
   * @param parameter - parmetro manipulado pela viso.
   * @param mode - Modo de visualizao. No aceita {@code null}, os possveis
   *        valores so: {@link ParameterView.Mode#CONFIGURATION} ou
   *        {@link ParameterView.Mode#REPORT}.
   */
  public ClientSGAFileParameterView(ClientSGAFileParameter parameter, Mode mode) {
    super(parameter, mode);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JComponent createConfigurationComponent(Object... componentArgs) {
    return new ClientSGAFileConfigurationComponent();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JComponent createReportComponent(Object... componentArgs) {
    return new ClientSGAFileReportComponent(
      (ClientSGAFileParameter) getParameter());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void updateViewContents() {
    if (getComponent() instanceof ClientSGAFileConfigurationComponent) {
      ((ClientSGAFileConfigurationComponent) getComponent()).updateView();
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean fillVerticalSpace() {
    return false;
  }

  /**
   * Retorna o nome do(s) arquivo(s) escolhido(s) pelo usurio.
   * 
   * @param file - arquivo(s) escolhido(s) pelo usurio.
   * @return nome dos arquivos formatado e separados por ','
   */
  public static String formatPathNames(ClientSGAFile file) {
    StringBuilder pathNames = new StringBuilder();

    pathNames.append(file.getName());

    if (!file.getBrothers().isEmpty()) {
      for (ClientSGAFile brother : file.getBrothers()) {
        pathNames.append(",");
        pathNames.append(brother.getName());
      }
    }
    return pathNames.toString();
  }

  /**
   * Retorna os paths do(s) arquivo(s) escolhido(s) pelo usurio.
   * 
   * @param file - arquivo(s) escolhido(s) pelo usurio.
   * @return paths dos arquivos formatado e separados por ','
   */
  public static String formatPaths(ClientSGAFile file) {
    StringBuilder pathNames = new StringBuilder();

    pathNames.append(file.getStringPath());

    if (!file.getBrothers().isEmpty()) {
      for (ClientSGAFile brother : file.getBrothers()) {
        pathNames.append(",");
        pathNames.append(brother.getStringPath());
      }
    }
    return pathNames.toString();
  }

  /**
   * Painel que permite a escolha de um SGA e os paths que este tem acesso.
   * 
   * @author Tecgraf
   */
  private class ClientSGAFileConfigurationComponent extends JPanel {

    /** Combo que exibe os SGAs disponveis. */
    private JComboBox sgaComboBox;

    /** Campo que exibe o arquivo escolhido. */
    private JTextField fileChoosed;

    /** Boto que dispara o chooser. */
    private JButton chooserButton;

    /** Construtor padro. */
    public ClientSGAFileConfigurationComponent() {
      super(new GridBagLayout());
      sgaComboBox = new JComboBox(getComboList().toArray());

      fileChoosed = new JTextField();
      fileChoosed.setEditable(false);

      chooserButton =
        GUIUtils.createImageButton(ApplicationImages.ICON_BROWSEFILE_16);
      chooserButton.setEnabled(false);

      addListeners();
      buildInterface();
    }

    /** Adiciona ouvintes nos componentes. */
    private void addListeners() {
      sgaComboBox.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          chooserButton.setEnabled(sgaComboBox.getSelectedIndex() != 0);
          setParameter(null);
        }
      });

      chooserButton.addActionListener(new ActionListener() {
        @Override
        public void actionPerformed(ActionEvent e) {
          String sgaName = (String) sgaComboBox.getSelectedItem();

          ClientSGAFileParameter parameter =
            (ClientSGAFileParameter) getParameter();

          ClientSGAFile root = new ClientSGAFile(sgaName, "/");
          root.setDir(true);

          ClientSGAFile choosed =
            getFilesSelected(parameter.isInputMode(), parameter.isShowFiles(),
              parameter.isMultipleSelection(), parameter.getTypes(), root);

          setParameter(choosed);
        }
      });
    }

    /**
     * Retorna o arquivo selecionado.
     * 
     * @param isInputMode - true se for modo de input, false para save.
     * @param onlyFiles - true se for pra permitir a seleo apenas de arquivos,
     *        false para diretrios.
     * @param isMultiple - true se o usurio pode escolher mltiplos arquivos,
     *        false caso contrrio.
     * @param types - tipos de arquivos usados no filtro.
     * @param root - diretrio raz.
     * 
     * @return arquivo selecionado.
     */
    private ClientSGAFile getFilesSelected(boolean isInputMode,
      boolean onlyFiles, boolean isMultiple, List<String> types,
      ClientSGAFile root) {

      if (onlyFiles) {
        if (isInputMode) {
          if (isMultiple) {
            List<ClientSGAFile> files =
              ClientSGAFileChooserUtil.browseMultipleFilesInOpenMode(null,
                types, getString("select.paths"), true, root);

            ClientSGAFile choosed = null;

            if (files != null) {
              for (ClientSGAFile f : files) {
                if (choosed == null) {
                  choosed = f;
                }
                else {
                  choosed.getBrothers().add(f);
                }
              }
            }
            return choosed;
          }
          else {
            return ClientSGAFileChooserUtil.browseSingleFileInOpenMode(null,
              types, getString("select.paths"), true, root);
          }
        }
        else {
          return ClientSGAFileChooserUtil.browseSingleFileInSaveMode(null,
            types, null, getString("select.paths"), true, root);
        }

      }
      else {
        if (isInputMode) {
          if (isMultiple) {
            List<ClientSGAFile> files =
              ClientSGAFileChooserUtil.browseMultipleDirectoriesInOpenMode(null,
                types, getString("select.paths"), true, root);

            ClientSGAFile choosed = null;

            if (files != null) {
              for (ClientSGAFile f : files) {
                if (choosed == null) {
                  choosed = f;
                }
                else {
                  choosed.getBrothers().add(f);
                }
              }
            }
            return choosed;
          }
          else {
            return ClientSGAFileChooserUtil.browseSingleDirectoryInOpenMode(
              null, types, getString("select.paths"), true, root);
          }
        }
        else {
          return ClientSGAFileChooserUtil.browseSingleDirectoryInSaveMode(null,
            types, null, getString("select.paths"), true, root);
        }
      }
    }

    /** Constroi a interface que permite a escolha de um arquivo em um SGA. */
    private void buildInterface() {
      add(sgaComboBox, new GBC(0, 0).right(5).horizontal());
      add(fileChoosed, new GBC(0, 1).top(5).right(4).horizontal());
      add(chooserButton, new GBC(1, 1).top(5).none());
    }

    /**
     * Lista com todos os nomes de SGA cadastrados no servidor.
     * 
     * @return lista com todos os nomes de SGA cadastrados no servidor.
     */
    private List<String> getComboList() {
      RemoteTask<List<String>> task = new RemoteTask<List<String>>() {
        @Override
        protected void performTask() throws Exception {
          SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
          List<String> sgaNames;
          try {
            sgaNames = sgaService.getAllSGANames();
          }
          catch (RemoteException e) {
            sgaNames = new LinkedList<String>();
          }
          setResult(sgaNames);
        }
      };

      boolean result =
        task.execute(null, getString("sga.names.task.title"),
          getString("sga.names.task.body"));

      List<String> comboList = new LinkedList<String>();
      comboList.add(getString("select.sga"));

      if (result) {
        comboList.addAll(task.getResult());
      }
      return comboList;
    }

    /**
     * Define o parmetro escolhido pelo usurio.
     * 
     * @param file - path escolhido pelo usurio.
     */
    private void setParameter(ClientSGAFile file) {
      getParameter().setValue(file);

      if (file == null) {
        fileChoosed.setText("");
      }
      else {
        fileChoosed.setText(formatPaths(file));
      }
    }

    /**
     * Retorna a mensagem internacionalizada.
     * 
     * @param key - chave da string.
     * @return mensagem internacionalizada.
     */
    private String getString(String key) {
      return LNG.get(getClass().getSimpleName() + "." + key);
    }

    /** Atualiza interface. */
    public void updateView() {
      ClientSGAFile value = getParameter().getValue();

      if (value == null) {
        return;
      }

      ActionListener[] actionListeners = sgaComboBox.getActionListeners();
      for (ActionListener l : actionListeners) {
        sgaComboBox.removeActionListener(l);
      }
      sgaComboBox.setSelectedItem(value.getSGAName());

      for (ActionListener l : actionListeners) {
        sgaComboBox.addActionListener(l);
      }

      fileChoosed.setText(formatPaths(value));
      chooserButton.setEnabled(sgaComboBox.getSelectedIndex() != 0);
    }
  }

  /**
   * Painel que exibe o parmetro {@link ClientSGAFileParameter} em modo report.
   * 
   * @author Tecgraf
   */
  private class ClientSGAFileReportComponent extends JPanel {

    /**
     * Construtor padro.
     * 
     * @param parameter - parmetro a ser exibido.
     */
    public ClientSGAFileReportComponent(ClientSGAFileParameter parameter) {
      super(new GridBagLayout());

      JTextField sga = new JTextField();
      sga.setEditable(false);

      JTextField path = new JTextField();
      path.setEditable(false);

      ClientSGAFile value = parameter.getValue();
      if (value != null) {
        sga.setText(value.getSGAName());
        path.setText(formatPaths(value));
      }

      add(sga, new GBC(0, 0).horizontal().top(5).bottom(5));
      add(path, new GBC(0, 1).horizontal().top(5).bottom(5));
    }
  }
}
