/**
 * $Id$
 */
package csbase.client.applications.diskinfo;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.GridBagLayout;
import java.awt.HeadlessException;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.text.DecimalFormat;
import java.util.Hashtable;

import javax.swing.AbstractAction;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.KeyStroke;
import javax.swing.table.DefaultTableCellRenderer;

import tecgraf.javautils.gui.GBC;
import csbase.client.applications.Application;

/**
 * Classe que modela uma janela com os dados de ocupao de disco por usurio em
 * forma tabular. As colunas so: id do usurio, espao ocupado em MB e espao
 * ocupado em percentual
 * 
 * @author Tecgraf
 */
public class DiskUserTableFrame extends JFrame {

  /**
   * Construtor
   * 
   * @param app referncia para aplicao (para obter rtulos)
   * @param usersData dados dos usurios: hashtable onde a chave  o id do
   *        usurio e o valor  quantidade em megabytes ocupado por ele
   * 
   * @throws HeadlessException
   */
  public DiskUserTableFrame(Application app, Hashtable<Object, Double> usersData)
    throws HeadlessException {
    super(app.getString("tableFrame.title"));
    setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
    // Cria o painel com os dados
    setContentPane(getTablePanel(app, usersData));
    // Fecha o frame ao clicar em ESC
    getRootPane().getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(
      KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0), "Cancel");
    getRootPane().getActionMap().put("Cancel", new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        close();
      }
    });
  }

  /**
   * Mostra a janela
   */
  public void showFrame() {
    setLocationRelativeTo(null);
    pack();
    setVisible(true);
  }

  /**
   * Retorna o painel com a tabela
   * 
   * @param app referncia para aplicao (para obter rtulos)
   * @param usersData dados dos usurios: hashtable onde a chave  o id do
   *        usurio e o valor  quantidade em megabytes ocupado por ele
   * 
   * @return painel com tabela
   * 
   */
  private JPanel getTablePanel(Application app,
    Hashtable<Object, Double> usersData) {
    JPanel tablePanel = new JPanel(new GridBagLayout());
    tablePanel.setOpaque(true);

    // rtulo para dar a dica de ordenao
    GBC gbc = new GBC(0, 0).northwest().insets(10, 10, 0, 10);
    JLabel orderHintLabel = new JLabel(app.getString("tableFrame.orderHint"));
    tablePanel.add(orderHintLabel, gbc);

    // Cria a tabela
    JTable table = new JTable(new DiskUserTableModel(app, usersData));
    table.setPreferredScrollableViewportSize(new Dimension(500, 200));
    table.setFillsViewportHeight(true);
    table.setAutoCreateRowSorter(true);
    /*
     * Alterna cores das linhas pra facilitar visualizao. Tem que fazer pra
     * cada coluna, porque se fizer pra tabela inteira no funciona
     */
    for (int i = 0; i < table.getModel().getColumnCount(); i++) {
      table.getColumn(table.getModel().getColumnName(i)).setCellRenderer(
        new DefaultTableCellRenderer() {
          @Override
          public Component getTableCellRendererComponent(JTable table,
            Object value, boolean isSelected, boolean hasFocus, int row,
            int column) {
            /*
             * A primeira coluna  string, as demais so double e precisam ser
             * formatadas de acordo
             */
            if (column > 0) {
              DecimalFormat formatter = new DecimalFormat("##0.##");
              value = formatter.format(value);
            }
            Component c =
              super.getTableCellRendererComponent(table, value, isSelected,
                hasFocus, row, column);
            // Se est selecionado, mantm o fundo, caso contrrio, colore
            if (isSelected) {
              c.setBackground(table.getSelectionBackground());
            }
            else {
              c.setBackground(row % 2 == 0 ? Color.white : Color.LIGHT_GRAY);
            }
            c.setForeground(table.getForeground());
            return c;
          };
        });
    }

    // Cria o scroll pane e coloca a tabela nele
    JScrollPane scrollPane = new JScrollPane(table);

    // Adiciona o scroll pane neste painel
    gbc = new GBC(0, 1).northwest().insets(10, 10, 0, 10).both();
    tablePanel.add(scrollPane, gbc);

    // Cria e adiciona o boto de fechar neste frame
    JButton closeButton = new JButton(app.getString("tableFrame.close"));
    closeButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        close();
      }
    });
    gbc = new GBC(0, 2).center().insets(10, 10, 10, 0);
    tablePanel.add(closeButton, gbc);

    return tablePanel;
  }

  /**
   * Fecha este frame
   */
  private void close() {
    dispose();
    setVisible(false);
  }
}