package csbase.client.applications.projectsmanager.dialogs;

import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.util.ArrayList;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;

import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.dialogs.core.ProjectsManagerDialog;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.panels.ProjectSharingInfoPanel;
import csbase.logic.UserProjectInfo;

/**
 * 
 * Dilogo utilizado para apresentar informaes referentes a execuo de aes
 * sobre um conjunto de projetos.
 * 
 * TODO - Colocar icone de erro quando nenhum projeto se aplica a ao.
 * 
 * @author jnlopes
 */
public class StatusDialog extends ProjectsManagerDialog {

  /**
   * Mensagem de concluso, perguntando que deciso o usurio quer tomar.
   */
  final private String decisionMessage;

  /**
   * A mensagem que descreve porque projetos foram rejeitados.
   */
  private String descriptionMessage;

  /**
   * Lista de propjetos a serem apresentados.
   */
  private List<ProjectsManagerData> projects;

  /**
   * Valor de retorno.
   */
  private int returnValue;

  /**
   * Insets genricos do dilogo.
   */
  final private Insets insets = new Insets(7, 11, 7, 12);

  /**
   * Insets do topo do dilogo.
   */
  final private Insets topInsets = new Insets(15, 11, 2, 12);

  /**
   * Construtor para a verso do dilogo onde alguns projetos foram rejeitados,
   * mas no todos.
   * 
   * @param projectsManager A aplicao
   * @param descriptionMessage A mensagem que descreve porque projetos foram
   *        rejeitados.
   * @param decisionMessage Mensagem de concluso, perguntando que deciso o
   *        usurio quer tomar.
   * @param projects A lista de projetos vlidos.
   */
  private StatusDialog(final ProjectsManager projectsManager,
    final String descriptionMessage, final List<ProjectsManagerData> projects,
    final String decisionMessage) {
    super(projectsManager);
    this.projects = projects;
    this.descriptionMessage = descriptionMessage;
    this.decisionMessage = decisionMessage;

    setTitle(getString("StatusDialog.invalid.projects.title"));
    setLayout(new GridBagLayout());
  }

  /**
   * Construtor para a verso do dilogo onde nenhum projeto foi rejeitado.
   * 
   * @param projectsManager A aplicao.
   * @param decisionMessage A mesagem de deciso sobre a execuo.
   */
  private StatusDialog(final ProjectsManager projectsManager,
    final String decisionMessage) {
    super(projectsManager);
    this.decisionMessage = decisionMessage;

    setModalityType(ModalityType.APPLICATION_MODAL);
    setTitle(getString("StatusDialog.projects.status.title"));
    setLayout(new GridBagLayout());
  }

  /**
   * Constri o painel que exibe a mensagem de aviso sobre os quesitos que um
   * projeto precisa atender para estar sujeito a esta ao.
   * 
   * @param error Indica se a mensagem deve ser de erro ou apenas um aviso.
   */
  private void buildProjectMessagePanel(final boolean error) {

    final JPanel descriptionPanel = new JPanel(new GridBagLayout());

    if (descriptionMessage.contains("<li>")) {
      final int idx = descriptionMessage.indexOf("<li>");
      String msg1 = descriptionMessage.substring(0, idx);
      String msg2 = descriptionMessage.substring(idx);

      msg1 = "<html>" + msg1.replace("\n", "<br>") + "</html>";
      msg2 = "<html>" + msg2.replace("\n", "<br>") + "</html>";

      final JLabel label1 = new JLabel(msg1);

      final ImageIcon icon =
        !error ? ApplicationImages.ICON_WARNING_24
          : ApplicationImages.ICON_ERROR_24;
      label1.setIcon(icon);

      final JLabel label2 = new JLabel(msg2);
      descriptionPanel.add(label1, new GBC(0, 0).west());
      descriptionPanel.add(label2, new GBC(0, 1).west());
    }
    else {
      final JLabel label =
        new JLabel("<html>" + descriptionMessage.replace("\n", "<br>")
          + "</html>");

      final ImageIcon icon =
        !error ? ApplicationImages.ICON_WARNING_24
          : ApplicationImages.ICON_ERROR_24;
      label.setIcon(icon);

      descriptionPanel.add(label, new GBC(0, 0).west());
    }

    add(descriptionPanel, new GBC(0, 0).insets(topInsets));
  }

  /**
   * Constri o painel que exibe a lista de projetos em questo que sero
   * alterados.
   * 
   * @param showOwnerId Boolean que especifica se o id do owner deve ser exibido
   *        juntamente com o nome do projeto.
   */
  private void buildProjectListPanel(final boolean showOwnerId) {

    final ProjectSharingInfoPanel infoPanel =
      new ProjectSharingInfoPanel(getProjectsManager(), projects, showOwnerId);
    infoPanel.setPreferredSize(new Dimension(330, 70));

    add(infoPanel, new GBC(0, 1).center());
  }

  /**
   * Constri o painel que apresenta as informaes conclusivas, para que o
   * usurio tome sua deciso.
   */
  private void buildDecisionPanel() {

    final JPanel confirmPanel = new JPanel(new GridBagLayout());
    final String labelText =
      "<html>" + decisionMessage.replace("\n", "<br>") + "</html>";
    final JLabel label = new JLabel(labelText);
    confirmPanel.add(label, new GBC(0, 0).west());
    add(confirmPanel, new GBC(0, 2).center().horizontal().insets(insets));
  }

  /**
   * Adiciona painel com boto "ok". Este painel  adicionado quando no h
   * nenhuma escolha a ser tomada pelo usurio.
   */
  private void addOkButtons() {
    final JButton button = new JButton(new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        StatusDialog.this.dispose();
      }
    });

    button.setText(getString("StatusDialog.ok.text"));

    final JPanel buttonPanel =
      GUIUtils.createBasicGridPanel(new JComponent[][] { { button } });
    add(buttonPanel, new GBC(0, 3).center());
  }

  /**
   * Adiciona painel com botes "yes" e "no". Este painel  adicionado quando h
   * uma escolha a ser tomada pelo usurio.
   */
  private void addYesNoButtons() {
    final JButton yesButton = new JButton(new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        returnValue = 1;
        StatusDialog.this.dispose();
      }
    });
    yesButton.setText(getString("StatusDialog.yes.text"));

    final JButton noButton = new JButton(new AbstractAction() {
      @Override
      public void actionPerformed(ActionEvent e) {
        returnValue = 0;
        StatusDialog.this.dispose();
      }
    });
    noButton.setText(getString("StatusDialog.no.text"));

    final JComponent[] buttons = { yesButton, noButton };
    GUIUtils.matchPreferredSizes(buttons);

    final JPanel buttonsPanel =
      new JPanel(new FlowLayout(FlowLayout.CENTER, 11, 11));
    buttonsPanel.add(buttons[0]);
    buttonsPanel.add(buttons[1]);

    add(buttonsPanel, new GBC(0, 3).horizontal());
  }

  /**
   * Mtodo que instancia dilogo que representa falha na validao do projeto
   * selecionado.
   * 
   * @param projectsManager A aplicao.
   * @param descriptionMessage A mensagem que descreve as condies necessrias
   *        para que um projeto possa ser processado pela determinada ao.
   * @param project Projetos rejeitado.
   * @param decisionMessage Mensagem de concluso, perguntando que deciso o
   *        usurio quer tomar.
   */
  public static void showNoneOkDialog(final ProjectsManager projectsManager,
    final String descriptionMessage, final ProjectsManagerData project,
    final String decisionMessage) {

    if (project == null) {
      showNoneOkDialog(projectsManager, descriptionMessage,
        (List<ProjectsManagerData>) null, decisionMessage);
    }

    final List<ProjectsManagerData> projects =
      new ArrayList<ProjectsManagerData>();
    projects.add(project);

    showNoneOkDialog(projectsManager, descriptionMessage, projects,
      decisionMessage);
  }

  /**
   * Mtodo que instancia dilogo que representa falha na validao de todos os
   * projetos selecionados.
   * 
   * @param projectsManager A aplicao.
   * @param descriptionMessage A mensagem que descreve as condies necessrias
   *        para que um projeto possa ser processado pela determinada ao.
   * @param projects Lista de projetos rejeitados (neste caso, todos os
   *        projetos).
   * @param decisionMessage Mensagem de concluso, perguntando que deciso o
   *        usurio quer tomar.
   */
  public static void showNoneOkDialog(final ProjectsManager projectsManager,
    final String descriptionMessage, final List<ProjectsManagerData> projects,
    final String decisionMessage) {

    final StatusDialog statusDialog =
      new StatusDialog(projectsManager, descriptionMessage, projects,
        decisionMessage);
    statusDialog.buildProjectMessagePanel(true);
    statusDialog.buildDecisionPanel();
    statusDialog.addOkButtons();

    statusDialog.pack();
    statusDialog.setVisible(true);
  }

  /**
   * Mtodo que instancia dilogo que representa falha na validao de alguns
   * projetos selecionados.
   * 
   * @param projectsManager A aplicao.
   * @param descriptionMessage A mensagem de aviso, que descreve as condies
   *        necessrias para que um projeto possa ser processado pela
   *        determinada ao.
   * @param projects Lista de projetos aceitos.
   * @param decisionMessage Mensagem de concluso, perguntando que deciso o
   *        usurio quer tomar.
   * @return Inteiro indicando a opo selecionada pelo usurio. 1) Sim - 2) No
   */
  public static int showSomeOkDialog(final ProjectsManager projectsManager,
    final String descriptionMessage, final List<ProjectsManagerData> projects,
    final String decisionMessage) {

    final StatusDialog statusDialog =
      new StatusDialog(projectsManager, descriptionMessage, projects,
        decisionMessage);
    statusDialog.buildProjectMessagePanel(false);
    statusDialog.buildProjectListPanel(true);
    statusDialog.buildDecisionPanel();
    statusDialog.addYesNoButtons();

    statusDialog.pack();
    statusDialog.setVisible(true);
    return statusDialog.getReturnValue();
  }

  /**
   * Mtodo que instancia dilogo que representa sucesso na validao de todos
   * os projetos selecionados.
   * 
   * @param projectsManager A aplicao.
   * @param decisionMessage Mensagem de concluso, perguntando que deciso o
   *        usurio quer tomar, em funo das consequncias que a ao ter.
   * @return Inteiro indicando a opo selecionada pelo usurio. 1) Sim - 2) No
   */
  public static int showAllOkDialog(final ProjectsManager projectsManager,
    final String decisionMessage) {

    final StatusDialog statusDialog =
      new StatusDialog(projectsManager, decisionMessage);
    statusDialog.buildDecisionPanel();
    statusDialog.addYesNoButtons();

    statusDialog.pack();
    statusDialog.setVisible(true);
    return statusDialog.getReturnValue();
  }

  /**
   * Mtodo que instancia dilogo que representa falha na validao de alguns
   * projetos selecionados.
   * 
   * @param projectsManager A aplicao.
   * @param failureMessage Mensagem que explica a falha.
   * @param failPrjs Lista de projetos que no puderam ser carregados.
   * @param conclusionMessage Mensagem de concluso.
   */
  public static void showLoadingFailureDialog(ProjectsManager projectsManager,
    String failureMessage, List<UserProjectInfo> failPrjs,
    String conclusionMessage) {
    List<ProjectsManagerData> failPrjList =
      new ArrayList<ProjectsManagerData>();
    for (UserProjectInfo prj : failPrjs) {
      failPrjList.add(new ProjectsManagerData(prj.getProjectId(), prj
        .getProjectName(), prj.getOwnerId()));
    }
    StatusDialog statusDialog =
      new StatusDialog(projectsManager, failureMessage, failPrjList,
        conclusionMessage);
    statusDialog.buildProjectMessagePanel(false);
    statusDialog.buildProjectListPanel(false);
    statusDialog.buildDecisionPanel();
    statusDialog.addOkButtons();
    statusDialog.pack();
    GUIUtils.centerOnScreen(statusDialog);
    statusDialog.setVisible(true);
  }

  /**
   * Obtm o valor de retorno.
   * 
   * @return Valor de retorno.
   */
  private int getReturnValue() {
    return returnValue;
  }

}
