package csbase.client.applications.algorithmsmanager.versiontree;

import java.awt.Component;
import java.awt.Rectangle;
import java.awt.Window;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.Enumeration;
import java.util.Vector;

import javax.swing.ImageIcon;
import javax.swing.JTree;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.TreePath;
import javax.swing.tree.TreeSelectionModel;

import csbase.client.applications.ApplicationImages;
import csbase.client.applications.algorithmsmanager.versiontree.datatransfer.VersionTreeTransferHandler;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmVersionInfo;

/**
 * Essa classe representa uma rvore com as verses do algoritmo. Cada n possui
 * um boto popup com as funcionalidades que existem em cada tipo de n ou
 * sub-n.
 *
 */
public final class VersionTree extends JTree {

  /** Referncia para a janela pai da rvore de verses. */
  private Window owner;

  /** Modo em que esta rvore est trabalhando. */
  private Mode mode;

  /** Informaes do algoritmo selecionado */
  private AlgorithmInfo algorithmInfo;

  /** Lista de verses do algoritmo selecionado */
  private Vector<AlgorithmVersionInfo> versions;

  /** Modelo da rvore de verses de um algoritmo */
  private VersionTreeModel model;

  /** Backup de ns expandidos da rvore. */
  private Enumeration<TreePath> expandedNodes;

  /** Backupo da poro da rvore visvel dentro do ScrollPane. */
  private Rectangle visibleRectangle;

  /**
   * Indica se o menu popup da rvore deve ser desabilitado ou no. Isso deve
   * ocorrer caso ns de tipos diferentes sejam selecionados na rvore.
   */
  private boolean disablePopupMenu = false;

  /**
   * Construtor.
   *
   * @param owner janela que criou a rvore.
   * @param algorithmInfo
   */
  public VersionTree(Window owner, AlgorithmInfo algorithmInfo) {
    super();
    this.owner = owner;
    this.mode = getDefaultMode();

    this.algorithmInfo = algorithmInfo;
    getSelectionModel().setSelectionMode(
      TreeSelectionModel.DISCONTIGUOUS_TREE_SELECTION);
    initializeTree(algorithmInfo);
    setTransferHandler(new VersionTreeTransferHandler());
    setDragEnabled(true);

    addTreeSelectionListener(new SelectionListener());
    addMouseListener(new MouseListener());
    setCellRenderer(new CellRenderer());
  }

  /**
   * Obtm o modo em que esta rvore est trabalhando.
   *
   * @return o modo em que esta rvore est trabalhando.
   */
  public Mode getMode() {
    return this.mode;
  }

  /**
   * Obtm a janela que criou a rvore.
   *
   * @return a janela que criou a rvore
   */
  public Window getOwner() {
    return owner;
  }

  /**
   * Verifica se esta rvore contm um determinado n.
   *
   * @param versionName nome do n procurado
   *
   * @return retorna true, se essa rvore contm o n, caso contrrio, retorna
   *         false
   */
  public boolean contains(String versionName) {
    if (null == this.model || null == this.model.getRoot()) {
      return false;
    }
    else {
      return 0 <= ((VersionTreeRootNode) this.model.getRoot()).getNodeIndex(
        versionName);
    }
  }

  /**
   * Cria um n raiz.
   *
   * @return o n raiz
   */
  private VersionTreeRootNode createRootNode() {
    VersionTreeRootNode versionRootNode = new VersionTreeRootNode(this,
      loadAlgorithmVersions());
    return versionRootNode;
  }

  /**
   * Carrega as verses do algoritmo.
   *
   * @return as verses do algoritmo
   */
  private Vector<AlgorithmVersionInfo> loadAlgorithmVersions() {
    if (algorithmInfo == null) {
      return null;
    }
    versions = algorithmInfo.getVersions();
    return versions;
  }

  /**
   * Obtm o n selecionado associado ao caminho da rvore especificado.
   *
   * @param path caminho da rvore
   *
   * @return o n selecionado
   */
  private AbstractVersionTreeNode getSelectedNode(TreePath path) {
    if (path != null) {
      Object tmp = path.getLastPathComponent();
      if (AbstractVersionTreeNode.class.isAssignableFrom(tmp.getClass())) {
        return (AbstractVersionTreeNode) tmp;
      }
    }
    return null;
  }

  /**
   * Obtm o modo de funcionamento da rvore.
   *
   * @return o modo de funcionamento da rvore.
   */
  public Mode getDefaultMode() {
    return Mode.CLASSIC;
  }

  /**
   * Gerencia a seleo mltipla de ns da rvore. <br>
   * Se a rvore j tiver n(s) selecionado(s), o novo n dever ser aceitos por
   * estes, caso contrrio sua seleo ser desfeita.
   */
  class SelectionListener implements TreeSelectionListener {
    /**
     * {@inheritDoc}
     */
    @Override
    public void valueChanged(TreeSelectionEvent ev) {
      JTree tree = (JTree) ev.getSource();

      /*
       * Se no tiver nenhum ou apenas um selecionado, ele dever continuar
       * assim.
       */
      if (tree.isSelectionEmpty() || tree.getSelectionCount() == 1) {
        disablePopupMenu = false;
        return;
      }

      /*
       * Define quem ser o n lider da seleo que ter a funo de validar os
       * demais. Se o antigo lder ainda estiver entre os selecionados, ele ser
       * utilizado para validar os novos ns, caso contrrio, o ltimo n
       * selecionado ir validar os demais.
       */
      AbstractVersionTreeNode leadNode = null;
      TreePath oldLeadPath = ev.getOldLeadSelectionPath();
      if (oldLeadPath != null && isPathSelected(oldLeadPath)) {
        leadNode = getSelectedNode(ev.getOldLeadSelectionPath());
      }
      else {
        leadNode = getSelectedNode(ev.getNewLeadSelectionPath());
      }
      if (leadNode == null) {
        return;
      }

      Class<? extends AbstractVersionTreeNode> leadNodeClass = leadNode
        .getClass();
      Class<? extends AbstractVersionTreeNode> selectedNodeClass = null;
      TreePath[] paths = ev.getPaths();
      for (TreePath aPath : paths) {
        AbstractVersionTreeNode aSelectedNode = getSelectedNode(aPath);
        if (aSelectedNode != null) {
          selectedNodeClass = aSelectedNode.getClass();
        }
        if (selectedNodeClass == null || !selectedNodeClass.equals(
          leadNodeClass)) {
          disablePopupMenu = true;
          return;
        }
      }
      disablePopupMenu = false;
    }
  }

  /**
   * Gerencia as requisies aos ns por menus pop-up.
   */
  class MouseListener extends MouseAdapter {
    /**
     * {@inheritDoc}
     */
    @Override
    public void mousePressed(MouseEvent ev) {
      showPopup(ev);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void mouseReleased(MouseEvent ev) {
      showPopup(ev);
    }

    /**
     * Caso o evento de mouse tenha sido uma requisio de pop-up, pede para o
     * n aonde ele ocorreu para que construa um menu pop-up e mostra esse menu
     * ao usurio.
     *
     * @param ev evento de mouse.
     */
    private void showPopup(MouseEvent ev) {
      if (!ev.isPopupTrigger() || disablePopupMenu) {
        return;
      }

      JTree tree = (JTree) ev.getSource();
      TreePath newPath = tree.getPathForLocation(ev.getX(), ev.getY());
      if (newPath != null) {

        // Pega o n lder selecionado.
        TreePath leadPath = getLeadSelectionPath();
        AbstractVersionTreeNode leadNode = getSelectedNode(leadPath);
        // Pega o n cujo pop-up foi requisitado.
        AbstractVersionTreeNode newNode = getSelectedNode(newPath);

        /*
         * Se j havia um n, se o n cujo pop-up foi requisitad aceita o n que
         * j havia e se o CTRL estava pressionado o n cujo pop-up foi
         * requisitad deve ser adicionado a lista de ns selecionados, caso o
         * contrrio, ele deve ser o nico n selecionado.
         */
        if (leadPath != null && newNode != null && newNode
          .allowMultipleSelection(leadNode)) {
          tree.addSelectionPath(newPath);
        }
        else {
          tree.setSelectionPath(newPath);
        }
        // Cria e executa o menu pop-up do n escolhido.
        Object selected = newPath.getLastPathComponent();
        if (selected instanceof AbstractVersionTreeNode) {
          AbstractVersionTreeNode node = (AbstractVersionTreeNode) selected;
          node.createPopupMenu().show(tree, ev.getX(), ev.getY());
        }
      }
    }
  }

  /**
   * Renderiza os ns da rvore de acordo com cones providos pelos mesmos.
   */
  class CellRenderer extends DefaultTreeCellRenderer {

    /**
     * {@inheritDoc}
     */
    @Override
    public Component getTreeCellRendererComponent(JTree tree, Object value,
      boolean selected, boolean expanded, boolean leaf, int row,
      boolean hasFocus) {

      super.getTreeCellRendererComponent(tree, value, selected, expanded, leaf,
        row, hasFocus);

      if (value instanceof AbstractVersionTreeNode) {
        AbstractVersionTreeNode node = (AbstractVersionTreeNode) value;
        ImageIcon icon = node.getImageIcon();
        this.setIcon(icon);
      }
      else {
        this.setIcon(ApplicationImages.ICON_FOLDER_16);
      }
      return this;
    }
  }

  /**
   * Adiciona um n de algoritmo ao modelo da rvore.
   *
   * @param newAlgorithmInfo informaes sobre o algoritmo cujo n ser criado.
   */
  public void replaceAlgoNode(AlgorithmInfo newAlgorithmInfo) {
    backupExpandedNodes();
    if (!algorithmInfo.equals(newAlgorithmInfo)) {
      return;
    }
    algorithmInfo = newAlgorithmInfo;
    updateVersionTreeModel(newAlgorithmInfo);
    restoreExpandedNodes();
  }

  /**
   * Guarda uma cpia do estado atual de expanso dos ns da rvore.
   */
  private void backupExpandedNodes() {
    TreePath rootPath = new TreePath(model.getRoot());
    expandedNodes = this.getExpandedDescendants(rootPath);
    visibleRectangle = this.getVisibleRect();
  }

  /**
   * Restaura o estado de expanso dos ns da rvore, modificado aps alguma
   * atualizao.
   */
  private void restoreExpandedNodes() {
    if (expandedNodes == null) {
      return;
    }
    while (expandedNodes.hasMoreElements()) {
      TreePath oldPath = expandedNodes.nextElement();
      TreePath newPath = getNewPath(oldPath);
      if (this.isCollapsed(newPath)) {
        this.expandPath(newPath);
      }
    }
    this.scrollRectToVisible(visibleRectangle);
  }

  /**
   * Obtm o novo caminho equivalente a um caminho antigo, aps a substituio
   * do modelo da rvore.
   *
   * @param oldPath caminho para um n no modelo antigo.
   *
   * @return novo caminho equivalente no novo modelo, ou null se o caminho no
   *         mais existir.
   */
  private TreePath getNewPath(TreePath oldPath) {
    Object[] path = oldPath.getPath();
    if (path.length == 1) {
      return new TreePath(model.getRoot());
    }
    DefaultMutableTreeNode node = (DefaultMutableTreeNode) model.getRoot();
    for (int i = 1; i < path.length; i++) {
      DefaultMutableTreeNode oldNode = (DefaultMutableTreeNode) path[i];
      Enumeration children = node.children();
      if (children.equals(DefaultMutableTreeNode.EMPTY_ENUMERATION)) {
        return null;
      }
      DefaultMutableTreeNode childNode = null;
      boolean found = false;
      while (children.hasMoreElements()) {
        childNode = (DefaultMutableTreeNode) children.nextElement();
        if (oldNode.getUserObject() instanceof AlgorithmVersionInfo && childNode
          .getUserObject() instanceof AlgorithmVersionInfo) {
          AlgorithmVersionInfo oldVersion = (AlgorithmVersionInfo) oldNode
            .getUserObject();
          AlgorithmVersionInfo childVersion = (AlgorithmVersionInfo) childNode
            .getUserObject();
          if (oldVersion.getId().equals(childVersion.getId())) {
            found = true;
            break;
          }
          continue;
        }
        if (childNode.toString().equals(path[i].toString())) {
          found = true;
          break;
        }
      }
      if (!found) {
        return null;
      }
      node = childNode;
    }
    return new TreePath(node.getPath());
  }

  /**
   * Atualiza o modelo da rvore de verses a partir das informaes do
   * algoritmo que foi modificado.
   *
   * @param modifiedAlgoInfo informaes do algoritmo modificado
   */
  private void updateVersionTreeModel(AlgorithmInfo modifiedAlgoInfo) {
    algorithmInfo = modifiedAlgoInfo;
    VersionTreeRootNode root = createRootNode();
    this.model = new VersionTreeModel(root);
    setModel(model); // gera o evento de mudana de n selecionado
  }

  /**
   * Seleciona uma determinada linha da rvore de dados.
   *
   * @param row linha a ser selecionada
   *
   */
  public void selectNode(int row) {
    setSelectionInterval(row, row);
  }

  /**
   * Inicializa a rvore de verses para o algoritmo especificado.
   *
   * @param algoInfo informaes do algoritmo selecionado
   */
  public void initializeTree(AlgorithmInfo algoInfo) {
    if (!algorithmInfo.equals(algoInfo) || model == null) {
      updateVersionTreeModel(algoInfo);
    }
  }
}
