package csbase.client.applications.flowapplication.graph.actions;

import java.awt.event.ActionEvent;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import javax.swing.ImageIcon;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.flowapplication.Workspace;
import csbase.client.applications.flowapplication.filters.AddGraphPopupActionFilter;
import csbase.client.applications.flowapplication.filters.WorkspaceFilter;
import csbase.client.applications.flowapplication.graph.Graph;
import csbase.client.applications.flowapplication.graph.GraphElement;
import csbase.client.applications.flowapplication.graph.GraphNode;
import csbase.logic.algorithms.flows.configurator.FlowAlgorithmConfigurator;

/**
 * Ao de menu popup que marca o n escolhido para que seja desviado durante a
 * execuo do fluxo. O desvio pode no ser possvel @see {@link GraphNode},
 * {@link FlowAlgorithmConfigurator}.
 * 
 * @author isabella
 */
public class BypassAction extends GraphAction {

  /**
   * Cria a ao.
   * 
   * @param graph Grafo cujo(s) n(s) deve(m) ser desviado(s)/removido(s) do
   *        desvio.
   */
  protected BypassAction(final Graph graph) {
    super(graph, getName(graph), getIcon(graph));
    setEnabled(graph.hasSelectedElements());
  }

  /**
   * Determina o texto que aparecer no menu popup, dependendo do estado atual
   * do n. Se o n j estiver desviado, mostra a opo de desfazer o desvio
   * caso contrrio, mostra a opo de fazer o desvio.
   * 
   * @param graph Grafo cujo(s) n(s) deve(m) ser desviado(s)/removido(s) do
   *        desvio.
   * @return o texto a ser mostrado no menu popup.
   */
  private static String getName(final Graph graph) {
    String suffix;
    if (areBeingBypassed(graph.getSelectedNodes())) {
      suffix = ".undo_bypass";
    }
    else {
      suffix = ".bypass";
    }
    return BypassAction.class.getName() + suffix;
  }

  /**
   * Determina o cone que aparecer no menu popup, dependendo do estado atual
   * do n. Se o n j estiver desviado, mostra o cone de desfazer o desvio
   * caso contrrio, mostra o cone de fazer o desvio.
   * 
   * @param graph Grafo cujo(s) n(s) deve(m) ser desviado(s)/removido(s) do
   *        desvio.
   * @return o cone a ser mostrado no menu popup.
   */
  private static ImageIcon getIcon(final Graph graph) {
    if (areBeingBypassed(graph.getSelectedNodes())) {
      return ApplicationImages.ICON_TURNON_16;
    }
    return ApplicationImages.ICON_TURNOFF_16;
  }

  /**
   * Determina se os ns de uma coleo esto sendo desviados ou no. S ser
   * retornado verdadeiro se todos os ns estiverem sendo desviados.
   * 
   * @param nodes Ns que devem ser testados.
   * @return verdadeiro se todos os ns estivem sendo desviados ou falso caso
   *         contrrio.
   */
  private static boolean areBeingBypassed(final Collection<GraphNode> nodes) {
    for (final GraphNode node : nodes) {
      if (!node.canBeBypassed() || !node.isBypassed()) {
        return false;
      }
    }
    return true;
  }

  /**
   * Determina se os ns de uma coleo podem ser desviados ou no. S ser
   * retornado verdadeiro se todos os ns puderem ser desviados.
   * 
   * @param nodes Ns que devem ser testados
   * @return verdadeiro se todos os ns puderem ser desviados ou falso caso
   *         contrrio
   */
  private static boolean canBeBypassed(final Collection<GraphNode> nodes) {
    for (final GraphNode node : nodes) {
      if (!node.canBeBypassed()) {
        return false;
      }
    }
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void actionPerformed(final ActionEvent event) {
    final Graph graph = getGraph();
    final Set<GraphNode> selectedElements = graph.getSelectedNodes();
    final boolean canBeBypassed = canBeBypassed(selectedElements);
    final boolean areBypassed = areBeingBypassed(selectedElements);
    if (canBeBypassed || areBypassed) {
      boolean bypassActionStatus = true;
      final Map<GraphNode, Boolean> memento = new HashMap<GraphNode, Boolean>();
      for (final GraphElement element : selectedElements) {
        final GraphNode node = (GraphNode) element;
        memento.put(node, node.isBypassed());
        bypassActionStatus = node.setBypassed(!areBypassed);
        if (!bypassActionStatus) {
          undoBypass(memento);
          break;
        }
      }
    }
    else {
      StandardDialogs
        .showErrorDialog(
          graph.getParentWindow(),
          LNG
            .get("csbase.client.applications.flowapplication.graph.actions.BypassAction.error_title"),
          LNG
            .get("csbase.client.applications.flowapplication.graph.actions.BypassAction.error_msg"));
    }
  }

  /**
   * Desfaz o desvio, utilizando
   * 
   * @param memento Map with previous bystate
   */
  private void undoBypass(final Map<GraphNode, Boolean> memento) {
    for (final GraphNode node : memento.keySet()) {
      node.setBypassed(memento.get(node));
    }
  }

  /**
   * Cria o filtro que cadastra a ao no menu popup.
   * 
   * @param workspace A rea de trabalho (No aceita {@code null}).
   * 
   * @return O filtro.
   */
  public static WorkspaceFilter createFilter(final Workspace workspace) {
    if (workspace == null) {
      throw new IllegalArgumentException("O parmetro workspace est nulo.");
    }
    return new AddGraphPopupActionFilter(workspace) {

      /**
       * {@inheritDoc}
       */
      @Override
      protected GraphAction createAction(final Graph graph) {
        if (!graph.getSelectedNodes().isEmpty()) {
          return new BypassAction(graph);
        }
        return null;
      }
    };
  }

}
