package csbase.client.applications.algorithmsmanager.dialogs;

import java.awt.GridBagLayout;
import java.awt.Window;
import java.text.MessageFormat;
import java.util.Hashtable;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.JPanel;

import tecgraf.javautils.gui.GBC;
import csbase.client.applications.algorithmsmanager.PropertiesListener;
import csbase.client.applications.algorithmsmanager.models.AlgorithmListItem;
import csbase.client.applications.algorithmsmanager.versiontree.PropertiesPanel;
import csbase.client.remote.srvproxies.AlgorithmManagementProxy;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.logic.algorithms.AlgorithmProperty;

/**
 * Painel com as informaes bsicas do algoritmo selecionado para edio.
 * 
 */
public class AlgorithmBasicInfoPanel extends CommonInfoEditPanel {
  /** Painel com as informaes estticas do algoritmo */
  private AlgorithmStaticInfoPanel algorithmHeaderPanel;

  /** Painel de propriedades de algoritmos */
  private PropertiesPanel propertiesPanel;

  /** Painel principal */
  private JPanel mainPanel;

  /** Indica que est sendo feita uma edio e os dados foram alterados */
  private boolean isEditing;

  /**
   * Constri o painel com as informaes bsicas do algoritmo.
   * 
   * @param algoPane painel com os dados de atualizao de um algoritmo
   */
  public AlgorithmBasicInfoPanel(CommonEditTabbedPanel algoPane) {
    super(algoPane);
  }

  /**
   * Obtm a janela da aplicao Gerenciador de Algoritmos.
   * 
   * @return a janela da aplcicao
   */
  private Window getWindow() {
    return getApplication().getApplicationFrame();
  }

  /**
   * Obtm o painel com os campos para alterao de uma categoria.
   * 
   * @return o painel criado
   */
  private JPanel getAlgoUpdatePanel() {
    JPanel updatePanel = new JPanel(new GridBagLayout());
    updatePanel.setBorder(BorderFactory.createEmptyBorder());

    List<AlgorithmProperty> algorithmProperties =
      AlgorithmManagementProxy.getAlgorithmProperties(getWindow());
    propertiesPanel =
      new PropertiesPanel(algorithmProperties, new Hashtable<String, String>(),
        getString("AlgorithmBasicInfoPanel.label.properties"), getWindow(),
        true);
    propertiesPanel.addPropertiesListener(new PropertiesListener() {

      @Override
      public void propertiesChanged(boolean wasModified) {
        isEditing = wasModified;
        changeOperationsState(wasModified);
      }
    });

    if (algorithmProperties.size() > 0) {
      updatePanel.add(propertiesPanel.getPropertiesPanel(), new GBC(0, 0)
        .horizontal().west().insets(0, 0, 0, 0));
    }
    updatePanel.add(buildEmptyPanel(), new GBC(0, 1).both().west().insets(0, 0,
      0, 0));
    return updatePanel;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void initializeData() {
    if (getSelectedAlgorithm() == null) {
      return;
    }
    if (!isEditing) {
      algorithmHeaderPanel.setSelectedAlgorithm(getSelectedAlgorithm());
      Hashtable<String, String> propertyValues =
        new Hashtable<String, String>();
      propertyValues = getSelectedAlgorithm().getItem().getPropertyValues();
      propertiesPanel.initializeFields(propertyValues);
      revalidate();
    }
    changeOperationsState(isEditing);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected JPanel buildMainInfoPanel() {
    if (mainPanel == null) {
      mainPanel = new JPanel(new GridBagLayout());

      algorithmHeaderPanel = new AlgorithmStaticInfoPanel(getApplication());
      JPanel algoPanel = getAlgoUpdatePanel();

      mainPanel.add(algorithmHeaderPanel, new GBC(0, 0).horizontal().west()
        .insets(0, 0, 0, 0));
      mainPanel.add(algoPanel, new GBC(0, 1).both().west().insets(0, 0, 0, 0));
    }
    return mainPanel;
  }

  /**
   * Salva efetivamente as informaes no servidor.
   * 
   * @return retorna true se as informaes foram salvas com sucesso, caso
   *         contrrio, retorna false
   */
  protected boolean saveInfo() {
    if (!propertiesPanel.validateProperties()) {
      return false;
    }
    analyzePropertyChange();
    return true;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void apply() {
    isEditing = false;
    if (!saveInfo()) {
      return;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected void cancel() {
    if (confirmCancelling()) {
      initializeData();
    }
  }

  /**
   * Confirma com o usurio se a operao de cancelamento deve ser efetivada, j
   * que perder todas as alteraes feitas sobre o dado.
   * 
   * @return retorna true, se a alterao do dado deve ser cancelada, caso
   *         contrrio, retorna false
   */
  private boolean confirmCancelling() {
    int confirm =
      getApplication().showOptionDialog(
        getString("AlgorithmBasicInfoPanel.msg.cancel.confirm"),
        new String[] {
            getString("AlgorithmBasicInfoPanel.msg.cancel.confirm.yes"),
            getString("AlgorithmBasicInfoPanel.msg.cancel.confirm.no") });
    if (confirm == 0) {
      isEditing = false;
      return true;
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void setDataChanged() {
    isEditing = false;
    changeOperationsState(isEditing);
  }

  /**
   * Analisa se ouve ou nao mudanca em alguma das propriedades. Em caso
   * positivo, executa a atualizao.
   */
  private void analyzePropertyChange() {
    if (getSelectedAlgorithm() == null) {
      return;
    }

    if (propertiesPanel.wasModified()) {
      String name = getSelectedAlgorithm().getName();
      AlgorithmInfo algorithm =
        AlgorithmManagementProxy.changeAlgorithmProperties(name,
          propertiesPanel.getProperties(), getWindow());
      String dialogTitle =
        getString("AlgorithmBasicInfoPanel.label.properties");

      if (algorithm == null) {
        String title =
          MessageFormat.format(
            getString("AlgorithmBasicInfoPanel.title.error"),
            new Object[] { dialogTitle });
        StandardErrorDialogs.showErrorDialog(getWindow(), title, MessageFormat
          .format(getString("AlgorithmBasicInfoPanel.error.update"),
            new Object[] { name }));
      }
      else {
        /*
         * Recarrega os algoritmos para que as alteraes se reflitam nos outros
         * usurios e no SchedulerService.
         */
        // TODO VERIFICAR SE PODE REMOVER
        // getApplication().reloadAlgorithmsFromServer();
      }
    }
  }

  /**
   * Obtm o algoritmo selecionado para edio.
   * 
   * @return o algoritmo selecionado para edio
   */
  private AlgorithmListItem getSelectedAlgorithm() {
    return getAlgorithmEditDataPanel().getSelectedAlgorithm();
  }

  /**
   * Obtm o painel global de edio de dados de algoritmo.
   * 
   * @return o painel global de edio de dados de algoritmo
   */
  private AlgorithmEditDataPanel getAlgorithmEditDataPanel() {
    return (AlgorithmEditDataPanel) getEditPanel();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean wasModified() {
    if (propertiesPanel.wasModified()) {
      return true;
    }
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected String getTitle() {
    return getApplication().getString("CommonEditTabbedPanel.tab.info.basic");
  }
}
