/*
 * $Author:$ $Date:$ $Release:$
 */
package csbase.client.algorithms;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Point;
import java.awt.Toolkit;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.text.MessageFormat;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import javax.swing.AbstractAction;
import javax.swing.ActionMap;
import javax.swing.BorderFactory;
import javax.swing.InputMap;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.KeyStroke;

import csbase.client.algorithms.parameters.ParameterView.Mode;
import csbase.client.algorithms.tasks.ViewValidationTask;
import csbase.client.algorithms.validation.ViewValidator;
import csbase.client.desktop.DesktopComponentDialog;
import csbase.client.desktop.DesktopFrame;
import csbase.client.project.ProjectTreePath;
import csbase.client.util.ClientUtilities;
import csbase.exception.ParseException;
import csbase.logic.CommonClientProject;
import csbase.logic.algorithms.AlgorithmConfigurator;
import csbase.logic.algorithms.AlgorithmConfiguratorListener;
import csbase.logic.algorithms.AlgorithmVersionInfo;
import csbase.logic.algorithms.validation.Validation;
import csbase.logic.algorithms.validation.ValidationMode;
import tecgraf.javautils.configurationmanager.Configuration;
import tecgraf.javautils.configurationmanager.ConfigurationManager;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;

/**
 * Viso do Configurador de Algoritmos.
 *
 * @author lmoreira
 */
public abstract class AlgorithmConfiguratorView implements ViewValidator {
  /**
   * Largura mnima.
   */
  private static final int MINIMUM_WIDTH = 600;
  /**
   * Altura mnima.
   */
  private static final int MINIMUM_HEIGHT = 100;

  /**
   * O Boto Cancelar.
   */
  private JButton cancelButton;
  /**
   * O Boto Fechar.
   */
  private JButton closeButton;
  /**
   * O Configurador Lgico.
   */
  private AlgorithmConfigurator configurator;
  /**
   * O Boto Confirmar.
   */
  private JButton confirmButton;
  /**
   * O caminho do arquivo de parmetro corrente.
   */
  private ProjectTreePath currentPath;
  /**
   * O dilogo.
   */
  private DesktopComponentDialog dialog;
  /**
   * O Boto Ajuda.
   */
  private JButton helpButton;
  /**
   * Os observadores.
   */
  private final List<AlgorithmConfiguratorViewListener> listeners;

  /**
   * A janela que  dona do dilogo.
   */
  private Window owner;
  /**
   * O <i>memento</i> dos valores dos parmetros.
   */
  private Map<Object, Object> previousValues;
  /**
   * O Boto Restaurar.
   */
  private JButton restoreButton;

  /**
   * O Boto Aplicar.
   */
  private JButton applyButton;

  /**
   * O modo de validao padro.
   */
  private ValidationMode defaultValidationMode;

  /**
   * Modo de visualizao do configurador. No aceita {@code null}, os possveis
   * valores so: {@link Mode#CONFIGURATION} e {@link Mode#REPORT}
   */
  private Mode mode;


  /**
   * Cria a viso.
   *
   * @param owner A janela que  dona deste dilogo (No aceita {@code null}).
   * @param configurator O configurador lgico (No aceita {@code null}).
   * @param mode Modo de visualizao do configurador. No aceita {@code null},
   * os possveis valores so: {@link Mode#CONFIGURATION} ou
   * {@link Mode#REPORT}
   * @param defaultValidationMode O modo de validao padro.
   */
  protected AlgorithmConfiguratorView(Window owner,
    AlgorithmConfigurator configurator, Mode mode,
    ValidationMode defaultValidationMode) {
    if (mode == null) {
      throw new IllegalArgumentException("O parmetro mode est nulo.");
    }
    if (defaultValidationMode == null) {
      throw new IllegalArgumentException(
        "O parmetro defaultValidationMode est nulo.");
    }
    this.mode = mode;
    this.defaultValidationMode = defaultValidationMode;
    this.listeners = new LinkedList<AlgorithmConfiguratorViewListener>();
    setConfigurator(configurator);
    setOwner(owner);
    createDialog();
    createButtons();
  }

  /**
   * Cria a viso com validao completa.
   *
   * @param owner A janela que  dona deste dilogo (No aceita {@code null}).
   * @param configurator O configurador lgico (No aceita {@code null}).
   * @param mode Modo de visualizao do configurador. No aceita {@code null},
   * os possveis valores so: {@link Mode#CONFIGURATION} ou
   * {@link Mode#REPORT}
   */
  protected AlgorithmConfiguratorView(Window owner,
    AlgorithmConfigurator configurator, Mode mode) {
    this(owner, configurator, mode, ValidationMode.FULL);
  }

  /**
   * Adiciona um observador.
   *
   * @param listener O observador (No aceita {@code null}).
   */
  public final void addAlgorithmConfiguratorViewListener(
    AlgorithmConfiguratorViewListener listener) {
    if (listener == null) {
      throw new IllegalArgumentException("O parmetro listener est nulo.");
    }
    this.listeners.add(listener);
  }

  /**
   * <p>
   * Obtm os observadores.
   * </p>
   *
   * <p>
   * A lista retornada  imutvel (veja
   * {@link Collections#unmodifiableList(List)}).
   * </p>
   *
   * @return Os observadores (Se no houver observadores, retorna uma lista
   * vazia).
   */
  public final List<AlgorithmConfiguratorViewListener>
  getAlgorithmConfiguratorViewListeners() {
    return Collections.unmodifiableList(this.listeners);
  }

  /**
   * Obtm o configurador lgico.
   *
   * @return .
   */
  public AlgorithmConfigurator getConfigurator() {
    return this.configurator;
  }

  /**
   * Obtm o caminho do arquivo de parmetro corrente.
   *
   * @return .
   */
  public final ProjectTreePath getCurrentPath() {
    return this.currentPath;
  }

  /**
   * @return Obtm o projeto corrente.
   */
  public final CommonClientProject getProject() {
    CommonClientProject currentProject =
      DesktopFrame.getInstance().getProject();
    return currentProject;
  }

  /**
   * Exibe o dalodo.
   */
  public final void launch() {
    this.previousValues = this.configurator.exportValues();
    populateDialog();
    this.dialog.center(this.owner);
    this.dialog.setVisible(true);
  }

  /**
   * Atribui o caminho do arquivo de parmetro corrente.
   *
   * @param currentPath O caminho do arquivo (Aceita {@code null}).
   */
  public void setCurrentPath(ProjectTreePath currentPath) {
    this.currentPath = currentPath;
  }

  /**
   * Habilita/Desabilita o configurador de algoritmos.
   *
   * @param isEnabled {@code true}, habilita; {@code false}, desabilita.
   * @return {@code true}, sucesso; {@code false}, caso no haja mudana de
   * estado.
   */
  public final boolean setEnabled(boolean isEnabled) {
    return configurator.setEnabled(isEnabled);
  }

  /**
   * @see java.lang.Object#toString()
   */
  @Override
  public String toString() {
    return getConfigurator().toString();
  }

  /**
   * Obtm o dilogo.
   *
   * @return O dilogo.
   */
  public final DesktopComponentDialog getDialog() {
    return this.dialog;
  }

  /**
   * Obtm o componente principal que  exibido no dilogo do configurador de
   * algoritmos.
   *
   * @return .
   */
  public abstract JComponent getMainComponent();

  /**
   * Atualiza viso do configurador.
   */
  public abstract void updateView();

  /**
   * Obtm a janela que  dona do dilogo deste configurador.
   *
   * @return .
   */
  protected final Window getOwner() {
    return this.owner;
  }

  /**
   * Reconecta os componentes ao dilogo.
   */
  protected final void repopulateDialog() {
    populateDialog();
    updateLocation();
  }

  /**
   * Cancela a edio dos parmetros.
   */
  private void cancel() {
    if (this.previousValues != null) {
      try {
        this.configurator.importValues(this.previousValues);
      }
      catch (ParseException e) {
        StandardDialogs
          .showErrorDialog(getDialog(), getString("title_wrong_parameters"),
            e.getLocalizedMessage());
      }
    }
    close();
    for (AlgorithmConfiguratorViewListener listener : this.listeners) {
      listener.wasCancelled(this);
    }
  }

  /**
   * Fecha o dilogo.
   */
  private void close() {
    this.previousValues = null;
    this.dialog.close();
  }

  /**
   * Confirmar a edio dos valores dos parmetros.
   */
  private void confirm() {
    Validation validation = ViewValidationTask
      .runTask(getDialog(), this, defaultValidationMode, true);
    if (validation.isWellSucceded()) {
      close();
      for (AlgorithmConfiguratorViewListener listener : this.listeners) {
        listener.wasConfirmed(this);
      }
    }
  }

  /**
   * aplicar a edio dos valores dos parmetros.
   */
  private void apply() {
    Validation validation = ViewValidationTask
      .runTask(getDialog(), this, ValidationMode.ALLOW_EMPY_VALUES, true);
    this.previousValues = this.configurator.exportValues();
    if (validation.isWellSucceded()) {
      for (AlgorithmConfiguratorViewListener listener : this.listeners) {
        listener.wasApplied(this);
      }
    }
  }

  /**
   * Adiciona key stroke no dilogo de configurao.
   *
   * @param keyStroke stroke (teclado)
   * @param action ao.
   */
  private void addStrokeAction(KeyStroke keyStroke, AbstractAction action) {
    JComponent component = this.dialog.getRootPane();
    int condition = JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT;
    InputMap inputMap = component.getInputMap(condition);
    ActionMap actionMap = component.getActionMap();

    String actionMapKey = keyStroke.toString();
    inputMap.put(keyStroke, actionMapKey);
    actionMap.put(actionMapKey, action);
  }

  /**
   * Cria os botes.
   */
  private void createButtons() {

    this.cancelButton = new JButton();
    AbstractAction cancelAction =
      new AbstractAction(getString("cancelButton")) {
        @Override
        public void actionPerformed(ActionEvent e) {
          cancelButton.requestFocusInWindow();
          cancel();
        }
      };
    addStrokeAction(KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0),
      cancelAction);

    this.cancelButton.setAction(cancelAction);


    this.closeButton = new JButton(getString("closeButton"));
    this.closeButton.addActionListener(e -> close());

    this.confirmButton = new JButton();
    AbstractAction enterAction =
      new AbstractAction(getString("confirmButton")) {
        @Override
        public void actionPerformed(ActionEvent e) {
          confirmButton.requestFocusInWindow();
          confirm();
        }
      };
    addStrokeAction(
      KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, KeyEvent.SHIFT_DOWN_MASK),
      enterAction);

    this.confirmButton.setAction(enterAction);

    this.applyButton = new JButton();
    AbstractAction applyAction = new AbstractAction(getString("applyButton")) {
      @Override
      public void actionPerformed(ActionEvent e) {
        applyButton.requestFocusInWindow();
        apply();
      }
    };
    this.applyButton.setAction(applyAction);

    AlgorithmVersionInfo algorithmVersionInfo =
      getConfigurator().getAlgorithmVersion();
    AlgorithmHelpAction algorithmHelpAction =
      getHelpAction(this.dialog, algorithmVersionInfo);
    algorithmHelpAction.hideIcon();
    this.helpButton = new JButton(algorithmHelpAction);
    addStrokeAction(KeyStroke.getKeyStroke(KeyEvent.VK_F1, 0),
      algorithmHelpAction);

    this.restoreButton = new JButton(getString("resetDefaultValues"));
    this.restoreButton.addActionListener(e -> getConfigurator().resetValues());
  }

  /**
   * Recupera a ao de ajuda para a view.
   *
   * @param dialog Componente dono da ao.
   * @param algorithmVersionInfo Verso do algoritmo para qual a ajuda ser
   * exibida (pode ser nulo).
   * @return a ao de ajuda para a view.
   */
  private AlgorithmHelpAction getHelpAction(DesktopComponentDialog dialog,
    AlgorithmVersionInfo algorithmVersionInfo) {
    if (algorithmVersionInfo == null) {
      return new AlgorithmHelpAction(dialog);
    }
    else {
      return new AlgorithmHelpAction(dialog, algorithmVersionInfo);
    }
  }

  /**
   * Cria o dilogo.
   */
  private void createDialog() {
    this.dialog = new DesktopComponentDialog(this.owner);
    this.dialog.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent e) {
        if (configurator.isEnabled()) {
          // Forar a atualizao de todos os parmetros para que aqueles
          // que estiverem com foco tambm sejam alterados antes do
          // fechamento da janela.
          updateView();
          cancel();
        }
        else {
          close();
        }
      }
    });
    this.dialog.setModal(false);
  }

  /**
   * Cria o painel externo de botes.
   *
   * @return O painel externo.
   */
  private JPanel createOuterButtonPanel() {
    FlowLayout flowLayout = new FlowLayout();
    JPanel outerButtonPanel = new JPanel(flowLayout);
    if (configurator.isEnabled()) {
      flowLayout.setAlignment(FlowLayout.CENTER);
      ClientUtilities.adjustEqualSizes(this.cancelButton, this.confirmButton,
        this.helpButton, this.applyButton);
      outerButtonPanel.add(this.confirmButton);
      if (defaultValidationMode == ValidationMode.ALLOW_EMPY_VALUES) {
        outerButtonPanel.add(this.applyButton);
      }
      outerButtonPanel.add(this.cancelButton);
      outerButtonPanel.add(this.helpButton);
    }
    else {
      flowLayout.setAlignment(FlowLayout.CENTER);
      outerButtonPanel.add(this.closeButton);
    }
    return outerButtonPanel;
  }

  /**
   * Obtm um texto no mecanismo de internacionalizao.
   *
   * @param key A chave para o texto (Ser prfixado o nome deste classe
   * seguido
   * de '.'; no aceita {@code null}).
   * @return O texto.
   */
  private String getString(String key) {
    return LNG.get(AlgorithmConfiguratorView.class.getName() + "." + key);
  }

  /**
   * Obtm um texto no mecanismo de internacionalizao.
   *
   * @param key A chave para o texto (Ser prfixado o nome deste classe
   * seguido
   * de '.'; no aceita {@code null}).
   * @param keyArgs Os argumentos para montar a mensagem.
   * @return O texto.
   */
  private String getString(String key, Object... keyArgs) {
    return MessageFormat.format(getString(key), keyArgs);
  }

  /**
   * Liga os componentes ao painel principal.
   */
  private void populateDialog() {
    JPanel contentPanel = new JPanel(new BorderLayout());
    final int N = 10;
    contentPanel.setBorder(BorderFactory.createEmptyBorder(N, N, N, N));

    JComponent mainComponent = getMainComponent();
    contentPanel.add(mainComponent, BorderLayout.CENTER);

    JPanel outerButtonPanel = createOuterButtonPanel();
    contentPanel.add(outerButtonPanel, BorderLayout.SOUTH);
    this.dialog.setContentPane(contentPanel);
    this.dialog.pack();
    updateSize();
    updateTitle();
  }

  /**
   * Atribui o configurador lgico a esta viso.
   *
   * @param configurator O configurador (No aceita {@code null}).
   */
  private void setConfigurator(AlgorithmConfigurator configurator) {
    if (configurator == null) {
      throw new IllegalArgumentException("O parmetro configurator est nulo.");
    }
    this.configurator = configurator;
    this.configurator
      .addAlgorithmConfiguratorListener(new AlgorithmConfiguratorListener() {
        @Override
        public void parameterLabelWasChanged(AlgorithmConfigurator cnf,
          String parameterName, String parameterLabel) {
          // Ignora o evento.
        }

        @Override
        public <V> void parameterValueWasChanged(AlgorithmConfigurator cnf,
          String parameterName, V parameterValue) {
          // Ignora o evento.
        }

        @Override
        public void parameterWasSetEnabled(AlgorithmConfigurator cnf,
          String parameterName, boolean parameterIsEnabled) {
          // Ignora o evento.
        }

        @Override
        public void parameterWasSetVisible(AlgorithmConfigurator cnf,
          String parameterName, boolean parameterIsVisible) {
          resizeOnVisibilityChanged();
        }

        @Override
        public void wasSetEnabled(AlgorithmConfigurator cnf) {
          repopulateDialog();
        }
      });
  }

  /**
   * Atribui a janela que  dona deste dilogo.
   *
   * @param owner A janela (No aceita {@code null}).
   */
  private void setOwner(Window owner) {
    if (owner == null) {
      throw new IllegalArgumentException("O parmetro owner est nulo.");
    }
    this.owner = owner;
  }

  /**
   * Ajusta a localizao do dilogo para que ele no passe dos limites do
   * monitor.
   */
  private void updateLocation() {
    Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
    Dimension currentSize = this.dialog.getSize();
    double usuableWidth = screenSize.getWidth();
    double currentWidth = currentSize.getWidth();
    double usuableHeight = screenSize.getHeight();
    double currentHeight = currentSize.getHeight();
    Point currentLocation = this.dialog.getLocation();
    double currentLeftmostX = currentLocation.getX();
    double availableWidth = usuableWidth - currentLeftmostX;
    double currentUpperY = currentLocation.getY();
    double availableHeight = usuableHeight - currentUpperY;
    double newLeftmostX;
    if (availableWidth < currentWidth) {
      newLeftmostX = currentLeftmostX - (currentWidth - availableWidth);
      newLeftmostX = Math.max(newLeftmostX, 0);
    }
    else {
      newLeftmostX = currentLeftmostX;
    }
    double newUpperY;
    if (availableHeight < currentHeight) {
      newUpperY = currentUpperY - (currentHeight - availableHeight);
      newUpperY = Math.max(newUpperY, 0);
    }
    else {
      newUpperY = currentUpperY;
    }
    Point newLocation = new Point();
    newLocation.setLocation(newLeftmostX, newUpperY);
    this.dialog.setLocation(newLocation);
  }

  /**
   * Faz o resize do dilogo do configurador caso este tenha algum widget que
   * sofreu mudana de visibilidade (e seu configurador de classes <b>esteja</b>
   * configurado para tal.
   *
   * @see ConfigurationManager
   */
  final public void resizeOnVisibilityChanged() {
    final ConfigurationManager cnfManager = ConfigurationManager.getInstance();
    if (cnfManager == null) {
      return;
    }
    try {
      final Class<AlgorithmConfiguratorView> propClass =
        AlgorithmConfiguratorView.class;
      final Configuration cnf = cnfManager.getConfiguration(propClass);
      final String propName = "resize.on.visibility.changed";
      if (cnf.getOptionalBooleanProperty(propName, false)) {
        updateSize();
      }
    }
    catch (Exception e) {
      // No faz nada mesmo, porque a opo default  no efetuar
      // o resize do dilogo.
    }
  }

  /**
   * Ajusta o tamanho do dilogo para que ele no passe dos limites do monitor.
   */
  private void updateSize() {
    /*
     * Existe o percentual mximo do tamanho de uma screen utilizvel pelo
     * dilogo de configurao. Isto para no permitir que toda a tela seja
     * usada. Tambm deixamos um offset para o tamanho final para alguma folga e
     * que evita o aparecimento da scrollbar horizontal.
     */
    final double factor = 0.85;
    final int offset = 35;

    final Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
    // dialog.pack();
    // final Dimension currentSize = this.dialog.getSize();
    final Dimension currentSize = dialog.getContentPane().getPreferredSize();
    double usuableWidth = screenSize.getWidth() * factor;
    double currentWidth = currentSize.getWidth();
    double newWidth = Math.min(currentWidth, usuableWidth);
    newWidth = Math.max(MINIMUM_WIDTH, newWidth);
    double usuableHeight = screenSize.getHeight() * factor;
    double currentHeight = currentSize.getHeight();
    double newHeight = Math.min(currentHeight, usuableHeight);
    newHeight = Math.max(MINIMUM_HEIGHT, newHeight);
    final Dimension newSize = new Dimension();
    newSize.setSize(newWidth + offset, newHeight + offset);
    this.dialog.setSize(newSize);
  }

  /**
   * Atualiza o ttulo do dilogo.
   */
  private void updateTitle() {
    String title = getString("title", this);
    this.dialog.setTitle(title);
  }

  /**
   * Retorna o modo de visualizao do configurador.
   *
   * @return o modo de visualizao do configurador.
   */
  public Mode getMode() {
    return mode;
  }
}
