package csbase.client.algorithms;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import csbase.client.remote.RecentAlgorithmsListener;
import csbase.logic.SharedObject;
import csbase.logic.User;
import csbase.logic.algorithms.AlgorithmInfo;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.SharedObjectServiceInterface;

/**
 * Gerencia a persistncia da lista de projetos abertos recentemente pelo
 * usurio. Os ltimos projetos so armazenados em uma fila.
 *
 * @author Tecgraf
 */
public class RecentAlgorithmsManager {

  /**
   * Nmero mximo de algoritmos armazendos na fila.
   */
  private static final int MAX_ALGORITHMS = 5;

  /**
   * Nome do objeto que armazena os projetos recentes como shared Object
   */
  private static final String RECENT_ALGORITHMS = "recent_algorithms";

  /**
   * Nome da categoria do shared object
   */
  private static final String RECENT_ALGORITHMS_CATEGORY = "RecentAgorithms";

  /**
   * Lista dos algoritmos recentes.
   */
  private LinkedList<AlgorithmInfo> recentLocalAlgoritms;

  /**
   * Booleano que indica se os dados de algoritmos recentes devem ser no
   * contexto da aplicao (False) ou no contexto da instncia do gerenciador
   * (True).
   */
  private boolean isLocal;

  /**
   * Listeners interessados em mudanas no conjunto de algoritmos recentes.
   */
  public static List<RecentAlgorithmsListener> managementList =
    new ArrayList<RecentAlgorithmsListener>();

  /**
   * Construtor.
   */
  public RecentAlgorithmsManager() {
    new RecentAlgorithmsManager(false);
  }

  /**
   * Construtor.
   * 
   * @param isLocal Booleano que indica se os dados de algoritmos recentes devem
   *        ser no contexto da aplicao (False) ou no contexto da instncia do
   *        gerenciador (True).
   */
  public RecentAlgorithmsManager(boolean isLocal) {
    recentLocalAlgoritms = new LinkedList<AlgorithmInfo>();
    this.isLocal = isLocal;
  }

  /**
   * Define se os dados de algoritmos recentes devem ser no contexto da
   * aplicao (True) ou no contexto da instncia do gerenciador (False).
   * 
   * @param isLocal indicativo.
   */
  public void setLocal(boolean isLocal) {
    this.isLocal = isLocal;
    notifyAlgorithmUpdated();
  }

  /**
   * Retorna se o gerenciamento de algoritmos recentes est sendo feito por
   * escopo da instncia deste objeto (True) ou por escopo da aplicao (False).
   * 
   * @return o indicativo.
   */
  public boolean isLocal() {
    return isLocal;
  }

  /**
   * L a lista de algoritmos abertos recentemente no contexto da aplicao.
   *
   * @return o {@link SharedObject} que armazena a lista de algoritmos abertos
   *         recentemente pelo usurio
   * @throws RemoteException em caso de falha na comunicao com o servidor.
   */
  private SharedObject readSharedObject() throws RemoteException {
    SharedObjectServiceInterface sharedObjectService =
      ClientRemoteLocator.sharedObjectService;
    try {
      SharedObject sharedObject = sharedObjectService.getSharedObject(
        RECENT_ALGORITHMS_CATEGORY, User.getLoggedUser().getId(),
        RECENT_ALGORITHMS);
      return sharedObject;
    }
    catch (Exception e) {
      return null;
    }
  }

  /**
   * Obtm as informaes de algoritmos recentes abertos a partir do
   * {@link SharedObject}.
   *
   * @param so Shared Object usado para persistir os algoritmos recentes.
   * @return Coleo com os algoritmos abertos recentemente. Os ltimos
   *         utilizados esto nas primeiras posies.
   */
  @SuppressWarnings("unchecked")
  private LinkedList<AlgorithmInfo> getAlgorithmInfosFromSharedObject(
    SharedObject so) {
    LinkedList<AlgorithmInfo> algorithmHistoryInfos =
      new LinkedList<AlgorithmInfo>();
    if (so == null || so.getContents() == null) {
      return algorithmHistoryInfos;
    }
    algorithmHistoryInfos.addAll((Collection<AlgorithmInfo>) so.getContents());
    return algorithmHistoryInfos;
  }

  /**
   * Obtm a coleo de projetos abertos recentemente. Os ltimos utilizados
   * esto nas primeiras posies.
   *
   * @return A coleo de dados de projeto armazenados.
   * @throws RemoteException em caso de erro na leitra dos dados.
   */
  public LinkedList<AlgorithmInfo> getAlgorithmInfosFromHistory()
    throws RemoteException {
    SharedObject so = readSharedObject();
    return getAlgorithmInfosFromSharedObject(so);
  }

  /**
   * Obtm as informaes de algoritmos recentes utilizados.
   *
   * @return Coleo com dados de algoritmos abertos recentemente. Os ltimos
   *         utilizados esto nas primeiras posies.
   * @throws RemoteException em caso de erro na leitra dos dados.
   */
  public List<AlgorithmInfo> getRecentAlgoritmsInfos() throws RemoteException {
    return Collections.unmodifiableList(getAlgoritmsInfos());
  }

  /**
   * Obtm as informaes de algoritmos recentes utilizados, de acordo com o
   * contexto de gerenciamento definido.
   * 
   * @return Coleo com dados de algoritmos abertos recentemente. Os ltimos
   *         utilizados esto nas primeiras posies.
   * @throws RemoteException
   */
  private LinkedList<AlgorithmInfo> getAlgoritmsInfos() throws RemoteException {
    if (!isLocal) {
      return getAlgorithmInfosFromHistory();
    }
    return recentLocalAlgoritms;
  }

  /**
   * Salva dados de algoritmos recentes para que possam ser recuperados
   * futuramente.
   *
   * @param algorithmInfo dados de um algoritmo.
   * @throws RemoteException
   */
  public void saveAlgorithmInfo(AlgorithmInfo algorithmInfo)
    throws RemoteException {

    saveGlobalRecents(algorithmInfo);
    saveLocalRecents(algorithmInfo);

    notifyAlgorithmUpdated();
  }

  /**
   * Salva um algoritmo na lista global (do SharedObject) de recentes.
   * 
   * @param algorithmInfo algoritmo a ser salvo.
   * @throws RemoteException em caso de erro na leitra dos dados.
   */
  private void saveGlobalRecents(AlgorithmInfo algorithmInfo)
    throws RemoteException {
    LinkedList<AlgorithmInfo> recentAlgoritms = new LinkedList<AlgorithmInfo>();
    recentAlgoritms = getAlgorithmInfosFromHistory();

    saveRecents(algorithmInfo, recentAlgoritms);

    SharedObject sharedObj = readSharedObject();
    if (sharedObj == null) {
      sharedObj = createSharedObject(recentAlgoritms);
    }
    else {
      sharedObj.setContents(recentAlgoritms);
    }
    SharedObjectServiceInterface sharedObjectService =
      ClientRemoteLocator.sharedObjectService;
    sharedObjectService.saveSharedObject(sharedObj);
  }

  /**
   * Salva um algoritmo na lista local de recentes.
   * 
   * @param algorithmInfo algoritmo a ser salvo.
   */
  private void saveLocalRecents(AlgorithmInfo algorithmInfo) {
    LinkedList<AlgorithmInfo> recentAlgoritms = new LinkedList<AlgorithmInfo>();
    recentAlgoritms = recentLocalAlgoritms;

    saveRecents(algorithmInfo, recentAlgoritms);
  }

  /**
   * Salva um algoritmo numa lista de algoritmos recentes recebida como
   * parmetro.
   * 
   * @param algorithmInfo algoritmo a ser salvo.
   * @param recentAlgoritms lista em que o algoritmo ser salvo.
   */
  private void saveRecents(AlgorithmInfo algorithmInfo,
    LinkedList<AlgorithmInfo> recentAlgoritms) {
    if (recentAlgoritms.contains(algorithmInfo)) {
      recentAlgoritms.remove(algorithmInfo);
    }
    /* Se j estiver com o nmero limite, exclui o ltimo */
    if (recentAlgoritms.size() == MAX_ALGORITHMS) {
      recentAlgoritms.removeLast();
    }
    /* Inclui o item atual no incio da fila */
    recentAlgoritms.addFirst(algorithmInfo);
  }

  /**
   * Cria um shared object com o contedo indicado
   *
   * @param contents o contedo
   * @return um novo shared object com o contedo indicado
   */
  private SharedObject createSharedObject(Object contents) {
    SharedObject obj = new SharedObject(RECENT_ALGORITHMS_CATEGORY, User
      .getLoggedUser().getId(), RECENT_ALGORITHMS, true, contents);
    return obj;
  }

  /**
   * Adiciona um listener para mudanas ocorridas na lista de algoritmos
   * recentes.
   *
   * @param listener listener a ser adicionado
   */
  public void addRecentAlgorithmsListener(RecentAlgorithmsListener listener) {
    managementList.add(listener);
  }

  /**
   * Notifica os listeners da aplicao que a lista de algoritmos recentes foi
   * alterada.
   */
  private void notifyAlgorithmUpdated() {
    for (RecentAlgorithmsListener listener : managementList) {
      listener.recentAlgorithmsUpdated();
    }
  }
}
