package csbase.client.applications.algorithmsmanager.report.core;

import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.stream.Collectors;

import javax.swing.BorderFactory;
import javax.swing.ButtonGroup;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JScrollPane;
import javax.swing.JTextPane;
import javax.swing.border.TitledBorder;

import tecgraf.javautils.gui.GBC;

/**
 * Paienl da rea de resultados
 *
 *
 * @author Tecgraf/PUC-Rio
 */
public class ReportPanel extends JPanel {

  // ORDENES
  /** Chave de ordenao por teste */
  private String TEST = "T";
  /** Chave de ordenao por status */
  private String STATUS = "S";
  /** Chave de ordenao por assunto */
  private String SUBJECT = "D";

  /**
   * Classe que armazena os resultados sendo mostrados
   *
   * @author Tecgraf/PUC-Rio
   * @param <T> Tipo do assunto testado
   */
  class ResultInfo<T> {
    /** Teste */
    protected ITest<T> test;

    /** Resultado */

    protected IResultMessage<T> message;

    /**
     * Retorna o nome do assunto da mensagem
     *
     * @return O nome do assunto
     */
    String getSubjectName() {
      return test.getSubjectName(message.getSubject());
    }
  }

  /** CheckBox para mostrar resultados OK */
  private JCheckBox okFilter;
  /** CheckBox para mostrar resultados AVISO */
  private JCheckBox warningFilter;
  /** CheckBox para mostrar resultados ERRO */
  private JCheckBox errorFilter;

  /** Boto para escolher texto colapsado */
  private JRadioButton colapsedButton;
  /** Boto para escolher texto expandido */
  private JRadioButton extendedButton;

  /** Combo para escolher 1o critrio de ordenao */
  private JComboBox<String> firstSorter;

  /** Combo para escolher 2o critrio de ordenao */
  private JComboBox<String> secondSorter;

  /** rea com o contedo */
  private JTextPane textArea;

  /** Os testes e seus resultados */
  private List<ResultInfo<?>> messages;

  /** Bundle de internacionalizao */
  ResourceBundle i18n;

  /**
   * Construtor
   *
   * @param locale Locale para os textos.
   */
  public ReportPanel(Locale locale) {
    i18n = createBundle(locale);
    messages = new ArrayList<>();
    mountPanel();
  }

  /**
   * Atualiza a rea de resultados
   *
   * @param map Mapa de resultados
   */
  public void updateContent(Map<ITest<?>, ITestResult<?>> map) {
    messages = convertResult(map);
    updateTextArea();

  }

  /**
   * Atualiza o contedo
   */
  private void updateTextArea() {

    List<ResultInfo<?>> messageSubList =
      filterData(this.messages, okFilter.isSelected(), warningFilter
        .isSelected(), errorFilter.isSelected());
    messageSubList =
      sortData(messageSubList, (String) firstSorter.getSelectedItem(),
        (String) secondSorter.getSelectedItem());

    if (colapsedButton.isSelected()) {
      printColapsed(messageSubList, (String) firstSorter.getSelectedItem(),
        (String) secondSorter.getSelectedItem());
    }
    else {
      printExpanded(messageSubList, (String) firstSorter.getSelectedItem(),
        (String) secondSorter.getSelectedItem());
    }
  }

  /**
   * Imprime as mensagens repetindo teste, assunto e statusa todas as vezes
   *
   * @param messageSubList Lista de mensagens filtradas e ordenadas
   * @param firstOrder Primeiros critrio de ordenao
   * @param secondOrder Segundo critrio de ordenao
   */
  private void printExpanded(List<ResultInfo<?>> messageSubList,
    String firstOrder, String secondOrder) {

    String thirdOrder = computeThird(firstOrder, secondOrder);
    StringBuilder builder = new StringBuilder("<html>");

    for (ResultInfo<?> info : messageSubList) {

      Map<String, String> map = new HashMap<>();
      map.put(TEST, "<i>" + info.test.getName() + "</i>");
      map.put(SUBJECT, info.getSubjectName());
      TestStatus status = info.message.getStatus();
      map.put(STATUS, getFontColor(status) + getString(status.toString())
        + "</font>");

      builder.append("<strong>");
      builder.append(map.get(firstOrder));
      builder.append(" - ");
      builder.append(map.get(secondOrder));
      builder.append(" - ");
      builder.append(map.get(thirdOrder));
      builder.append("</strong>");
      builder.append(" - ").append(info.message.getMessage());
      builder.append("<br><br>");
    }

    builder.append("</html>");

    textArea.setText(builder.toString());
    textArea.setCaretPosition(0);
  }

  /**
   * Imprime as mensagens hierarquicamnete. I.e. sem repetir os as informaes
   * que se repetem em mensagens consecutivas.
   *
   * @param messageSubList Lista de mensagens filtradas e ordenadas
   * @param firstOrder Primeiro critrio de ordenao
   * @param secondOrder Segundo critrio de ordenao
   */
  private void printColapsed(List<ResultInfo<?>> messageSubList,
    String firstOrder, String secondOrder) {

    String thirdOrder = computeThird(firstOrder, secondOrder);

    StringBuilder builder = new StringBuilder("<html>");

    String previousFirst = "";
    String previousSecond = "";

    Map<String, String> map = new HashMap<>();

    for (ResultInfo<?> info : messageSubList) {
      map.put(TEST, "<i>" + info.test.getName() + "</i>");
      map.put(SUBJECT, info.getSubjectName());
      TestStatus status = info.message.getStatus();
      map.put(STATUS, getFontColor(status) + getString(status.toString())
        + "</font>");

      String first = map.get(firstOrder);
      String second = map.get(secondOrder);
      String third = map.get(thirdOrder);

      builder.append("<strong>");
      if (!first.equals(previousFirst)) {
        if (!previousFirst.equals("")) {
          builder.append("<br>");
        }
        builder.append(first).append("<br>");
        previousFirst = first;
        previousSecond = "";
      }
      if (!second.equals(previousSecond)) {
        builder.append("&nbsp;&nbsp;").append(second).append("<br>");
        previousSecond = second;
      }

      builder.append("&nbsp;&nbsp;&nbsp;&nbsp;").append(third);
      builder.append("</strong>");
      builder.append(" - ").append(info.message.getMessage());
      builder.append("<br>");
    }
    builder.append("</html>");
    textArea.setText(builder.toString());
    textArea.setCaretPosition(0);
  }

  /**
   * Cria a tag que muda a cor em funo do status
   *
   * @param status O status
   * @return A tag de alterao da cor.
   */
  private Object getFontColor(TestStatus status) {
    if (status.equals(TestStatus.OK)) {
      return "<font color=\"green\">";
    }
    if (status.equals(TestStatus.WARNING)) {
      return "<font color=\"orange\">";
    }
    if (status.equals(TestStatus.ERROR)) {
      return "<font color=\"red\">";
    }

    return "<font color=\"black\">";
  }

  /**
   * Computa a chave que falta, do terceiro elemento a ser impresso
   *
   * @param firstOrder Chave do primeiro critrio de ordenao.
   * @param secondOrder Chave do segundo critrio de ordenao.
   * @return Chave do terceiro item.
   */
  private String computeThird(String firstOrder, String secondOrder) {
    if (!TEST.equals(firstOrder) && !TEST.equals(secondOrder)) {
      return TEST;
    }
    if (!SUBJECT.equals(firstOrder) && !SUBJECT.equals(secondOrder)) {
      return SUBJECT;
    }

    return STATUS;
  }

  /**
   * Ordena a lista de resultados
   *
   * @param list Lista de resultados.
   * @param firstCriteria Primeiro critrio de ordenao
   * @param secondCriteria Segundo critrio de ordenao
   * @return Lista ordenada
   */
  private List<ResultInfo<?>> sortData(List<ResultInfo<?>> list,
    String firstCriteria, String secondCriteria) {

    Map<String, Comparator<ResultInfo<?>>> compMap = new HashMap<>();
    compMap.put(TEST, new TestComparator());
    compMap.put(SUBJECT, new SubjectComparator());
    compMap.put(STATUS, new StatusComparator());

    DoubleLevelComparator comparator =
      new DoubleLevelComparator(compMap.get(firstCriteria), compMap
        .get(secondCriteria));

    Collections.sort(list, comparator);
    return list;
  }

  /**
   * Filtra a lista de resultados
   *
   * @param messagesList Lista de resultados.
   * @param okSelected Se deve incluir OK.
   * @param warningSelected Se deve incluir AVISO.
   * @param errorSelected Se deve incluir ERRO.
   * @return Lista filtrada.
   */
  private List<ResultInfo<?>> filterData(List<ResultInfo<?>> messagesList,
    boolean okSelected, boolean warningSelected, boolean errorSelected) {

    return messagesList
      .stream()
      .filter(
        r -> (r.message.getStatus().equals(TestStatus.OK) && okSelected == true)
        || (r.message.getStatus().equals(TestStatus.WARNING) && warningSelected == true)
        || (r.message.getStatus().equals(TestStatus.ERROR) && errorSelected == true))
      .collect(Collectors.toList());

  }

  /**
   * Monta o painel principal
   */
  private void mountPanel() {
    this.setLayout(new GridBagLayout());

    initializeStatusFilters();
    initializeTextStyleButtons();
    initializeSortingCombos();

    final Font titleFont = new Font(Font.SANS_SERIF, Font.BOLD, 14);

    // Filtros
    JComponent[][] filters =
      new JComponent[][] { { okFilter }, { warningFilter }, { errorFilter } };
    JPanel filtersPanel = createGridPanel(filters);
    filtersPanel.setBorder(BorderFactory.createTitledBorder(BorderFactory
      .createEmptyBorder(), getString("filters.title"),
      TitledBorder.DEFAULT_JUSTIFICATION, TitledBorder.LEFT, titleFont));

    // Estilo de texto
    JComponent[][] textStyleButtons =
      new JComponent[][] { { colapsedButton }, { extendedButton } };
    JPanel textStylePanel = createGridPanel(textStyleButtons);
    textStylePanel.setBorder(BorderFactory.createTitledBorder(BorderFactory
      .createEmptyBorder(), getString("text.style.title"),
      TitledBorder.DEFAULT_JUSTIFICATION, TitledBorder.LEFT, titleFont));

    // Ordenao
    JComponent[][] sortingCombos =
      new JComponent[][] {
      { new JLabel(getString("first.order.label")), firstSorter },
      { new JLabel(getString("second.order.label")), secondSorter } };
    JPanel sortingPanel = createGridPanel(sortingCombos);
    sortingPanel.setBorder(BorderFactory.createTitledBorder(BorderFactory
      .createEmptyBorder(), getString("sorting.title"),
      TitledBorder.DEFAULT_JUSTIFICATION, TitledBorder.LEFT, titleFont));

    textArea = new JTextPane();
    textArea.setFont(new Font(Font.MONOSPACED, Font.PLAIN, textArea.getFont()
      .getSize()));
    textArea.setEditable(false);
    textArea.setContentType("text/html");
    JScrollPane scrollPane = new JScrollPane(textArea);
    scrollPane.setMinimumSize(new Dimension(600, 400));
    scrollPane.setPreferredSize(new Dimension(600, 400));
    scrollPane
    .setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);

    JPanel leftPanel = new JPanel(new GridBagLayout());
    leftPanel.add(filtersPanel, new GBC(0, 0).insets(6, 6, 0, 6).none()
      .northwest());
    leftPanel.add(sortingPanel, new GBC(1, 0).insets(6, 11, 0, 6).none()
      .northwest());
    leftPanel.add(textStylePanel, new GBC(2, 0).insets(6, 11, 0, 6)
      .horizontal().northwest());
    leftPanel.add(new JPanel(), new GBC(3, 0).insets(6, 6, 6, 6).both());

    scrollPane.setBorder(BorderFactory.createEtchedBorder());

    this.setLayout(new GridBagLayout());
    this.add(leftPanel, new GBC(0, 0).horizontal().insets(6, 6, 6, 6));
    this.add(scrollPane, new GBC(0, 1).both().insets(6, 6, 6, 6));
  }

  /**
   * Cria e inicia as combos de ordenao
   */
  private void initializeSortingCombos() {
    if (firstSorter != null && secondSorter != null) {
      return;
    }

    TEST = getString("sort.by.test");
    STATUS = getString("sort.by.status");
    SUBJECT = getString("sort.by.subject");

    final List<String> allSorters = Arrays.asList(SUBJECT, STATUS, TEST);

    firstSorter = new JComboBox<>(new String[] { SUBJECT, STATUS, TEST });
    firstSorter.setSelectedIndex(0);
    secondSorter = new JComboBox<>(new String[] { STATUS, TEST });
    firstSorter.setSelectedIndex(0);

    firstSorter.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        String first = (String) firstSorter.getSelectedItem();
        String second = (String) secondSorter.getSelectedItem();

        List<String> secondSorters = new ArrayList<>(allSorters);
        secondSorters.remove(first);
        secondSorter.setModel(new DefaultComboBoxModel<>(secondSorters
          .toArray(new String[2])));

        if (second.equals(first)) {
          secondSorter.setSelectedIndex(0);
        }
        else {
          secondSorter.setSelectedItem(second);
        }

        updateTextArea();
      }
    });

    secondSorter.addActionListener(e -> updateTextArea());
  }

  /**
   * Cria e inicia os botes de estilo de texto (agrupamento)
   */
  private void initializeTextStyleButtons() {
    if (colapsedButton != null && extendedButton != null) {
      return;
    }

    colapsedButton = new JRadioButton(getString("collapsed.button"));
    extendedButton = new JRadioButton(getString("expanded.button"));

    ButtonGroup group = new ButtonGroup();
    group.add(colapsedButton);
    group.add(extendedButton);

    colapsedButton.setSelected(true);
    extendedButton.setSelected(false);

    ActionListener action = e -> updateTextArea();
    colapsedButton.addActionListener(action);
    extendedButton.addActionListener(action);
  }

  /**
   * Cria e inicia os checkbox de filtro
   */
  private void initializeStatusFilters() {
    if (okFilter != null && warningFilter != null && errorFilter != null) {
      return;
    }

    okFilter = new JCheckBox(getString("ok.check"));
    warningFilter = new JCheckBox(getString("warning.check"));
    errorFilter = new JCheckBox(getString("error.check"));

    ActionListener action = e -> updateTextArea();

    okFilter.setSelected(false);
    warningFilter.setSelected(true);
    errorFilter.setSelected(true);

    okFilter.addActionListener(action);
    warningFilter.addActionListener(action);
    errorFilter.addActionListener(action);
  }

  /**
   * Transforma o mapa de entrada numa lista com o formato que ser usado na
   * janela
   *
   * @param testMap Mapa de entrada (passado no construtor).
   * @return Lista da estrutura usada localmente.
   */
  // Assume que os tipos do teste e da resposta sejam compatveis pois o mapa foi gerado com o runner
  @SuppressWarnings("unchecked")
  private List<ResultInfo<?>> convertResult(
    Map<ITest<?>, ITestResult<?>> testMap) {
    List<ResultInfo<?>> messages = new ArrayList<>();

    for (ITest<?> test : testMap.keySet()) {
      ITestResult<?> result = testMap.get(test);

      for (IResultMessage<?> message : result.getMessages()) {
        // Assume que os tipos do teste e da resposta sejam compatveis pois o mapa foi gerado com o runner
        @SuppressWarnings("rawtypes")
        ResultInfo info = new ResultInfo();
        info.test = test;
        info.message = message;
        messages.add(info);
      }
    }
    return messages;
  }

  /**
   * Class usada para ordenar por teste.
   *
   * @author Tecgraf/PUC-Rio
   */
  private static class TestComparator implements Comparator<ResultInfo<?>> {

    /** {@inheritDoc} */
    @Override
    public int compare(ResultInfo<?> o1, ResultInfo<?> o2) {
      return o1.test.getName().compareToIgnoreCase(o2.test.getName());
    }
  }

  /**
   * Class usada para ordenar por dado.
   *
   * @author Tecgraf/PUC-Rio
   */
  private static class SubjectComparator implements Comparator<ResultInfo<?>> {
    /** {@inheritDoc} */
    @Override
    public int compare(ResultInfo<?> o1, ResultInfo<?> o2) {
      String s1 = o1.getSubjectName();
      String s2 = o2.getSubjectName();
      return s1.compareToIgnoreCase(s2);
    }
  }

  /**
   * Class usada para ordenar por status.
   *
   * @author Tecgraf/PUC-Rio
   */
  private static class StatusComparator implements Comparator<ResultInfo<?>> {

    /** {@inheritDoc} */
    @Override
    public int compare(ResultInfo<?> o1, ResultInfo<?> o2) {
      TestStatus s1 = o1.message.getStatus();
      TestStatus s2 = o2.message.getStatus();
      if (s1.equals(s2)) {
        return 0;
      }
      if (s1 == TestStatus.OK) {
        return -1;
      }
      if (s2 == TestStatus.OK) {
        return 1;
      }
      if (s1 == TestStatus.WARNING) {
        return -1;
      }
      if (s2 == TestStatus.WARNING) {
        return 1;
      }
      if (s1 == TestStatus.ERROR) {
        return -1;
      }
      if (s2 == TestStatus.ERROR) {
        return 1;
      }
      return 0;

    }
  }

  /**
   * Classe usada para ordenar por 2 critrios.
   *
   *
   * @author Tecgraf/PUC-Rio
   */
  static class DoubleLevelComparator implements Comparator<ResultInfo<?>> {
    /** primeiro critrio */
    final private Comparator<ResultInfo<?>> comp1;
    /** segundo critrio */
    final private Comparator<ResultInfo<?>> comp2;

    /**
     * Construtor
     *
     * @param comp1 primeiro critrio
     * @param comp2 segundo critrio
     */
    public DoubleLevelComparator(Comparator<ResultInfo<?>> comp1,
      Comparator<ResultInfo<?>> comp2) {
      this.comp1 = comp1;
      this.comp2 = comp2;
    }

    /** {@inheritDoc} */
    @Override
    public int compare(ResultInfo<?> o1, ResultInfo<?> o2) {
      int result1 = comp1.compare(o1, o2);
      if (result1 == 0) {
        return comp2.compare(o1, o2);
      }
      return result1;
    }
  }

  /**
   * Cria o bundle de internacionalizao.
   *
   * @param locale O idioma.
   *
   * @return o bundle.
   */
  private ResourceBundle createBundle(Locale locale) {
    try {
      return ResourceBundle.getBundle(this.getClass().getName(), locale);
    }
    catch (MissingResourceException e) {
      return null;
    }
  }

  /**
   * Obtm o texto usando a chave o locale atual
   *
   * @param key Chave do texto.
   * @return O texto internacionalizado.
   */
  protected String getString(String key) {

    if ((i18n != null) && i18n.containsKey(key)) {
      return i18n.getString(key);
    }

    return key;
  }

  /**
   * Cria o grupos com insects menores que os do GuiUtils
   * 
   * @param rows
   * @return Painel
   */
  public static final JPanel createGridPanel(JComponent[][] rows) {

    if (rows == null) {
      return null;
    }

    final JPanel panel = new JPanel();
    panel.setLayout(new GridBagLayout());

    final int T = 6;
    final int TI = 3;
    final int L = 6;
    final int R = 6;

    int y = 0;
    for (JComponent[] row : rows) {
      int t = TI;
      if (y == 0) {
        t = T;
      }

      panel.add(row[0], new GBC(0, y).insets(t, L, 0, R).none().west().weights(
        1.0, 0.0));

      if (row.length > 1) {
        panel.add(row[1], new GBC(1, y).insets(t, L, 0, R).horizontal()
          .weightx(100.0).west().weights(1.0, 0.0));
      }
      y++;
    }
    return panel;
  }

}
