package csbase.client.applications.algorithmsmanager.versiontree.datatransfer;

/*
 * $Id$
 */

import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.event.InputEvent;
import java.io.Serializable;

import javax.swing.JComponent;
import javax.swing.JTree;
import javax.swing.TransferHandler;
import javax.swing.tree.TreePath;

/**
 * @author Tecgraf / PUC-Rio
 * 
 *         Implementao de um {@link TransferHandler} para rvores.<br>
 *         Os ns que representam dados passveis de serem transmitidos devem
 *         implementar {@link ITransferableSource} e os ns representando
 *         o possvel estino dos dados devem implementar a interface {Link
 *         {@link ITransferableTarget}.
 */
public class VersionTreeTransferHandler extends TransferHandler {

  /**
   * {@inheritDoc}
   */
  @Override
  public int getSourceActions(JComponent c) {
    return COPY;
  }

  /**
   *  chamada ao incio de uma ao de transferncia para o clipboard.
   * 
   * @param source Componente fonte do objeto sendo transferido.
   * @param clip Deve ser <code>null</code>, pois est sendo utilizado um
   *        clipboard prprio.
   * @param action A ao que iniciou a transferncia.
   */
  @Override
  public void exportToClipboard(JComponent source, Clipboard clip, int action) {
    /**
     * S deve entrar aqui se chamou do menu (Recortar ou Copiar) (Para no
     * entrar com <CTRL>V ou <CTRL>C ). O TransferHandler s chama o ImportData
     * se existir algo no Clipboard, por isso, no funciona.
     */
    if (clip != null)
      throw new IllegalArgumentException("Clipboard deve ser vazio.");

    Transferable transferable = createTransferable(source);
    if (null != transferable) {
      VersionTreeClipboard.setObject(transferable);
    }

    super.exportToClipboard(source, clip, action);
  }

  /**
   *  chamada ao incio de uma ao de <i>Drag</i>.
   * 
   * @param source Componente fonte do objeto sendo transferido.
   * @param e o evento que disparou a operao.
   * @param action A ao que iniciou a transferncia.
   */
  public void exportAsDrag(JComponent source, InputEvent e, int action) {
    /*
     * Por enquanto, as operaes de Drag/Drop so controladas por
     * TransferHandler. O objeto a ser transferido no precisa ser colocado em
     * TransferObjectArea. Coloquei null para que o ImportData possa saber pode
     * usar o Transferable que recebeu como parametro.
     */
    VersionTreeClipboard.setObject(null);
    super.exportAsDrag(source, e, action);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Transferable createTransferable(JComponent c) {

    if (c instanceof JTree) {
      JTree tree = (JTree) c;
      try {
        TreePath[] paths = tree.getSelectionPaths();
        if (null == paths) {
          return null;
        }
        Serializable[] data = new Serializable[paths.length];
        DataFlavor flavor = null;
        for (int inx = 0; inx < data.length; inx++) {

          Object node = paths[inx].getLastPathComponent();
          if (ITransferableSource.class.isAssignableFrom(node.getClass())) {

            ITransferableSource source = (ITransferableSource) node;
            if (null != flavor && !flavor.equals(source.getDataFlavor())) {
              return null;
            }
            else {
              data[inx] = source.getData();
              flavor = source.getDataFlavor();
            }
          }
        }

        return new TransferableNode(data, flavor);
      }
      finally {
        tree.clearSelection();
      }
    }
    return null;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canImport(TransferSupport supp) {
    ITransferableTarget target = getDropTarget(supp);
    if (null == target) {
      return false;
    }
    else {
      try {
        Transferable transferable = supp.getTransferable();
        Object[] data = (Object[]) transferable.getTransferData(supp
          .getDataFlavors()[0]);
        DataFlavor flavor = transferable.getTransferDataFlavors()[0];
        return target.canImport(data, flavor);
      }
      catch (Exception e) {
        e.printStackTrace();
        return false;
      }
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean importData(TransferSupport supp) {
    try {
      // Fetch the Transferable and its data
      Transferable transferable = supp.getTransferable();
      Object[] data = (Object[]) transferable.getTransferData(supp
        .getDataFlavors()[0]);
      ITransferableTarget target = getDropTarget(supp);
      target.importData(data);

      return true;
    }
    catch (Exception e) {
      e.printStackTrace();

      return false;
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void exportDone(JComponent c, Transferable t, int action) {

  }

  /**
   * Obtm o destino dos dados sendo transferidos.
   * 
   * @param support o suporte  transferncia.
   * 
   * @return o destino dos dados sendo transferidos.
   */
  private ITransferableTarget getDropTarget(TransferSupport support) {

    TreePath path = null;
    if (support.isDrop()) {
      JTree.DropLocation dl = (JTree.DropLocation) support.getDropLocation();
      path = dl.getPath();
    }
    else { //  uma operao de copiar e colar
      if (support.getComponent() instanceof JTree) {
        path = ((JTree) support.getComponent()).getSelectionPath();
      }
    }

    if (null != path) {
      Object component = path.getLastPathComponent();
      if (null != component
        && ITransferableTarget.class.isAssignableFrom(component.getClass())) {
        return (ITransferableTarget) component;
      }
    }

    return null;
  }

  /**
   * A classe <code>TransferableNode</code> representa um ou mais objetos que
   * esto sendo transferidos por uma operao de drag&drop ou atravs do
   * clipboard. Essa classe implementa a interface <code>Transferable</code>.
   */
  public class TransferableNode implements Transferable {
    /**
     * O array com os formatos de dados de objetos que podem sofrer aes de
     * transferncia na rvore de projeto.
     */
    private final DataFlavor[] flavors;

    /** Os objetos que esto sendo transferidos. */
    private final Serializable[] data;

    /**
     * Constri uma representao dos objetos da rvore que sero transferidos.
     * 
     * @param data Os objetos a serem transferidos.
     * @param flavor formato de dados suportados pelo objeto a ser transferido.
     */
    public TransferableNode(Serializable[] data, DataFlavor flavor) {
      this.data = data;
      this.flavors = new DataFlavor[] { flavor };
    }

    /**
     * Obtm o objeto que est sendo transferido, de acordo com o formato
     * especificado.
     * 
     * @param flavor formato do dado requisitado. Deve ser
     *        <code>ProjectFileContainer.PROJECT_FILE_FLAVOR</code>.
     * 
     * @return uma instncia de ObjectToTransfer encapsulando os arquivos e/ou
     *         diretrios sendo transferidos.
     * @throws UnsupportedFlavorException se o formato do dado sendo transferido
     *         no for <code>ProjectFileContainer.PROJECT_FILE_FLAVOR</code>.
     */
    public synchronized Object getTransferData(DataFlavor flavor)
      throws UnsupportedFlavorException {
      if (!isDataFlavorSupported(flavor)) {
        throw new UnsupportedFlavorException(flavor);
      }
      return data;
    }

    /**
     * Obtm os formatos de dados que podem ser transferidos.
     * 
     * @return .
     */
    public DataFlavor[] getTransferDataFlavors() {
      return flavors;
    }

    /**
     * Verifica se esse objeto suporta um determinado formato de dados para
     * transferncia.
     * 
     * @param flavor formato de dados a ser verificado se  suportado pelo
     *        objeto sendo transferido.
     * 
     * @return true se o objeto sendo transferido suporta o formato de dados.
     */
    public boolean isDataFlavorSupported(DataFlavor flavor) {
      for (DataFlavor aFlavor : flavors) {
        if (aFlavor.equals(flavor)) {
          return true;
        }
      }
      return false;
    }
  }
}
