/*
 * $Id$
 */

package csbase.client.applications.projectsmanager.models;

import java.util.Set;

import csbase.logic.ProjectPermissions.SharingType;
import csbase.logic.User;

/**
 * Classe que contm as informaes relevantes de projetos para a aplicao
 * ProjectsManager.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class ProjectsManagerData {

  /**
   * Identificador do projeto.
   */
  final private Object projectId;

  /**
   * Nome do projeto.
   */
  final private String projectName;

  /**
   * Descrio do projeto
   */
  private String projectDescription;

  /**
   * Identificador do dono do projeto
   */
  final private Object ownerId;

  /**
   * Nome do dono do projeto
   */
  private String ownerName;

  /**
   * Espao alocado para este projeto
   */
  private long allocatedSpace;

  /**
   * Espao solicitado para este projeto.
   */
  private long requestedSpace;

  /**
   * Espao ocupado por este projeto
   */
  private double occupiedSpace;

  /**
   * Data de criao do projeto
   */
  private long creationDate;

  /**
   * Data da ltima modificao do projeto.
   */
  private long modificationDate;

  /**
   * Escopo de visibilidade do projeto.
   */
  private ProjectsManagerScope scope;

  /**
   * Tipo de compartilhamento do projeto.
   */
  private SharingType sharingType;

  /**
   * Estado de alocao do projeto.
   */
  private ProjectSpaceAllocation projectSpaceAllocation;

  /**
   * Indica a localizao do projeto no servidor.
   */
  private String location;

  /**
   * Conjunto de usurios com acesso RO.
   */
  private Set<Object> usersRO;
  /**
   * Conjunto de usurios com acesso RW.
   */
  private Set<Object> usersRW;

  /**
   * Nome do servidor que criou o projeto.
   */
  private String serverOwnerName;

  /**
   * Retorna
   * 
   * @return serverOwnerName
   */
  public String getServerOwnerName() {
    return serverOwnerName;
  }

  /**
   * @param serverOwnerName
   */
  public void setServerOwnerName(String serverOwnerName) {
    this.serverOwnerName = serverOwnerName;
  }

  /**
   * Consulta a data de criao do projeto.
   * 
   * @return a data
   */
  public long getCreationDate() {
    return creationDate;
  }

  /**
   * Ajusta a data de criao do projeto.
   * 
   * @param creationDate a data
   */
  public void setCreationDate(final long creationDate) {
    this.creationDate = creationDate;
  }

  /**
   * Consulta a data de modificao do projeto.
   * 
   * @return a data
   */
  public long getModificationDate() {
    return modificationDate;
  }

  /**
   * Consulta a localizao no servidor do projeto.
   * 
   * @return a localizao
   */
  final public String getLocationInServer() {
    return location;
  }

  /**
   * Ajusta a localizao no servidor.
   * 
   * @param location a localizao
   */
  public void setLocationInServer(final String location) {
    this.location = location;
  }

  /**
   * Ajusta a data de modificao do projeto.
   * 
   * @param modificationDate a data
   */
  public void setModificationDate(final long modificationDate) {
    this.modificationDate = modificationDate;
  }

  /**
   * Consulta o identificador do projeto
   * 
   * @return O identificador do projeto.
   */
  public Object getProjectId() {
    return projectId;
  }

  /**
   * Consulta o nome do projeto
   * 
   * @return o nome.
   */
  public String getProjectName() {
    return projectName;
  }

  /**
   * Consulta o id do dono do projeto.
   * 
   * @return o id
   */
  public Object getOwnerId() {
    return ownerId;
  }

  /**
   * Consulta o nome do dono do projeto.
   * 
   * @return o nome
   */
  public String getOwnerName() {
    return ownerName;
  }

  /**
   * Ajusta o nome do dono do projeto.
   * 
   * @param ownerName o nome
   */
  public void setOwnerName(final String ownerName) {
    this.ownerName = ownerName;
  }

  /**
   * Consulta o espao alocado por este projeto.
   * 
   * @return o espao alocado por este projeto
   */
  public long getAllocatedSpace() {
    return allocatedSpace;
  }

  /**
   * Ajusta o espao alocado por este projeto
   * 
   * @param allocatedSpace o espao alocado por este projeto
   */
  public void setAllocatedSpace(final long allocatedSpace) {
    this.allocatedSpace = allocatedSpace;
  }

  /**
   * Consulta o espao ocupado por este projeto (em MB).
   * 
   * @return o espao ocupado por este projeto (em MB)
   */
  public double getOccupiedSpace() {
    return occupiedSpace;
  }

  /**
   * Ajusta o espao ocupado por este projeto (em MB).
   * 
   * @param occupiedSpace o espao em MB
   */
  public void setOccupiedSpace(final double occupiedSpace) {
    this.occupiedSpace = occupiedSpace;
  }

  /**
   * Consulta a descrio do projeto.
   * 
   * @return a descrio.
   */
  public String getProjectDescription() {
    return projectDescription;
  }

  /**
   * Ajusta a descrio do projeto.
   * 
   * @param projectDescription a descrio.
   */
  public void setProjectDescription(final String projectDescription) {
    this.projectDescription = projectDescription;
  }

  /**
   * Retorna o conjunto de usurios com acesso RO ao projeto.
   * 
   * @return o conjunto de usurios com acesso RO ao projeto
   */
  public Set<Object> getUsersRO() {
    return usersRO;
  }

  /**
   * Define os conjuntos de usurios com acesso ao projeto.
   * 
   * @param usersRO - usurios com acesso RO. Se for <code>null</code>, no
   *        altera o conjunto existente.
   * @param usersRW - usurios com acesso RW. Se for <code>null</code>, no
   *        altera o conjunto existente.
   * @return <code>true</code> se algum dos conjuntos foi redefinido
   */
  public boolean setUsers(final Set<Object> usersRO, final Set<Object> usersRW) {
    boolean changed = false;
    if (usersRO != null) {
      this.usersRO = usersRO;
      changed = true;
    }
    if (usersRW != null) {
      this.usersRW = usersRW;
      changed = true;
    }
    return changed;
  }

  /**
   * Retorna o conjunto de usurios com acesso RW ao projeto.
   * 
   * @return o conjunto de usurios com acesso RW ao projeto
   */
  public Set<Object> getUsersRW() {
    return usersRW;
  }

  /**
   * Verifica se um usurio  o dono do projeto.
   * 
   * @param uid - ID do usurio
   * @return true se o usurio  o dono do projeto (ou  o admin)
   */
  public boolean userIsOwner(final Object uid) {
    return User.isAdmin(uid) || ownerId.equals(uid);
  }

  /**
   * Ajusta escopo do projeto
   * 
   * @param scope escopo
   */
  final public void setScope(final ProjectsManagerScope scope) {
    if (scope == null) {
      throw new IllegalArgumentException("scope == null!");
    }
    this.scope = scope;
  }

  /**
   * Consulta o escopo do projeto.
   * 
   * @return scope o escopo
   */
  public final ProjectsManagerScope getScope() {
    return scope;
  }

  /**
   * Remove todas as informaes do projeto que so exclusivas de projetos sem
   * alocao ou devidamente alocados.
   */
  public void clearData() {
    sharingType = SharingType.PRIVATE;
    creationDate = 0;
    modificationDate = 0;
    occupiedSpace = -1;
    scope = ProjectsManagerScope.NOT_APPLICABLE;
  }

  /**
   * Consulta o tipo de alocao.
   * 
   * @return o tipo de alocao
   */
  final public ProjectSpaceAllocation getProjectSpaceAllocation() {
    return projectSpaceAllocation;
  }

  /**
   * Ajusta o tipo de alocao.
   * 
   * @param projectSpaceAllocation o tipo de alocao.
   */
  final public void setProjectSpaceAllocation(
    final ProjectSpaceAllocation projectSpaceAllocation) {
    this.projectSpaceAllocation = projectSpaceAllocation;
  }

  /**
   * Consulta o espao requerido
   * 
   * @return o espao.
   */
  public long getRequestedSpace() {
    return requestedSpace;
  }

  /**
   * Ajusta o espao requerido
   * 
   * @param requestedSpace o espao
   */
  public void setRequestedSpace(final long requestedSpace) {
    this.requestedSpace = requestedSpace;
  }

  /**
   * Obtm o tipo de compartilhamento do projeto.
   * 
   * @return tipo de compartilhamento do projeto
   */
  public SharingType getSharingType() {
    return sharingType;
  }

  /**
   * Define o tipo de compartilhamento. Define tambm o escopo.
   * 
   * @param st - tipo de compartilhamento
   * @return true se o tipo foi alterado, i.e. se o novo tipo era diferente do
   *         anterior
   */
  public boolean setSharingType(final SharingType st) {
    if (sharingType == st) {
      return false;
    }
    sharingType = st;
    if (projectSpaceAllocation == ProjectSpaceAllocation.WAITING_ALLOCATION
      || projectSpaceAllocation == ProjectSpaceAllocation.WAITING_DEALLOCATION) {
      scope = ProjectsManagerScope.NOT_APPLICABLE;
      return true;
    }
    switch (st) {
      case PRIVATE:
        scope = ProjectsManagerScope.PRIVATE;
        break;
      case ALL_RO:
      case ALL_RW:
        scope = ProjectsManagerScope.PUBLIC;
        break;
      case PARTIAL:
        scope = ProjectsManagerScope.SHARED;
        break;
    }
    return true;
  }

  /**
   * Construtor
   * 
   * @param projectId Identificador do projeto.
   * @param projectName Nome do projeto.
   * @param ownerId Identificador do dono do projeto.
   */
  public ProjectsManagerData(Object projectId, String projectName,
    Object ownerId) {
    this.projectId = projectId;
    this.projectName = projectName;
    this.ownerId = ownerId;
  }
}
