package csbase.client.project;

import java.awt.Window;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.event.InputEvent;

import javax.swing.JComponent;
import javax.swing.TransferHandler;

import tecgraf.javautils.core.lng.LNG;
import csbase.client.desktop.RemoteTask;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.ClientProjectFile;
import csbase.util.restart.RestartListener;
import csbase.util.restart.RestartManager;

/**
 * @author TecMSV
 */
public abstract class AbstractProjectFileTransferHandler extends
  TransferHandler {

  /**
   * O array com os formatos de dados de objetos que podem sofrer aes de
   * transferncia na rvore de projeto.
   */
  private static final DataFlavor[] _flavors =
    new DataFlavor[] { ProjectFileContainer.PROJECT_FILE_FLAVOR };

  /** A janela  qual pertence esse handler */
  private final Window _ownerWindow;

  /** O ttulo da janela dona do handler */
  private final String _ownerTitle;

  /**
   * Constri um handler de transferncia de objetos de uma rvore de projeto.
   * 
   * @param ownerWindow A janela sobre a qual uma DesktopTask deve ser efetuada.
   * @param ownerTitle O ttulo da janela de DesktopTask
   */
  public AbstractProjectFileTransferHandler(Window ownerWindow,
    String ownerTitle) {
    super();
    this._ownerWindow = ownerWindow;
    this._ownerTitle = ownerTitle;
  }

  /**
   * @return true se existe um arquivo para ser copiado/movido.
   */
  public boolean hasTransferableInClipboard() {
    TransferInfo info = TransferInfoManager.getTransferInfo();
    return null != info && info instanceof ClipboardTransferInfo;
  }

  /**
   * Limpa a rea de transferncia caso esteja ocorrendo uma transferncia que
   * tenha sido iniciada por uma ao de COPY/PASTE.
   */
  public void clearClipboard() {
    if (hasTransferableInClipboard()) {
      TransferInfoManager.clear();
    }
  }

  /**
   * Obtm os tipos de ao de transferncia suportados pela rvore de projeto.
   * Pode ser COPY ou MOVE.
   * 
   * @param c .
   * @return O cdigo da ao <code>TransferHandler.COPY_OR_MOVE</code>.
   */
  @Override
  public int getSourceActions(JComponent c) {
    return COPY_OR_MOVE;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean canImport(JComponent c, DataFlavor[] flavor) {
    for (int i = 0, n = flavor.length; i < n; i++) {
      for (int j = 0, m = _flavors.length; j < m; j++) {
        if (flavor[i].equals(_flavors[j])) {
          return true;
        }
      }
    }
    return false;
  }

  /**
   *  chamada ao incio de uma ao de <i>Drag</i>.
   * 
   * @param source Componente fonte do objeto sendo transferido.
   * @param event O evento que iniciou a transferncia.
   * @param action A ao (COPY/MOVE) que iniciou a transferncia.
   */
  @Override
  public void exportAsDrag(JComponent source, InputEvent event, int action) {

    // Por enquanto, as operaes de Drag/Drop so controladas 
    // por TransferHandler. O objeto a ser transferido no precisa 
    // ser colocado em TransferObjectArea. Coloquei null para que o ImportData
    // possa saber pode usar o Transferable que recebeu como parametro.    
    TransferInfoManager.setTransferInfo(new DragTransferInfo(source));

    super.exportAsDrag(source, event, action);
  }

  /**
   *  chamada ao incio de uma ao de transferncia para o clipboard.
   * 
   * @param source Componente fonte do objeto sendo transferido.
   * @param clip Deve ser <code>null</code>, pois est sendo utilizado um
   *        clipboard prprio.
   * @param action A ao (COPY/MOVE) que iniciou a transferncia.
   */
  @Override
  public void exportToClipboard(JComponent source, Clipboard clip, int action) {

    clearClipboard();

    // S deve entrar aqui se chamou do menu (Recortar ou Copiar) 
    // (Para no entrar com <CTRL>V ou <CTRL>C )
    // O TransferHandler s chama o ImportData se existir algo no Clipboard,
    // por isso, no funciona.
    if (clip != null) {
      throw new IllegalArgumentException("Clipboard deve ser vazio.");
    }

    Transferable transferable = createTransferable(source);
    if (null != transferable) {
      TransferInfoManager.setTransferInfo(new ClipboardTransferInfo(source,
        transferable, action));
    }
  }

  /**
   * Cria uma representao do objeto que vai ser transferido durante uma
   * operao de drag&drop ou atravs do clipboard. Nesse caso, o objeto criado
   * contm referncias para um ou mais instncias de <code>ProjectFile</code>
   * que foram selecionadas na rvore.
   * 
   * @param dragSource O compoenente <code>JTree</code> que contm os objetos
   *        que sero transferidos.
   * 
   * @return Um objeto <code>Transferable</code> que contm os ns de arquivos
   *         ou diretrios selecionados na rvore de projeto.
   */
  @Override
  public Transferable createTransferable(JComponent dragSource) {

    ClientProjectFile[] files = getFilesToExport(dragSource);
    if (null == files || 0 == files.length) {
      return null;
    }
    else {
      return new TransferableNode(files, _flavors);
    }
  }

  /**
   * Este mtodo deveria implementar a importao dos dados na fonte do drop,
   * porm como as aes rodam dentro de uma thread, a ao de mover no poderia
   * ser quebrada em importar(importData)+remover(exportDone), pois se um erro
   * ocorresse durante a importao, a etapa de remover ocorreria em seguida e
   * os dados seriam perdidos. Devido a isso, a implementao das aes ficou a
   * cargo do mtodo exportDone pois este recebe que ao deve ser executada,
   * copiar ou mover, e assim no precisamos quebrar a ao de mover em etapas. <br/>
   * Este mtodo ento, fica encarregado de testar se a importao poderia ser
   * feita e de salvar o destino da importao para ser utilizado pelo mtodo
   * exportDone.<br/>
   * Obs. Isso s foi possvel por que a rvore  tanto a fonte do drag quanto
   * do drop.
   * 
   * @param destination O componente que possui os objetos sobre o qual a
   *        transferncia ser efetuada.
   * @param data O objeto que representa o que est sendo transferido.
   * 
   * @return Verdadeiro, se a transferncia foi feita ou, falso, caso a
   *         transferncia no seja feita.
   */
  @Override
  public boolean importData(JComponent destination, Transferable data) {

    boolean imported = false;

    try {
      if (!canImport(destination, _flavors)) {
        return false;
      }

      TransferInfo info = TransferInfoManager.getTransferInfo();
      if (null == info) {
        return false;
      }

      if (TransferInfo.ExportMethod.CLIPBOARD == info.getMethod()) {
        data = ((ClipboardTransferInfo) info).getTransferable();
      }
      if (null == data) {
        return false;
      }

      if (!data.isDataFlavorSupported(_flavors[0])) {
        return false;
      }

      /**
       * Na implementao normal de um TransferHandler, se a ao do drag fosse
       * um copy ou um move, durante a chamada ao mtodo 'importData' seria
       * feita uma copia do dado exportado. Em caso de sucesso, o mtodo
       * 'exportDone(..)' seria chamado e a ao final seria checada para
       * verificar se foi um move. Neste caso, o dado exportado seria removido
       * de sua origem.
       * 
       * Essa quebra na implementacao da ao move ocorre por que na execuo do
       * mtodo 'importData(...)' no se sabe que ao est sendo executada,
       * pois ela pode ser diferente da ao inicial passada ao mtodo
       * 'exportAsDrag(...)', j que as aes mover, copiar e linkar podem
       * alternar entre si at o momento final de importar os dados. Essa
       * alternao de ao  feita com as teclas CTRL e SHIFT que podem
       * transformar um move em copiar e linkar respectivamente.
       * 
       * Esta implementao, porm, no foi possvel neste caso. Como tanto o
       * copy quanto o move acessam o servidor, estes devem rodarem uma thread
       * para nao travar o cliente. Com isso, se quebrassemos a implementao da
       * ao mover = importar(importData)+remover(exportDone) como pregado pelo
       * modelo original de implementao do TransferHandler, caso ocorresse um
       * erro na thread durante a importao, quando o exportDone fosse chamado
       * ele nao saberia que houve a falha e removeria o dado original e assim
       * perderamos o arquivo. Devido a isso, a implementao da ao move no
       * poderia ser quebrada e como s sabemos a ao de DnD no mtodo
       * 'exportDone', a implementao das aes foi movida para l. Isso s foi
       * possvel por que a rvore  tanto a fonte do drag quanto do drop, pois
       * o mtodo importData serve para avisar a fonte de drop que tem dado para
       * ser importado e em caso de sucesso, o mtodo 'exportDone' da fonte de
       * drag  chamado para informar que se foi uma ao de move, pode apagar o
       * original pois a fonte de drop j o copiou.
       */
      {
        /**
         * Salva o diretrio de destino para ser utilizado pelo mtodo
         * exportDone que ser quem realmente ir executar as aes.
         */
        final ClientProjectFile destinationDirectory =
          getDestinationDirectory(destination);
        if (null == destinationDirectory || !destinationDirectory.isDirectory()) {
          return false;
        }

        info.setDestinationDirectory(destinationDirectory);
        imported = true;
        if (TransferInfo.ExportMethod.CLIPBOARD == info.getMethod()) {
          exportDone(info.getSource(), data,
            ((ClipboardTransferInfo) info).getAction());
        }
        return imported;
      }
    }
    finally {
      if (!imported) {
        TransferInfoManager.clear();
      }
    }
  }

  /**
   * Este mtodo deveria apenas remover os dados de origem aps uma importao
   * bem sucedida no caso de uma ao de mover (<code>MOVE</code>), porm como
   * as aes rodam dentro de uma thread, este mtodo no teria como saber se a
   * importao foi bem sucedida ou no para ento remover o dado original.
   * Desta forma, este mtodo est disparando as aes, para que assim a ao de
   * mover no seja quebrada em duas (importar+apagar).<br/>
   * Obs. Isso s foi possvel por que a rvore  tanto a fonte do drag quanto
   * do drop.
   * 
   * @param source o componente que foi a fonte dos dados.
   * @param data o dado transferido ou null se a ao foi <code>NONE</code>.
   * @param action a ao que foi executada.
   */
  @Override
  protected void exportDone(JComponent source, Transferable data, int action) {

    // Testa se a ao  vlida
    int transferAction = getSourceActions(source) & action;
    if (transferAction == NONE) {
      return;
    }

    TransferInfo info = TransferInfoManager.getTransferInfo();
    if (null == info) {
      return;
    }

    if (TransferInfo.ExportMethod.CLIPBOARD == info.getMethod()) {
      data = ((ClipboardTransferInfo) info).getTransferable();
    }

    final ClientProjectFile destinationDirectory =
      info.getDestinationDirectory();
    if (null == destinationDirectory) {
      return;
    }

    if (data.isDataFlavorSupported(_flavors[0])) {
      try {
        final ClientProjectFile[] files =
          (ClientProjectFile[]) data.getTransferData(_flavors[0]);
        if (null == files || 0 == files.length) {
          return;
        }

        switch (transferAction) {
          case COPY: {
            copy(files, destinationDirectory);
            break;
          }
          case MOVE: {
            if (isMoveValid(files)) {
              move(files, destinationDirectory);
            }
            // Tirando o objeto da rea de transferncia
            clearClipboard();
            break;
          }
        }
      }
      catch (Exception e) {
        // Tirando o objeto da rea de transferncia
        clearClipboard();
        StandardErrorDialogs.showErrorDialog(_ownerWindow, _ownerTitle,
          LNG.get("AbstractProjectFileTransferHandler.error.export"), e);
      }
    }
  }

  /**
   * @param source componente fonte de uma ao de copiar ou de mover.
   * @return arquivos/diretrios que sero exportados.
   */
  protected abstract ClientProjectFile[] getFilesToExport(JComponent source);

  /**
   * @param destination componente que ir receber os arquivos/diretrios
   *        exportados.
   * @return o diretrio de destino dos arquivos exportados (copiados/movidos).
   */
  protected abstract ClientProjectFile getDestinationDirectory(
    JComponent destination);

  /**
   * @param files
   * @return true se os arquivos escolhidos podem ser movidos e false caso o
   *         contrrio.
   */
  private boolean isMoveValid(ClientProjectFile[] files) {
    for (ClientProjectFile file : files) {
      if (null == file || null == file.getParent()) {
        return false;
      }
    }

    return true;
  }

  /**
   * Solicita que um ou mais arquivos e/ou diretrios sejam copiados para um
   * diretrio da rvore. A cpia pode ser feita entre projetos diferentes.
   * 
   * @param files Arquivos e/ou diretrios a serem copiados.
   * @param dir Diretrio de destino.
   */
  private void copy(final ClientProjectFile[] files, final ClientProjectFile dir) {
    // Cria a tarefa remota
    RemoteTask<Void> task = new RemoteTask<Void>() {
      @Override
      protected void performTask() throws Exception {
        for (int inx = 0; inx < files.length; inx++) {
          files[inx].copy(dir);
        }
      }
    };

    // Executa a tarefa. Os erros j so tratados diretamente por ela.
    task.execute(_ownerWindow, _ownerTitle, LNG.get("PRJ_WAITING_FILE_COPY"));
  }

  /**
   * Solicita que um ou mais arquivos e/ou diretrios sejam movidos para um
   * diretrio da rvore. Pode mover arquivos e/ou diretrios entre projetos
   * diferentes.
   * 
   * @param files Os arquivo e/ou diretrios a serem movidos.
   * @param dir Diretrio parade destino.
   */
  private void move(final ClientProjectFile[] files, final ClientProjectFile dir) {
    // Cria a tarefa remota
    RemoteTask<Void> task = new RemoteTask<Void>() {

      @Override
      protected void performTask() throws Exception {
        for (int inx = 0; inx < files.length; inx++) {
          files[inx].move(dir);
        }
      }
    };

    // Executa a tarefa remota. Os erros so tratados diretamente por ela.
    task.execute(_ownerWindow, _ownerTitle, LNG.get("PRJ_WAITING_FILE_MOVE"));
  }

  /**
   * Esta classe tem como objetivo armazenar objetos a serem tranferidos,
   * evitando assim o uso so clipboard, uma vez que seu acesso no  permitido
   * para Applets.
   */
  static class TransferInfoManager {

    /** Informaes de uma transferncia em andamento. */
    private static TransferInfo _info;

    static {
      RestartManager.getInstance().addListener(new RestartListener() {
        public void restart() {
          clear();
        }
      });
    }

    /**
     * Clear clipboard content.
     */
    public synchronized static void clear() {
      setTransferInfo(null);
    }

    /**
     * @param info
     */
    public synchronized static void setTransferInfo(TransferInfo info) {
      if (null != TransferInfoManager._info) {
        TransferInfoManager._info.fireOnUnload();
      }

      TransferInfoManager._info = info;

      if (null != info) {
        info.fireOnLoad();
      }
    }

    private static void checkAndUnsetFilesMoving() {
      if (null == _info) {
        return;
      }
      if (TransferInfo.ExportMethod.CLIPBOARD != _info.getMethod()) {
        return;
      }

    }

    /**
     * @return uma instncia de TransferInfo contendo informaes de uma
     *         transferncia em andamento ou null casl no haja nenhuma.
     */
    public static TransferInfo getTransferInfo() {
      return _info;
    }
  }

  /**
   * Encapsula informaes que sero utilizadas na exportaso de
   * arquivos/diretrios.
   */
  static class TransferInfo {

    /**
     * Indica qual mtodo comeou uma transferncia.
     */
    public enum ExportMethod {
      /** Indica que a transferncia foi iniciada pelo mtodo exportToClipboard. */
      CLIPBOARD,
      /** Indica que a transferncia foi iniciada pelo mtodo exportAsDrag. */
      DRAG
    };

    /** Componente fonte do objeto sendo transferido. */
    private final JComponent _source;
    /** Mtodo de exportao. */
    private final ExportMethod _method;

    /** Diretrio de destino da transferncia. */
    private ClientProjectFile _destinationDirectory;

    /**
     * @param source
     * @param method
     */
    protected TransferInfo(JComponent source, ExportMethod method) {
      _source = source;
      _method = method;
    }

    /**
     * @return the sourceComponent
     */
    public JComponent getSource() {
      return _source;
    }

    /**
     * @return the method
     */
    public ExportMethod getMethod() {
      return _method;
    }

    /**
     * @return o diretrio de destino dos arquivos sendo transferidos.
     */
    public ClientProjectFile getDestinationDirectory() {
      return _destinationDirectory;
    }

    /**
     * @param destinationDirectory diretrio de destino dos arquivos sendo
     *        transferidos.
     */
    public void setDestinationDirectory(ClientProjectFile destinationDirectory) {
      _destinationDirectory = destinationDirectory;
    }

    /**
     * Evento disparado quando esta instncia  salva em memria.
     */
    public void fireOnLoad() {

    }

    /**
     * Evento disparado quando esta instncia  removida da memria.
     */
    public void fireOnUnload() {

    }
  }

  /**
   * Especializao da classe TransferInfo para quando a exportao ocorreu
   * atravs do mtodo exportToClipboard.
   */
  static class ClipboardTransferInfo extends TransferInfo {

    /** Objeto sendo transferido. */
    private final Transferable _transferable;
    /** Ao que gerou a transferncia do objeto. */
    private final int _action;

    /**
     * @param source
     * @param transferable
     * @param action
     */
    public ClipboardTransferInfo(JComponent source, Transferable transferable,
      int action) {
      super(source, ExportMethod.CLIPBOARD);

      _transferable = transferable;
      _action = action;
    }

    /**
     * @return a ao que iniciou essa exportao.
     */
    public int getAction() {
      return _action;
    }

    /**
     * @return uma instncia de Transferable que encapsula os arquivos/diretrio
     *         rio sendo transferidos.
     */
    public Transferable getTransferable() {
      return _transferable;
    }

    /**
     * Caso a ao que gerou a transferncia seja de MOVE, marca os arquivos da
     * transferncia de estarem sendo movidos. Desta forma, a fonte pode
     * desenhar o cone de acordo com a ao (COPY x MOVE).
     */
    @Override
    public void fireOnLoad() {
      if (TransferHandler.MOVE != getAction()) {
        return;
      }

      try {
        ClientProjectFile[] files =
          (ClientProjectFile[]) getTransferable().getTransferData(_flavors[0]);

        for (ClientProjectFile file : files) {
          file.setMoving(true);
        }
      }
      catch (Exception e) {
        StandardErrorDialogs.showErrorDialog(null, "Clipboard",
          LNG.get("AbstractProjectFileTransferHandler.error.export"), e);
      }

      getSource().repaint();
    }

    /**
     * Caso a ao que gerou a transferncia seja de MOVE, desmarca os arquivos
     * da transferncia de estarem sendo movidos. Desta forma, a fonte pode
     * desenhar o cone de acordo com a ao (COPY x MOVE).
     */
    @Override
    public void fireOnUnload() {
      if (TransferHandler.MOVE != getAction()) {
        return;
      }

      try {
        ClientProjectFile[] files =
          (ClientProjectFile[]) getTransferable().getTransferData(_flavors[0]);

        for (ClientProjectFile file : files) {
          file.setMoving(false);
        }
      }
      catch (Exception e) {
        StandardErrorDialogs.showErrorDialog(null, "Clipboard",
          LNG.get("AbstractProjectFileTransferHandler.error.export"), e);
      }

      getSource().repaint();
    }
  }

  /**
   * Especializao da classe TransferInfo para quando a exportao ocorreu
   * atravs do mtodo exportAsDrag.
   */
  static class DragTransferInfo extends TransferInfo {
    protected DragTransferInfo(JComponent source) {
      super(source, ExportMethod.DRAG);
    }
  }
}
