package csbase.client.util;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * Executar tarefas precedidas de uma contagem regressiva reinicivel. Ou seja,
 * a tarefa ser executada caso o {@link #restart()} no seja chamado por um
 * certo tempo. Por exemplo, se quisessemos validar um dado entrado em um campo
 * de texto, 1s depois que o usurio pare de digitar. 
 * <code>
 * final JTextField nameTf = new JTextField();
 * final CountDown nameValidation = new CountDown(1, TimeUnit.SECONDS, new Runnable() { 
 *   @Override 
 *   public void run() { 
 *      validateName(nameTf.getText()); 
 *   }
 * });
 * nameTf.addKeyListener(new KeyAdapter() {
 *    @Override
 *    public void keyReleased(KeyEvent e) {
 *      nameValidation.restart();
 *    }
 * }); 
 * </code>
 * 
 * @author Tecgraf
 */
public class CountDown implements Runnable {

  private long count;
  private Runnable task;

  private Lock lock = new ReentrantLock();
  private long started;
  private Thread thread;

  /**
   * Construtor.
   * 
   * @param count tempo a ser contado antes a execuo da tarefa.
   * @param unit unidade do tempo a ser contado.
   * @param task tarefa a ser executada quando a contagem terminar.
   */
  public CountDown(long count, TimeUnit unit, Runnable task) {
    this.count = TimeUnit.MILLISECONDS.convert(count, unit);
    this.task = task;
  }

  /**
   * (Re)Inicia a contagem que precede a execuo da tarefa.<br>
   * Ou seja, se h uma contagem em andamento, ele reinica ela. Caso contrrio,
   * inicia uma nova que ir preceder a execuo da tarefa.
   */
  public void restart() {
    lock.lock();
    try {
      if (thread == null) {
        thread = new Thread(this);
        thread.setDaemon(true);
        started = System.currentTimeMillis();
        thread.start();
      }
      else {
        started = System.currentTimeMillis();
      }
    }
    finally {
      lock.unlock();
    }
  }

  @Override
  public void run() {
    while (Thread.currentThread().equals(thread)) {
      long elapsed;
      lock.lock();
      try {
        elapsed = System.currentTimeMillis() - started;
        if (elapsed > count) {
          thread = null;
          break;
        }
      }
      finally {
        lock.unlock();
      }

      try {
        long remaining = count - elapsed;
        TimeUnit.MILLISECONDS.sleep(remaining);
      }
      catch (InterruptedException e) {
        e.printStackTrace();
      }
    }

    task.run();
  }
}
