/*
 * $Id: DirectoryChooser.java,v 1.3 2010/07/16 22:25:32 clinio Exp $
 */

package csbase.client.util.filechooser;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.GridBagLayout;
import java.awt.Toolkit;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.io.File;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ActionMap;
import javax.swing.DefaultComboBoxModel;
import javax.swing.DefaultListCellRenderer;
import javax.swing.ImageIcon;
import javax.swing.InputMap;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.JToggleButton;
import javax.swing.JToolBar;
import javax.swing.KeyStroke;

import tecgraf.javautils.core.io.FileUtils;
import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.BorderUtil;
import tecgraf.javautils.gui.ComboUtil;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.GUIUtils;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.ClientLocalFile;
import csbase.client.applications.ApplicationImages;
import csbase.client.desktop.DesktopFrame;
import csbase.client.project.DirectoryCreationDialog;
import csbase.client.util.StandardErrorDialogs;
import csbase.client.util.filechooser.fileparentcombopanel.ClientFileParentComboChooseListener;
import csbase.client.util.filechooser.fileparentcombopanel.ClientFileParentComboPanel;
import csbase.client.util.filechooser.filetablepanel.ClientFileTablePanel;
import csbase.client.util.filechooser.filters.ClientFileAllFilter;
import csbase.client.util.filechooser.filters.ClientFileFilterInterface;
import csbase.client.util.filechooser.filters.ClientFileFilterInterface.Operation;
import csbase.client.util.filechooser.images.ChooserImages;
import csbase.client.util.filechooser.util.ClientFileChooserUtils;
import csbase.exception.BugException;
import csbase.exception.PermissionException;
import csbase.logic.ClientFile;
import csbase.logic.ClientFileType;
import csbase.logic.ClientSGAFile;
import csbase.logic.CommonClientProject;
import csbase.logic.ProjectFileType;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.SGAServiceInterface;

/**
 * Chooser de arquivos {@link ClientFile}.
 *
 * A configurao deste chooser  composta por 3 partes: <br/>
 * 1 - Arquivos ou diretrios {@link ClientFileChooserSelectionMode} <br/>
 * 2 - Open ou Save {@link ClientFileChooserType}<br/>
 * 3 - Seleo mltipla ou seleo simples {@link ClientFileChooserCardinality}
 *
 * Alm disso,  possvel definir os filtros deste chooser atravs de
 * implementaes da interface {@link ClientFileFilterInterface}.
 *
 * Os utilitrios abaixo encapsulam toda essa configurao do chooser. Para que
 * estes utilitrios disparem este chooser ao invs do {@link JFileChooser}
 * padro,  preciso definir uma propriedade no arquivo
 * 'ApplicationProject.properties' com a seguinte linha: <br/>
 *
 * new.chooser = true
 *
 * @see ClientLocalFileChooserUtil
 * @see ClientProjectFileChooserUtil
 * @see ClientSGAFileChooserUtil
 *
 * @author Tecgraf/PUC-Rio
 */
public class ClientFileChooser extends JDialog {

  /**
   * Separador usado para seleo mltipla de arquivos.
   */
  private static final String MULTIPLE_PATHS_SEPARATOR = ";";

  /**
   * Boto de cancelamento
   */
  final private JButton cancelButton = new JButton();

  /**
   * Toggle de filtragem de arquivos ocultos.
   */
  final private JToggleButton filterToggle = new JToggleButton();

  /**
   * Indicativo de cancelamento.
   */
  private List<ClientFile> chosenItens = null;

  /**
   * Lista de filtro selecionado.
   */
  final private JComboBox filterCombo = new JComboBox();

  /**
   * Ao de ir para o home do usurio.
   */
  final private JButton localHomeDirButton = new JButton();

  /**
   * Boto de ok
   */
  final private ClientFileParentComboPanel hierarchyCombo =
    new ClientFileParentComboPanel();

  /**
   * Label de seleo
   */
  final private JLabel hierarchyLabel = new JLabel();

  /**
   * Boto de atalho:
   */
  final private JButton localRootDirButton = new JButton();

  /**
   * Boto de ok
   */
  final private JButton okButton = new JButton();

  /**
   * Boto de criao de diretrio.
   */
  final private JButton newDirectoryAction = new JButton();

  /**
   * Boto de atalho:
   */
  final private JButton projHomeDirButton = new JButton();

  /**
   * Path selecionado pelo usurio em caso de save (o arquivo pode no existir).
   */
  private String[] selectedSavePath = null;

  /**
   * Label de seleo
   */
  final private JTextField selectionText = new JTextField();

  /**
   * Painel de tabela
   */
  final private ClientFileTablePanel tablePanel = new ClientFileTablePanel();

  /**
   * Indicativo de que um novo arquivo pode ser escolhido.
   */
  private ClientFileChooserType type;

  /**
   * Lista de filtros de visualizao disponveis.
   */
  List<ClientFileFilterInterface> viewFilters =
    new ArrayList<ClientFileFilterInterface>();

  /**
   * Monta um path diretrio + arquivo.
   *
   * @param baseDir path do diretrio
   * @param relativePath nome do arquivo
   * @return path completo.
   */
  private String[] appendPath(final String[] baseDir,
    final String[] relativePath) {
    int dirLenght = baseDir.length;
    String[] dir = baseDir;
    if (dirLenght == 0) {
      dir = new String[] { "" };
      dirLenght++;
    }
    String[] path = new String[dirLenght + relativePath.length];
    System.arraycopy(dir, 0, path, 0, dirLenght);
    System.arraycopy(relativePath, 0, path, dirLenght, relativePath.length);
    return path;
  }

  /**
   * Constri o painel de botes.
   *
   * @return o painel
   */
  private JPanel buildButtonsPanel() {
    final AbstractAction cancelAction = createCancelAction();
    cancelButton.setAction(cancelAction);

    final AbstractAction okAction = createOkAction();
    okButton.setAction(okAction);

    GUIUtils.matchPreferredSizes(new JComponent[] { cancelButton, okButton });
    final JPanel panel = new JPanel();
    panel.setLayout(new FlowLayout(FlowLayout.CENTER));
    panel.add(cancelButton);
    panel.add(okButton);
    return panel;
  }

  /**
   * Criao da ao de cancelamento.
   *
   * @return ao
   */
  private AbstractAction createCancelAction() {
    final AbstractAction action = new AbstractAction() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        handleCancelButton();
      }
    };
    final int esc = KeyEvent.VK_ESCAPE;
    final KeyStroke stroke = KeyStroke.getKeyStroke(esc, 0);
    action.putValue(Action.ACCELERATOR_KEY, stroke);
    setKeyStroke(action, stroke);
    final String text = getString("cancel.button");
    action.putValue(Action.NAME, text);
    return action;
  }

  /**
   * Ajusta tecla de atalho.
   *
   * @param action ao
   * @param stroke tecla
   */
  private void setKeyStroke(final AbstractAction action, final KeyStroke stroke) {
    final int mode = JComponent.WHEN_ANCESTOR_OF_FOCUSED_COMPONENT;
    final InputMap inputMap = rootPane.getInputMap(mode);
    final ActionMap actionMap = rootPane.getActionMap();
    final String actionMapKey = stroke.toString();
    inputMap.put(stroke, actionMapKey);
    actionMap.put(actionMapKey, action);
  }

  /**
   * Criao da ao de cancelamento.
   *
   * @return ao
   */
  private AbstractAction createOkAction() {
    final AbstractAction action = new AbstractAction() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        handleOkButton();
      }
    };
    final String text = getString("ok.button");
    action.putValue(Action.NAME, text);
    return action;
  }

  /**
   * Constri o painel de botes.
   *
   * @return o painel
   */
  private JPanel buildSelectionPanel() {
    final JLabel selectionLabel = new JLabel();
    selectionLabel.setText(getString("selection.label"));

    selectionText.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent ae) {
        okButton.doClick();
      }
    });

    selectionText.addKeyListener(new KeyAdapter() {
      @Override
      public void keyTyped(final KeyEvent e) {
        final char c = e.getKeyChar();
        boolean acceptKey = true;
        if (c == '\\') {
          acceptKey = false;
        }
        else {
          if (getClientFileType() == ClientFileType.REMOTE) {
            final boolean isLetterOrDigit = Character.isLetterOrDigit(c);
            if (!isLetterOrDigit && c != '_' && c != '.' && c != '/') {
              acceptKey = false;
            }
          }
        }
        if (!acceptKey) {
          final Toolkit toolkit = getToolkit();
          toolkit.beep();
          e.consume();
        }
      }
    });

    final JLabel filterLabel = new JLabel();
    filterLabel.setText(getString("filter.label"));

    filterCombo.setRenderer(new DefaultListCellRenderer() {
      @Override
      public Component getListCellRendererComponent(final JList list,
        final Object value, final int index, final boolean isSelected,
        final boolean cellHasFocus) {
        super.getListCellRendererComponent(list, value, index, isSelected,
          cellHasFocus);
        final ClientFileFilterInterface filter =
          (ClientFileFilterInterface) value;
        setText(filter.getDescription());
        setIcon(filter.getImageIcon());
        setFont(getFont().deriveFont(Font.PLAIN));
        return this;
      }
    });

    filterCombo.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        final ClientFileFilterInterface flt = getCurrentViewFilter();
        ClientFileChooser.this.internalSetViewFilter(flt);
      }
    });

    final JComponent[][] components =
      new JComponent[][] { { selectionLabel, selectionText },
          { filterLabel, filterCombo }, };

    final JPanel panel = GUIUtils.createBasicGridPanel(components);
    return panel;
  }

  /**
   * Constri o painel de botes.
   *
   * @return o painel
   */
  private JPanel buildToolPanel() {
    final AbstractAction filterAction = createFilterAction();
    filterToggle.setAction(filterAction);

    final AbstractAction refreshAction = createRefreshAction();
    final JButton refreshButton = new JButton(refreshAction);

    final AbstractAction upDirAction = createUpDirAction();
    final JButton upDirButton = new JButton(upDirAction);

    final AbstractAction createNewDirAction = createNewDirectoryAction();
    newDirectoryAction.setAction(createNewDirAction);

    final AbstractAction projectHomeDirAction = createProjectHomeDirAction();
    projHomeDirButton.setAction(projectHomeDirAction);

    final AbstractAction localHomeDirAction = createLocalHomeDirAction();
    localHomeDirButton.setAction(localHomeDirAction);

    final AbstractAction localRootDirAction = createLocalRootDirAction();
    localRootDirButton.setAction(localRootDirAction);

    final JPanel hierarchyPanel = buildHierarchyPanel();

    final JToolBar toolbar = new JToolBar(JToolBar.HORIZONTAL);
    toolbar.setFloatable(false);
    toolbar.addSeparator();
    toolbar.add(filterToggle);
    toolbar.add(refreshButton);
    toolbar.addSeparator();
    toolbar.add(upDirButton);
    toolbar.addSeparator();
    toolbar.add(newDirectoryAction);
    toolbar.add(projHomeDirButton);
    toolbar.add(localHomeDirButton);
    toolbar.add(localRootDirButton);

    final JPanel panel = new JPanel();
    panel.setLayout(new GridBagLayout());
    panel.add(hierarchyPanel, new GBC(0, 0).horizontal());
    panel.add(toolbar, new GBC(1, 0).none());
    return panel;
  }

  /**
   * cria o painel de hiearquia de diretrios.
   *
   * @return o painel
   */
  private JPanel buildHierarchyPanel() {
    final ClientFileParentComboChooseListener comboChooseListener =
      new ClientFileParentComboChooseListener() {
        @Override
        public void parentChosen(final ClientFileParentComboPanel panel,
          final ClientFile object) {
          if (object == null) {
            return;
          }
          setCurrentDirectory(object);
          refresh();
        }
      };
    hierarchyCombo.addParentsComboPanelChooseListener(comboChooseListener);

    final String hierarchyText = getString("hierarchy.label");
    hierarchyLabel.setText(hierarchyText);

    final JComponent[][] components =
      new JComponent[][] { { hierarchyLabel, hierarchyCombo }, };
    final JPanel hPanel = GUIUtils.createBasicGridPanel(components);
    return hPanel;
  }

  /**
   * Ao de toggle de filtro de arquivos ocultos.
   *
   * @return ao
   */
  private AbstractAction createFilterAction() {
    final AbstractAction action = new AbstractAction() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        showHiddenFiles(!isHiddenFilesShown());
      }
    };
    final ImageIcon icon = ChooserImages.FILTER_HIDDEN_IMG;
    action.putValue(Action.SMALL_ICON, icon);
    final String tooltip = getString("filter.hidden.tooltip");
    action.putValue(Action.SHORT_DESCRIPTION, tooltip);

    final int fKey = KeyEvent.VK_F;
    final KeyStroke stroke = KeyStroke.getKeyStroke(fKey, KeyEvent.CTRL_MASK);
    action.putValue(Action.ACCELERATOR_KEY, stroke);
    setKeyStroke(action, stroke);

    return action;
  }

  /**
   * Cria ao para ir ao home
   *
   * @return ao
   */
  private AbstractAction createLocalHomeDirAction() {
    final AbstractAction action = new AbstractAction() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        setLocalHomeDirectory();
      }
    };
    final ImageIcon icon = ChooserImages.LOCAL_HOME_ICON;
    action.putValue(Action.SMALL_ICON, icon);
    final String tooltip = getString("home.tooltip");
    action.putValue(Action.SHORT_DESCRIPTION, tooltip);
    return action;
  }

  /**
   * Cria ao para ir ao home
   *
   * @return ao
   */
  private AbstractAction createUpDirAction() {
    final AbstractAction action = new AbstractAction() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        setUpDirectory();
      }
    };
    final ImageIcon icon = ChooserImages.PARENT_FOLDER_IMG;
    action.putValue(Action.SMALL_ICON, icon);
    final String tooltip = getString("up.tooltip");
    action.putValue(Action.SHORT_DESCRIPTION, tooltip);

    final int up = KeyEvent.VK_UP;
    final KeyStroke stroke = KeyStroke.getKeyStroke(up, KeyEvent.ALT_MASK);
    action.putValue(Action.ACCELERATOR_KEY, stroke);
    setKeyStroke(action, stroke);
    return action;
  }

  /**
   * Cria ao para ir ao home
   *
   * @return ao
   */
  private AbstractAction createProjectHomeDirAction() {
    final AbstractAction action = new AbstractAction() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        setProjectHomeDirectory();
      }
    };
    final ImageIcon icon = ChooserImages.PROJECT_HOME_ICON;
    action.putValue(Action.SMALL_ICON, icon);
    final String tooltip = getString("project.tooltip");
    action.putValue(Action.SHORT_DESCRIPTION, tooltip);
    return action;
  }

  /**
   * Cria ao para criar um novo diretrio.
   *
   * @return ao
   */
  private AbstractAction createNewDirectoryAction() {
    final AbstractAction action = new AbstractAction() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        DirectoryCreationDialog.show(getOwner(), getCurrentDirectory());
        refresh();
      }
    };
    final ImageIcon icon = ChooserImages.NEW_DIRECTORY_ICON;
    action.putValue(Action.SMALL_ICON, icon);
    final String tooltip = getString("new.dir.tooltip");
    action.putValue(Action.SHORT_DESCRIPTION, tooltip);
    return action;
  }

  /**
   * Cria ao para ir ao home
   *
   * @return ao
   */
  private AbstractAction createLocalRootDirAction() {
    final AbstractAction action = new AbstractAction() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        setLocalRootDirectory();
      }
    };
    final ImageIcon icon = ChooserImages.LOCAL_ROOT_ICON;
    action.putValue(Action.SMALL_ICON, icon);
    final String tooltip = getString("root.tooltip");
    action.putValue(Action.SHORT_DESCRIPTION, tooltip);
    return action;
  }

  /**
   * Cria ao de refresh.
   *
   * @return ao
   */
  private AbstractAction createRefreshAction() {
    final AbstractAction action = new AbstractAction() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        refresh();
      }
    };
    final String refreshToolTip = getString("refresh.tooltip");
    action.putValue(Action.SHORT_DESCRIPTION, refreshToolTip);
    final ImageIcon icon = ApplicationImages.ICON_REFRESH_16;
    action.putValue(Action.SMALL_ICON, icon);
    final int f5 = KeyEvent.VK_F5;
    final KeyStroke stroke = KeyStroke.getKeyStroke(f5, 0);
    action.putValue(Action.ACCELERATOR_KEY, stroke);
    setKeyStroke(action, stroke);
    return action;
  }

  /**
   * Consulta corretude sobre eventual ambiguidade de tipo de arquivo.
   *
   * @param fileName nome do arquivo.
   *
   * @see #selectedSavePath
   *
   * @return indicativo
   */
  private boolean areTypesMatched(String fileName) {
    if (fileName != null && !fileName.isEmpty()) {
      final String fileExtension = FileUtils.getFileExtension(fileName);
      boolean isDirectory;
      switch (getSelectionMode()) {
      case DIRECTORIES_ONLY:
        isDirectory = true;
        break;
      case FILES_ONLY:
        isDirectory = false;
        break;
      default:
        String msg = String.format("O modo %s no foi tratado.",
          getSelectionMode());
        throw new BugException(msg);
      }
      final ProjectFileType extType =
        ProjectFileType.getProjectFileTypeFromExtension(fileExtension,
            isDirectory);

      if (getSelectionType() == ClientFileChooserType.SAVE) {
        final ProjectFileType fltType = getFileTypeFromSelectedFilter();
        if (fltType != null && extType != null && !extType.equals(fltType)) {
          final String fltDesc = fltType.getDescription();
          final String extDesc = extType.getDescription();
          final String tag = "diff.types.extension.confirmation";
          final String fltCode = fltType.getCode();
          final String extCode = extType.getCode();
          final String title = getTitle();
          final String msg = getString(tag, fileName, fltCode, fltDesc, extCode,
            extDesc);
          final int op = StandardDialogs.showYesNoDialog(this, title, msg);
          return (op == 0);
        }
      }
    }
    return true;
  }

  /**
   * Consulta sobreescrita de arquivo explcito.
   *
   * @param stringPath path (array).
   *
   * @return indicativo
   */
  private boolean canCloseAsUserConfirmedOverwrite(final String[] stringPath) {
    final String path = FileUtils.joinPath(stringPath);
    final String title = getTitle();
    final String tag = "overwrite.confirmation";
    final String msg = getString(tag, path);
    final int op = StandardDialogs.showYesNoDialog(this, title, msg);
    return (op == 0);
  }

  /**
   * Limpa seleo textual de arquivo.
   */
  void clearSelectionText() {
    selectionText.setText(null);
  }

  /**
   * Fecha o dilogo definindo o arquivo (simples) selecionado.
   *
   * @param selected o arquivo selecionado.
   */
  public void closeOnSingleFile(final ClientFile selected) {
    List<ClientFile> chosen = new ArrayList<ClientFile>();
    chosen.add(selected);
    setChosenItens(chosen);
    selectedSavePath = selected.getPath();
    handleOkButton();
  }

  /**
   * Mata o dilogo por erro interno.
   */
  private void die() {
    selectedSavePath = null;
    chosenItens = null;
    clearSelectionText();
    filterCombo.setEnabled(false);
    filterCombo.setModel(new DefaultComboBoxModel());
    hierarchyCombo.setLeafDirectory(null);
    hierarchyCombo.setEnabled(false);
    this.setVisible(false);
  }

  /**
   * Habilita o boto de Ok.
   *
   * @param flag indicativo
   */
  public void enableOkButton(final boolean flag) {
    okButton.setEnabled(flag);
  }

  /**
   * Habilita seleo textual de arquivo.
   *
   * @param flag indicativo
   */
  void enableSelectionText(final boolean flag) {
    selectionText.setEditable(flag);
  }

  /**
   * Consulta do modo de escolha de nico elemento.
   *
   * @return modo.
   */
  final public ClientFileChooserCardinality getCardinality() {
    return tablePanel.getCardinality();
  }

  /**
   * Consulta de diretrio.
   *
   * @return o diretrio
   */
  final public List<ClientFile> getChosenItens() {
    return chosenItens;
  }

  /**
   * Consulta de diretrio.
   *
   * @return o diretrio
   */
  final public ClientFile getCurrentDirectory() {
    final ClientFile rootObject = tablePanel.getRootDirectory();
    if (rootObject == null) {
      return null;
    }
    return rootObject;
  }

  /**
   * Mostra o filtro de visualizao corrente.
   *
   * @return o filtro.
   */
  final public ClientFileFilterInterface getCurrentViewFilter() {
    final Object selected = filterCombo.getSelectedItem();
    final ClientFileFilterInterface flt = (ClientFileFilterInterface) selected;
    return flt;
  }

  /**
   * Busca um tipo de arquivo com base em um nome.
   *
   * @param fileName nome de arquivo.
   * @return o tipo.
   */
  final public ProjectFileType getFileTypeFromFileName(final String fileName) {
    boolean isDirectory;
    switch (getSelectionMode()) {
    case DIRECTORIES_ONLY:
      isDirectory = true;
      break;
    case FILES_ONLY:
      isDirectory = false;
      break;
    default:
      String message = String.format("O modo %s no foi tratado.", getSelectionMode());
      throw new BugException(message);
    }
    final String extension = FileUtils.getFileExtension(fileName);
    final ProjectFileType tp =
      ProjectFileType.getProjectFileTypeFromExtension(extension, isDirectory);
    return tp;
  }

  /**
   * Consulta o tipo de arquivo proposto com base no filtro escolhido.
   *
   * @return o tipo (ou {@code null} se isso no puder ser deduzido).
   */
  final public ProjectFileType getFileTypeFromSelectedFilter() {
    final ClientFileFilterInterface filter = getSelectedFilter();
    if (filter != null && filter.inferFileType() != null) {
      return filter.inferFileType();
    }
    return null;
  }

  /**
   * Consulta o projeto corrente, matandoo dilogo se houver erro de localizao
   * do projeto.
   *
   * @return o projeto
   */
  private CommonClientProject getProjectOrDie() {
    final DesktopFrame desktopFrame = DesktopFrame.getInstance();
    if (desktopFrame == null) {
      final String tag = "no.desktop.detected.error";
      final String msg = getString(tag);
      showErrorMessage(msg);
      die();
      return null;
    }

    final CommonClientProject project = desktopFrame.getProject();
    if (project == null) {
      final String tag = "no.project.detected.error";
      final String msg = getString(tag);
      showErrorMessage(msg);
      die();
    }
    return project;
  }

  /**
   * Consulta o filtro selecionado (ativo).
   *
   * @return o filtro
   */
  final public ClientFileFilterInterface getSelectedFilter() {
    final Object selected = filterCombo.getSelectedItem();
    return (ClientFileFilterInterface) selected;
  }

  /**
   * Ajusta o filtro corrente a ser usado.
   *
   * @param filter o filtro.
   */
  final public void setSelectedFilter(final ClientFileFilterInterface filter) {
    filterCombo.setSelectedItem(filter);
  }

  /**
   * Retorna o path selecionado em modo SAVE.
   *
   * @return selectedSavePath
   */
  public final String[] getSelectedSavePath() {
    return selectedSavePath;
  }

  /**
   * Retorna o path selecionado em modo SAVE com separadores
   *
   * @param sep separador
   * @return o texto
   */
  public final String getSelectedSavePath(final String sep) {
    String result = "";
    if (selectedSavePath == null) {
      return null;
    }
    final int length = selectedSavePath.length;
    for (int i = 0; i < length - 1; i++) {
      result = result + selectedSavePath[i] + sep;
    }
    final String fileName = selectedSavePath[length - 1];
    return result + fileName;
  }

  /**
   * Consulta ao modo de seleo para arquivos.
   *
   * @return modo
   */
  final public ClientFileChooserSelectionMode getSelectionMode() {
    return tablePanel.getSelectionMode();
  }

  /**
   * Consulta do tipo de operao
   *
   * @return indicativo
   */
  final public ClientFileChooserType getSelectionType() {
    return type;
  }

  /**
   * Consulta uma string internacionalizada.
   *
   * @param tag tag
   * @param args argumentos.
   * @return o texto
   */
  final private String getString(final String tag, final Object... args) {
    final String prefix = this.getClass().getSimpleName();
    final String fmt = LNG.get(prefix + "." + tag);
    final String text = String.format(fmt, args);
    return text;
  }

  /**
   * Consulta a lista de filtros disponveis.
   *
   * @return a lista
   */
  final public List<ClientFileFilterInterface> getViewFilters() {
    final List<ClientFileFilterInterface> list =
      new ArrayList<ClientFileFilterInterface>();
    list.addAll(viewFilters);
    Collections.unmodifiableList(list);
    return list;
  }

  /**
   * Tratamento do boto de cancelamento.
   */
  private void handleCancelButton() {
    setChosenItens(null);
    selectedSavePath = null;
    ClientFileChooser.this.setVisible(false);
  }

  /**
   * Tratamento do boto de OK.
   */
  private void handleOkButton() {
    final ClientFileChooserType tp = getSelectionType();
    final boolean isToClose;
    switch (tp) {
      case SAVE:
        isToClose = handleOkSaveType();
        break;
      case OPEN:
        isToClose = handleOkOpenType();
        break;
      default:
        isToClose = false;
    }
    ClientFileChooser.this.setVisible(!isToClose);
  }

  /**
   * Tratador do boto de Ok em modo OPEN.
   *
   * @return indicativo de fechamento
   */
  private boolean handleOkOpenType() {
    String selectedText = selectionText.getText();
    if (!selectedText.trim().isEmpty()) {
      String[] paths = parsePaths(selectedText);
      List<ClientFile> selectedFiles = new ArrayList<ClientFile>();
      ClientFile parent = null;
      for (String path : paths) {
        ClientFile selectedFile = getFileFromPath(path);
        if (selectedFile != null) {
          String fileName = selectedFile.getName();
          if (!areTypesMatched(fileName)) {
            return false;
          }
          ClientFileChooserSelectionMode selectionMode = getSelectionMode();
          ClientFileFilterInterface filter = getSelectedFilter();
          if (selectedFile.isDirectory()) {
            switch (selectionMode) {
            case FILES_ONLY:
              setCurrentDirectory(selectedFile);
              return false;
            case DIRECTORIES_ONLY:
              if (!filter.accept(selectedFile, Operation.SELECTION)) {
                setCurrentDirectory(selectedFile);
                return false;
              }
              break;
            default:
              String msg = String.format("Modo invlido: %s.", selectionMode);
              throw new IllegalStateException(msg);
            }
          }
          else if (selectionMode == ClientFileChooserSelectionMode.DIRECTORIES_ONLY) {
            final String tag = "not.dir.error";
            final String msg = getString(tag, selectedFile.getName());
            showErrorMessage(msg);
            return false;
          }

          if (parent == null) {
            parent = selectedFile.getParent();
          }
          else {
            if (!parent.equals(selectedFile.getParent())) {
              final String tag = "different.parent.error";
              final String msg = getString(tag);
              showErrorMessage(msg);
              return false;
            }
          }
          selectedFiles.add(selectedFile);
        }
      }
      if (parent != null) {
        setCurrentDirectory(parent);
      }
      setSelection(selectedFiles);
      if (selectedFiles.size() > 0 && selectedFiles.size() == paths.length) {
        return true;
      }
    }
    return false;
  }

  /**
   * Analisa o texto para obter o(s) caminho(s) selecionado(s).
   *
   * @param text o texto que contm o(s) caminho(s).
   * @return o conjunto de caminhos.
   */
  private String[] parsePaths(String text) {
    ClientFileChooserCardinality cardinality = getCardinality();
    String separator = MULTIPLE_PATHS_SEPARATOR;
    String[] paths;
    if (cardinality == ClientFileChooserCardinality.MULTIPLE_CHOOSE
      && text.contains(separator)) {
      paths = text.split(separator);
    }
    else {
      paths = new String[] { text };
    }
    return paths;
  }

  /**
   * Obtm o arquivo no caminho especificado a partir do diretrio corrente.
   *
   * @param path o caminho.
   * @return o arquivo ou nulo, se no existir o arquivo.
   */
  private ClientFile getFileFromPath(String path) {
    String[] fileName = parsePath(path);
    ClientFile currentDir;
    if (FileUtils.isAbsolutePath(path)) {
      /*
       * O caminho escolhido  absoluto: os caminhos devem ser avaliados 
       * partir da raiz do sistema de arquivos em questo. Precisamos procurar o
       * caminho a partir da raiz para a busca de arquivos funcionar.
       */
      currentDir = getRoot();
    }
    else {
      currentDir = getCurrentDirectory();
    }
    ClientFile file = findFile(currentDir, fileName);
    if (file == null) {
      /*
       * O caminho escolhido  um arquivo no existente.
       */
      final String msg = getString("path.not.found.warning", path);
      StandardDialogs.showWarningDialog(this, getTitle(), msg);
    }
    return file;
  }

  /**
   * Obtm a raiz do sistema de arquivos atual.
   *
   * @return a raiz do sistema de arquivos.
   */
  private ClientFile getRoot() {
    ClientFileType clientFileType = getClientFileType();

    ClientFile root = null;
    switch (clientFileType) {
      case REMOTE:
        root = ClientFileChooserUtils.getProjectHomeDirectory();
        break;
      case LOCAL:
        root =
          ClientFileChooserUtils.getLocalRootDirectory(getCurrentDirectory());
        break;
      case SGA:
        ClientFile currDir = getCurrentDirectory();
        if ("/".equals(currDir.getStringPath())) {
          root = currDir;
        }
        else {
          root = new ClientSGAFile(((ClientSGAFile) currDir).getSGAName(), "/");
        }
        break;
    }
    return root;
  }

  /**
   * Retorna o tipo de arquivo gerenciado por este chooser.
   *
   * @see ClientFileType
   * @return tipo de arquivo gerenciado por este chooser.
   */
  private ClientFileType getClientFileType() {
    ClientFile currentDirectory = getCurrentDirectory();
    return currentDirectory.getClientFileType();
  }

  /**
   * Obtm o arquivo no caminho especificado a partir do diretrio base. Retorna
   * nulo caso no encontre o arquivo.
   *
   * @param baseDir Diretrio base para busca do caminho.
   * @param path o caminho pro arquivo.
   * @return o arquivo ou nulo, caso este no exista.
   */
  private ClientFile findFile(ClientFile baseDir, String[] path) {
    if (path == null) {
      return null;
    }
    if (path.length == 1 && path[0].equals(".")) {
      return baseDir;
    }
    if (getClientFileType() == ClientFileType.LOCAL) {
      File file = new File(baseDir.getStringPath(), FileUtils.joinPath(path));
      if (!file.exists()) {
        return null;
      }
      return new ClientLocalFile(file);
    }

    if (getClientFileType() == ClientFileType.SGA) {
      ClientSGAFile file = (ClientSGAFile) baseDir;

      String sgaPath = buildSGAPath(file, path);
      try {
        SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
        return sgaService.getFile(file.getSGAName(), sgaPath);
      }
      catch (RemoteException e) {
        return null;
      }
    }

    ClientFile[] children;
    ClientFile currentFile = baseDir;
    try {
      for (int i = 0; i < path.length; i++) {
        if (currentFile == null) {
          return null;
        }
        if (currentFile.isDirectory()) {
          children = currentFile.getChildren();
          if (children == null) {
            return null;
          }
        }
        else {
          return null;
        }
        currentFile = null;
        for (ClientFile child : children) {
          if (child.getName().equals(path[i])) {
            currentFile = child;
            break;
          }
        }
      }
      return currentFile;
    }
    catch (PermissionException pe) {
      return null;
    }
    catch (Exception e) {
      final String title = getTitle();
      String message = e.getMessage();
      StandardErrorDialogs.showErrorDialog(this, title, message, e);
      return null;
    }
  }

  /**
   * Constri o path de um arquivo ou diretrio de um SGA.
   *
   * @param baseDir diretrio base.
   * @param path path.
   * @return path de um arquivo ou diretrio.
   */
  private String buildSGAPath(ClientSGAFile baseDir, String[] path) {
    String basePath = baseDir.getStringPath();
    String separator = baseDir.getSeparator();
    String relativePath = FileUtils.joinPath(separator, path);

    StringBuilder sgaPath = new StringBuilder();
    sgaPath.append(basePath);
    if (!relativePath.startsWith(separator) && !basePath.endsWith(separator)) {
      sgaPath.append(separator);
    }
    sgaPath.append(relativePath);
    return sgaPath.toString();
  }

  /**
   * Analisa o caminho em formato string, quebrando-o em um array contendo seus
   * componentes.
   *
   * @param path caminho em formato string.
   *
   * @return o array ou nulo caso no seja possvel separar os componentes do
   *         caminho.
   */
  private String[] parsePath(String path) {
    if (path != null) {
      String filePath = path.trim();
      if (!filePath.isEmpty()) {
        String[] splitPath = FileUtils.splitPath(filePath, '/');
        return splitPath;
      }
    }
    return null;
  }

  /**
   * Tratador do boto de Ok em modo SAVE
   *
   * @return indicativo de fechamento
   */
  private boolean handleOkSaveType() {
    String text = selectionText.getText();
    String[] path = parsePath(text);

    if (path == null) {
      return false;
    }

    ClientFile currentDir;
    if (FileUtils.isAbsolutePath(text)) {
      /*
       * O caminho escolhido  absoluto: os caminhos devem ser avaliados 
       * partir da raiz do sistema de arquivos em questo. Precisamos colocar a
       * raiz como diretrio corrente para a busca de arquivos funcionar.
       */
      currentDir = getRoot();
    }
    else {
      currentDir = getCurrentDirectory();
    }
    ClientFile file = findFile(currentDir, path);
    if (file == null) {
      /*
       * O caminho escolhido  um arquivo no existente.
       */
      if (path.length > 1) {
        /*
         * O caminho contm um diretrio-pai: precisamos verificar se o pai
         * existe para evitar erros ao salvar o novo arquivo.
         */
        ClientFile dir = findParent(currentDir, path);
        if (dir == null) {
          String tag = "parent.not.found.warning";
          final String msg = getString(tag, text);
          StandardDialogs.showWarningDialog(this, getTitle(), msg);
          return false;
        }
        else {
          if (!dir.canWrite()) {
            showPermissionError();
            return false;
          }
        }
      }
      final String[] dirPath = currentDir.getPath();
      selectedSavePath = appendPath(dirPath, path);
      final String fileName = selectedSavePath[selectedSavePath.length - 1];
      final boolean matchedTypes = areTypesMatched(fileName);
      return matchedTypes;
    }

    ClientFileFilterInterface filter = getSelectedFilter();
    ClientFileChooserSelectionMode selectionMode = getSelectionMode();
    if (file.isDirectory()) {
      switch (selectionMode) {
      case FILES_ONLY:
        setCurrentDirectory(file);
        return false;
      case DIRECTORIES_ONLY:
        if (!filter.accept(file, Operation.SELECTION)) {
          setCurrentDirectory(file);
          return false;
        }
        break;
      default:
        String msg = String.format("Modo invlido: %s.", selectionMode);
        throw new IllegalStateException(msg);
      }
    }
    else if (selectionMode == ClientFileChooserSelectionMode.DIRECTORIES_ONLY) {
      final String tag = "not.dir.error";
      final String msg = getString(tag, file.getName());
      showErrorMessage(msg);
      return false;
    }

    /*
     * O caminho escolhido  um arquivo existente: precisamos confirmar se pode
     * ser sobrescrito.
     */
    if (!file.canWrite()) {
      showPermissionError();
      return false;
    }
    final String fileName;
    if (text.equals(".")) {
      selectedSavePath = path;
      fileName = currentDir.getName();
    }
    else {
      final String[] dirPath = currentDir.getPath();
      selectedSavePath = appendPath(dirPath, path);
      fileName = selectedSavePath[selectedSavePath.length - 1];
    }
    if (!areTypesMatched(fileName)) {
      return false;
    }
    if (!canCloseAsUserConfirmedOverwrite(selectedSavePath)) {
      return false;
    }
    return true;
  }

  /**
   * Encontra o diretrio pai de um determinado caminho de arquivo a partir do
   * diretrio base. Retorna nulo caso no encontre o diretrio pai.
   *
   * @param baseDir Diretrio base para busca do caminho.
   * @param path caminho do arquivo-filho.
   * @return o diretrio pai ou nulo caso no seja encontrado.
   */
  private ClientFile findParent(ClientFile baseDir, String[] path) {
    String[] parent = new String[path.length - 1];
    System.arraycopy(path, 0, parent, 0, path.length - 1);
    ClientFile dir = findFile(baseDir, parent);
    return dir;
  }

  /**
   * Inicializao.
   */
  private void initComponent() {
    setAllFilter();

    final ClientFileChooserActionListener mlst =
      new ClientFileChooserActionListener(this);
    tablePanel.addTablePanelActionListener(mlst);

    final ClientFileChooserSelectionListener slst =
      new ClientFileChooserSelectionListener(this);
    tablePanel.addTablePanelSelectionListener(slst);
    BorderUtil.setLowBorder(tablePanel);

    final JPanel selectionPanel = buildSelectionPanel();
    final JPanel buttonsPanel = buildButtonsPanel();
    final JPanel toolPanel = buildToolPanel();

    final JPanel southPanel = new JPanel();
    southPanel.setLayout(new BorderLayout());
    southPanel.add(selectionPanel, BorderLayout.NORTH);
    southPanel.add(buttonsPanel, BorderLayout.SOUTH);

    final Container contentPane = getContentPane();
    contentPane.setLayout(new BorderLayout());
    contentPane.add(tablePanel, BorderLayout.CENTER);
    contentPane.add(toolPanel, BorderLayout.NORTH);
    contentPane.add(southPanel, BorderLayout.SOUTH);

    final Dimension size = new Dimension(600, 400);
    setPreferredSize(size);
    setMinimumSize(size);

    setLocalHomeButtonVisible(true);
    setLocalRootButtonVisible(true);
    setProjectHomeButtonVisible(true);

    initCurrentDirectory();
    setChosenItens(null);
    setSelectionMode(ClientFileChooserSelectionMode.FILES_ONLY);
    setSelectionType(ClientFileChooserType.OPEN);
    setCardinality(ClientFileChooserCardinality.SINGLE_CHOOSE);
    setSelection(null);
    pack();
    setLocationRelativeTo(getParent());
  }

  /**
   * Inicializa com o diretrio corrente.
   */
  private void initCurrentDirectory() {
    final DesktopFrame desktopFrame = DesktopFrame.getInstance();
    if (desktopFrame == null) {
      final String tag = "no.desktop.detected.error";
      final String msg = getString(tag);
      showErrorMessage(msg);
      die();
      return;
    }

    final CommonClientProject project = desktopFrame.getProject();
    if (project == null) {
      setLocalHomeDirectory();
    }
    else {
      setProjectHomeDirectory();
    }
  }

  /**
   * Ajusta o filtro corrente de visualizao.
   *
   * @param filter o filtro.
   */
  private void internalSetViewFilter(final ClientFileFilterInterface filter) {
    if(getSelectionType() == ClientFileChooserType.OPEN) {
      setChosenItens(null);
    }
    tablePanel.setFilter(filter);
  }

  /**
   * Indica se arquivos ocultos devem ser vistos.
   *
   * @return indicativo
   */
  public final boolean isHiddenFilesShown() {
    return tablePanel.isHiddenFilesShown();
  }

  /**
   * Atualiza lista de dados exibidos.
   */
  final public void refresh() {
    setCurrentDirectory(getCurrentDirectory());
  }

  /**
   * Ajuste para filtro total.
   */
  private void setAllFilter() {
    final List<ClientFileFilterInterface> all =
      new ArrayList<ClientFileFilterInterface>();
    all.add(new ClientFileAllFilter());
    setViewFilters(all);
  }

  /**
   * Ajuste do modo de seleo simples.
   *
   * @param flag indicativo
   */
  final public void setCardinality(final ClientFileChooserCardinality flag) {
    final ClientFileChooserType tp = getSelectionType();
    if (tp == ClientFileChooserType.SAVE) {
      return;
    }
    tablePanel.setCardinality(flag);
  }

  /**
   * Ajuste da escolha atual do usurio.
   *
   * @param list a lista
   */
  public void setChosenItens(final List<ClientFile> list) {
    if (list != null) {
      for (ClientFile file : list) {
        if (!file.canRead()) {
          showPermissionError();
          chosenItens = null;
          return;
        }
      }
    }
    chosenItens = list;
    updateSelectionText(chosenItens);
  }

  /**
   * Ajuste de diretrio
   *
   * @param directory diretrio.
   */
  final public void setCurrentDirectory(final ClientFile directory) {
    if (!directory.canExecute() || !directory.canRead()) {
      showPermissionError();
      return;
    }
    tablePanel.setSelection(null);
    tablePanel.setRootDirectory(directory);
    hierarchyCombo.setLeafDirectory(directory);
    clearSelectionText();
    setChosenItens(null);
    selectedSavePath = null;
  }

  /**
   * Mostra janela com erro de permisso de acesso ao arquivo/diretrio.
   */
  public void showPermissionError() {
    StandardErrorDialogs.showErrorDialog(getOwner(), new PermissionException());
  }

  /**
   * Ajuste de visibilidade (disponibilidade) de acesso ao home local.
   *
   * @param visible indicativo de visibilidade
   */
  public void setLocalHomeButtonVisible(final boolean visible) {
    localHomeDirButton.setVisible(visible);
  }

  /**
   * Ajusta diretrio corrente para raiz do projeto.
   */
  final public void setLocalHomeDirectory() {
    ClientLocalFile home = ClientFileChooserUtils.getLocalHomeDirectory();
    if (home != null) {
      setCurrentDirectory(home);
    }
  }

  /**
   * Ajuste de visibilidade (disponibilidade) de acesso ao root local.
   *
   * @param visible indicativo de visibilidade
   */
  public void setLocalRootButtonVisible(final boolean visible) {
    localRootDirButton.setVisible(visible);
  }

  /**
   * Ajusta diretrio corrente para raiz local.
   */
  final public void setLocalRootDirectory() {
    ClientLocalFile root =
      ClientFileChooserUtils.getLocalRootDirectory(getCurrentDirectory());
    if (root != null) {
      setCurrentDirectory(root);
    }
  }

  /**
   * Ajuste de visibilidade (disponibilidade) de acesso ao home do projeto.
   *
   * @param visible indicativo de visibilidade
   */
  public void setProjectHomeButtonVisible(final boolean visible) {
    projHomeDirButton.setVisible(visible);
  }

  /**
   * Ajusta diretrio corrente para raiz do projeto.
   */
  final public void setProjectHomeDirectory() {
    final CommonClientProject project = getProjectOrDie();
    final ClientFile home = project.getRoot();
    if (home != null) {
      setCurrentDirectory(home);
    }
  }

  /**
   * Ajuste de seleo.
   *
   * @param newSelection seleo.
   */
  public void setSelection(final List<ClientFile> newSelection) {
    if (newSelection == null) {
      final ArrayList<ClientFile> empty = new ArrayList<ClientFile>();
      setSelection(empty);
      return;
    }
    tablePanel.setSelection(newSelection);
  }

  /**
   * Ajuste do modo de seleo.
   *
   * @param mode modo
   */
  final public void setSelectionMode(final ClientFileChooserSelectionMode mode) {
    final ClientFileChooserType tp = getSelectionType();
    if (tp == ClientFileChooserType.SAVE) {
      return;
    }
    tablePanel.setSelectionMode(mode);
  }

  /**
   * Ajuste do tipo.
   *
   * @param type o tipo.
   */
  final public void setSelectionType(final ClientFileChooserType type) {
    switch (type) {
      case OPEN:
        break;
      case SAVE:
        setCardinality(ClientFileChooserCardinality.SINGLE_CHOOSE);
        break;
    }
    this.type = type;
  }

  /**
   * Ajusta o diretrio corrente para cima.
   */
  private void setUpDirectory() {
    final ClientFile dir = getCurrentDirectory();
    if (dir == null) {
      return;
    }
    final ClientFile parent = dir.getParent();
    if (parent == null) {
      return;
    }
    setCurrentDirectory(parent);
  }

  /**
   * Ajusta o filtro corrente de visualizao.
   *
   * @param filter o filtro.
   */
  final public void setViewFilters(final ClientFileFilterInterface filter) {
    final List<ClientFileFilterInterface> filters =
      new ArrayList<ClientFileFilterInterface>();
    filters.add(filter);
    setViewFilters(filters);
  }

  /**
   * Ajusta os filtros de visualizao disponveis.
   *
   * @param filters os filtros.
   */
  final public void setViewFilters(final ClientFileFilterInterface[] filters) {
    final List<ClientFileFilterInterface> list = Arrays.asList(filters);
    setViewFilters(list);
  }

  /**
   * Ajusta os filtros de visualizao disponveis.
   *
   * @param filters os filtros.
   */
  final public void setViewFilters(final List<ClientFileFilterInterface> filters) {
    viewFilters.clear();
    if (filters == null || filters.size() <= 0) {
      setAllFilter();
      return;
    }
    viewFilters.addAll(filters);
    ComboUtil.updateFromList(filterCombo, viewFilters, 0);
    final Object sel = filterCombo.getSelectedItem();
    internalSetViewFilter((ClientFileFilterInterface) sel);
    tablePanel.refresh();
  }

  /**
   * Exibe mensagem de erro.
   *
   * @param msg mensagem
   */
  private void showErrorMessage(final String msg) {
    StandardDialogs.showErrorDialog(this, getTitle(), msg);
  }

  /**
   * Ajusta indicao que arquivos ocultos devem ser vistos.
   *
   * @param flag indicatvo
   */
  public final void showHiddenFiles(final boolean flag) {
    tablePanel.showHiddenFiles(flag);
    filterToggle.setSelected(flag);
  }

  /**
   * Ajuste do texto da da seleo
   *
   * @param selections a lista de selees.
   */
  void updateSelectionText(final List<ClientFile> selections) {
    if (selections == null) {
      clearSelectionText();
      return;
    }
    StringBuilder builder = new StringBuilder();
    String sep = "";
    for (final ClientFile sel : selections) {
      final String path;
      if (sel.equals(getCurrentDirectory())) {
        path = ".";
      }
      else {
        path = sel.getName();
      }
      builder.append(sep);
      builder.append(path);
      sep = MULTIPLE_PATHS_SEPARATOR + " ";
    }
    String text = builder.toString().trim();
    selectionText.setText(text.trim());
  }

  /**
   * Construtor.
   *
   * @param window a janela.
   */
  public ClientFileChooser(final Window window) {
    super(window, "", ModalityType.DOCUMENT_MODAL);
    initComponent();
  }

}
