package csbase.client.applications.projectsmanager.actions;

import java.util.ArrayList;
import java.util.List;

import javax.swing.Action;
import javax.swing.ImageIcon;

import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.projectsmanager.ProjectsManager;
import csbase.client.applications.projectsmanager.actions.core.ProjectsManagerAction;
import csbase.client.applications.projectsmanager.dialogs.StatusDialog;
import csbase.client.applications.projectsmanager.models.ProjectSpaceAllocation;
import csbase.client.applications.projectsmanager.models.ProjectsManagerData;
import csbase.client.applications.projectsmanager.models.ProjectsManagerScope;
import csbase.client.applications.projectsmanager.proxy.SetProjectSharingStatsTask;
import csbase.logic.ProjectPermissions.SharingType;
import csbase.logic.User;

/**
 * Ao que altera o status de compartilhamento de projetos para modo "pblico".
 * 
 * @author jnlopes
 * 
 */
public class SetProjectPublicAction extends ProjectsManagerAction {

  /**
   * Modo de acesso (ALL_RO ou ALL_RW).
   */
  private SharingType sharingType;

  /**
   * Construtor
   * 
   * @param projectsManager A aplicao
   * @param readOnly Boolean que indica se os projetos sero RO ou RW.
   */
  public SetProjectPublicAction(final ProjectsManager projectsManager,
    final boolean readOnly) {
    super(projectsManager);

    if (readOnly) {
      sharingType = SharingType.ALL_RO;
    }
    else {
      sharingType = SharingType.ALL_RW;
    }
    setRightName();
  }

  /**
   * Define o nome da ao em funo do modo de acesso.
   */
  private void setRightName() {
    if (sharingType == SharingType.ALL_RO) {
      putValue(Action.NAME, getString("SetProjectPublicAction.readonly.name"));
    }
    else {
      putValue(Action.NAME, getString("SetProjectPublicAction.readwrite.name"));
    }
  }

  /**
   * {@inheritDoc} Mtodo que invoca a remote task que efetivamente altera o
   * status de compartilhamento de projetos para modo "pblico".
   */
  @Override
  public void actionDone() throws Exception {
    ProjectsManager projectsManager = getProjectsManager();
    List<ProjectsManagerData> projectList =
      projectsManager.getSelectedProjects();
    if (projectList == null || projectList.size() <= 0) {
      return;
    }

    final int confirm =
      StandardDialogs.showYesNoDialog(projectsManager.getApplicationFrame(),
        getString("SetProjectPublicAction.warning.title"),
        getString("SetProjectPublicAction.warning.msg"));

    if (confirm != StandardDialogs.YES_OPTION) {
      return;
    }

    final List<ProjectsManagerData> projects = validateProjects(projectList);
    if (projects == null) {
      return;
    }

    final boolean result = runTask(projects);
    if (result) {
      for (ProjectsManagerData pmd : projects) {
        pmd.setSharingType(sharingType);
      }
      projectsManager.refreshProjectsTable();
      projectsManager.refreshInfoPanel();
    }
    else {
      final String errMsg =
        getString("SetProjectPublicAction.set.public.error.message");
      final String errTitle =
        getString("SetProjectPublicAction.set.public.error.title");
      StandardDialogs.showErrorDialog(projectsManager.getApplicationFrame(),
        errTitle, errMsg);
    }
  }

  /**
   * Roda e trata a execuo da task.
   * 
   * @param projects Lista de projetos vlidos para a operao.
   * @return Boolean indicando se a operao foi bem sucedida.
   * @throws Exception em caso de erro.
   */
  private boolean runTask(List<ProjectsManagerData> projects) throws Exception {
    final ProjectsManager projectsManager = getProjectsManager();
    final SetProjectSharingStatsTask fptt =
      new SetProjectSharingStatsTask(projectsManager, projects,
        ProjectsManagerScope.PUBLIC, sharingType);
    final ApplicationFrame frame = projectsManager.getApplicationFrame();
    fptt.execute(frame, getString("SetProjectPublicAction.title"),
      getString("SetProjectPublicAction.message"));
    if (fptt.wasCancelled()) {
      final String err = getString("SetProjectPublicAction.cancelled.message");
      fptt.showError(err);
      return false;
    }
    if (fptt.getStatus() != true) {
      final Exception exception = fptt.getError();
      throw exception;
    }
    return fptt.getResult();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ImageIcon getStandardImageIcon() {
    return ProjectsManagerScope.PUBLIC.getIcon();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  protected List<ProjectsManagerData> validateProjects(
    List<ProjectsManagerData> projectList) {

    final List<ProjectsManagerData> prjList =
      new ArrayList<ProjectsManagerData>();
    prjList.addAll(projectList);

    final List<ProjectsManagerData> failedProjects =
      new ArrayList<ProjectsManagerData>();

    final User loggedUser = User.getLoggedUser();
    final Object loggedUserId = loggedUser.getId();

    int i = 0;
    while (i < prjList.size()) {
      final ProjectsManagerData pmd = prjList.get(i);
      final ProjectSpaceAllocation spaceAllocation =
        pmd.getProjectSpaceAllocation();

      final boolean isAdmin = loggedUser.isAdmin();
      final boolean isOwner = pmd.getOwnerId().equals(loggedUserId);

      /*
       * Para que um usurio possa tornar um projeto pblico  preciso que: - O
       * projeto possa ser aberto (no esteja aguardando alocao /
       * desalocao). - O usurio seja o administrador ou o dono do projeto.
       */
      if (spaceAllocation.isOpenable() && (isAdmin || isOwner)) {
        i++;
      }
      else {
        failedProjects.add(pmd);
        prjList.remove(i);
      }
    }

    final ProjectsManager projectsManager = getProjectsManager();
    if (prjList.size() == 0) {
      final String deniedMessage =
        getString("SetProjectPublicAction.project.selection.denied.message")
          + getString("SetProjectPublicAction.project.requirements.message");
      StatusDialog.showNoneOkDialog(projectsManager, deniedMessage,
        failedProjects,
        getString("SetProjectPublicAction.project.selection.failure.message"));
      return null;
    }
    else if (failedProjects.size() > 0) {
      final String someDeniedMessage =
        getString("SetProjectPublicAction.project.selection.some.denied.message")
          + getString("SetProjectPublicAction.project.requirements.message");
      final int res =
        StatusDialog
          .showSomeOkDialog(
            projectsManager,
            someDeniedMessage,
            prjList,
            getString("SetProjectPublicAction.project.selection.some.failure.message"));
      if (res == 0) {
        return null;
      }
    }

    return prjList;
  }

}
