/*
 * $Id$
 */
package csbase.client.applications.htmlviewer;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.net.URL;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JEditorPane;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.JToolBar;
import javax.swing.ScrollPaneConstants;
import javax.swing.event.HyperlinkEvent;
import javax.swing.event.HyperlinkListener;
import javax.swing.text.html.HTMLDocument;
import javax.swing.text.html.HTMLFrameHyperlinkEvent;

import tecgraf.javautils.gui.GUIUtils;
import csbase.client.applications.ApplicationAboutAction;
import csbase.client.applications.ApplicationExitAction;
import csbase.client.applications.ApplicationFrame;
import csbase.client.applications.ApplicationImages;
import csbase.client.applications.ApplicationProject;
import csbase.client.desktop.DesktopFrame;
import csbase.client.desktop.LocalTask;
import csbase.client.remote.srvproxies.HttpProxy;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.ClientProjectFile;
import csbase.logic.CommonClientProject;

/**
 * A classe <code>HtmlViewer</code> implementa o dilogo de exibio de arquivos
 * HTML.
 * 
 * @author Tecgraf/PUC-Rio
 */
public class HtmlViewer extends ApplicationProject {
  /**
   * Tipo de arquivo padro do programa
   */
  private final String FILE_TYPE = "HTML";

  /**
   * Painel onde ser exibida a pgina html.
   */
  private JEditorPane htmlPane = null;

  /**
   * Painel onde ser exibida a pgina html.
   */
  private JScrollPane htmlScrolledPane = null;

  /**
   * Dilogo principal do visualizador.
   */
  private ApplicationFrame mainFrame = null;

  /**
   * Botes da ToolBar.
   */
  private JButton openButton = null;

  /**
   * Back
   */
  private JButton backButton = null;

  /**
   * Forward
   */
  private JButton forwardButton = null;

  /**
   * Rtulo do endereo corrente.
   */
  private JTextField urlText = null;

  /**
   * Histrico de URL's visitadas
   */
  private Vector<URL> history = null;

  /** Apontador para a pgina corrente dentro do histrico. */
  private int histPnt = 0;

  /**
   * Abertura de arquivo.
   */
  private void actionOpen() {
    try {
      final ClientProjectFile newFile = browseFileOpen(FILE_TYPE);
      openFile(newFile);
    }
    catch (final Exception e) {
      StandardErrorDialogs.showErrorDialog(mainFrame,
        getString("HTML_VIEWER_TITLE"), getString("HTML_VIEWER_ERROR"), e);
    }
  }

  /**
   * Limpa as informaes da URL
   */
  private void clearURL() {
    htmlPane.setText("");
    urlText.setText("");
    backButton.setEnabled(false);
    forwardButton.setEnabled(false);
  }

  /**
   * Montagem do menu de arquivos.
   * 
   * @return .
   */
  private JMenu createFileMenu() {
    final JMenu fileMenu = new JMenu(getString("HTML_VIEWER_MENU_FILE"));
    final JMenuItem openItem = fileMenu.add(getString("HTML_VIEWER_FILE_OPEN"));
    openItem.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        actionOpen();
      }
    });
    openItem.setIcon(ApplicationImages.ICON_OPEN_16);
    fileMenu.addSeparator();
    fileMenu.add(new ApplicationExitAction(this));
    return fileMenu;
  }

  /**
   * Montagem do menu de arquivos.
   * 
   * @return .
   */
  private JMenu createHelpMenu() {
    final JMenu helpMenu = new JMenu(getString("HTML_VIEWER_MENU_HELP"));
    helpMenu.add(new ApplicationAboutAction(this));
    return helpMenu;
  }

  /**
   * Cria a janela de exibio do HTML
   * 
   * @return referncia para a janela criada
   */
  private ApplicationFrame createMainFrame() {
    final ApplicationFrame mainFrame = getApplicationFrame();
    mainFrame.setJMenuBar(createMenuBar());
    final Container cp = mainFrame.getContentPane();
    cp.add(createNorthPanel(), BorderLayout.NORTH);
    cp.add(createViewer(), BorderLayout.CENTER);
    return mainFrame;
  }

  /**
   * Montagem do menu.
   * 
   * @return .
   */
  private JMenuBar createMenuBar() {
    final JMenuBar menuBar = new JMenuBar();
    menuBar.add(createFileMenu());
    menuBar.add(new JPanel());
    menuBar.add(createHelpMenu());
    return menuBar;
  }

  /**
   * Cria o painel que contem a toolBar e o URLPanel
   * 
   * @return o painel.
   */
  private JPanel createNorthPanel() {
    final JPanel np = new JPanel(new BorderLayout());
    np.add(createToolBar(), BorderLayout.NORTH);
    np.add(createURLPanel(), BorderLayout.CENTER);
    return np;
  }

  /**
   * Montagem da toolbar.
   * 
   * @return o painel.
   */
  private JToolBar createToolBar() {
    final JToolBar toolBar = new JToolBar(getName());
    openButton = new JButton(ApplicationImages.ICON_OPEN_16);
    openButton.setToolTipText(getString("HTML_VIEWER_FILE_OPEN"));
    openButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent e) {
        actionOpen();
      }
    });
    // Boto de back
    backButton = new JButton(ApplicationImages.ICON_BACK_16);
    backButton.setToolTipText(getString("HTML_VIEWER_BACK"));
    backButton.setEnabled(false);
    backButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent ev) {
        goBack();
      }
    });
    // Boto de forward
    forwardButton = new JButton(ApplicationImages.ICON_FORWARD_16);
    forwardButton.setToolTipText(getString("HTML_VIEWER_FORWARD"));
    forwardButton.setEnabled(false);
    forwardButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(final ActionEvent ev) {
        goForward();
      }
    });
    toolBar.add(openButton);
    // toolBar.add(closeButton);
    toolBar.add(new JToolBar.Separator());
    toolBar.add(backButton);
    toolBar.add(forwardButton);
    return toolBar;
  }

  /**
   * Cria o painel principal.
   * 
   * @return o painel
   */
  private JPanel createURLPanel() {
    // Boto de fechamento da janela
    urlText = new JTextField(120);
    urlText.setEditable(false);
    final JComponent[][] components =
      { { new JLabel(getString("HTML_VIEWER_URL")), urlText }, };
    final JPanel up = GUIUtils.createBasicGridPanel(components);
    return up;
  }

  /**
   * Cria o painel de botes.
   * 
   * @return painel de botes
   */
  private JComponent createViewer() {
    htmlPane = new JEditorPane();
    htmlPane.setEditable(false);
    htmlPane.setContentType("text/html");
    htmlPane.addHyperlinkListener(new HyperlinkListener() {
      @Override
      public void hyperlinkUpdate(final HyperlinkEvent e) {
        if (e.getEventType() != HyperlinkEvent.EventType.ACTIVATED) {
          return;
        }
        final JEditorPane pane = (JEditorPane) e.getSource();
        if (e instanceof HTMLFrameHyperlinkEvent) {
          final HTMLFrameHyperlinkEvent evt = (HTMLFrameHyperlinkEvent) e;
          final HTMLDocument doc = (HTMLDocument) pane.getDocument();
          doc.processHTMLFrameHyperlinkEvent(evt);
          setCurrentURL(e.getURL(), true);
        }
        else {
          try {
            pane.setPage(e.getURL());
            setCurrentURL(e.getURL(), true);
          }
          catch (final Throwable t) {
            StandardErrorDialogs
              .showErrorDialog(mainFrame, getString("HTML_VIEWER_TITLE"),
                getString("HTML_VIEWER_ERROR"), t);
          }
        }
      }
    });
    htmlScrolledPane = new JScrollPane(htmlPane);
    htmlScrolledPane
      .setVerticalScrollBarPolicy(ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
    return htmlScrolledPane;
  }

  /**
   * Constri os elementos da interface
   */
  private void createWidgets() {
    mainFrame = createMainFrame();
    mainFrame.setTitle(getString("HTML_VIEWER_TITLE"));
  }

  /**
   * Volta  pgina anterior
   */
  protected void goBack() {
    try {
      histPnt = histPnt - 1;
      if (histPnt >= 0 || histPnt < history.size()) {
        final URL url = history.get(histPnt);
        htmlPane.setPage(url);
        setCurrentURL(url, false);
      }
    }
    catch (final Exception e) {
      showErrorURL(null, e.getMessage());
    }
  }

  /**
   * Volta  pgina anterior
   */
  protected void goForward() {
    try {
      histPnt = histPnt + 1;
      if (histPnt >= 0 || histPnt < history.size()) {
        final URL url = history.get(histPnt);
        htmlPane.setPage(url);
        setCurrentURL(url, false);
      }
    }
    catch (final Exception e) {
      showErrorURL(null, e.getMessage());
    }
  }

  /**
   * Trmino ASAP da aplicao.
   */
  @Override
  public void killApplication() {
  }

  /**
   * Monta uma URL atravs de uma string.
   * 
   * @param addr o endereo que representa a URL.
   * 
   * @return .
   */
  private URL mountURL(final String addr) {
    URL url = null;
    try {
      url = new URL(addr);
    }
    catch (final Exception e) {
      url = null;
    }
    return url;
  }

  /**
   * Abertura de arquivo.
   * 
   * @param newFile novo arquivo.
   */
  private void openFile(final ClientProjectFile newFile) {
    if (newFile == null) {
      return;
    }
    final DesktopFrame desktop = DesktopFrame.getInstance();
    final CommonClientProject project = desktop.getProject();
    // Aqui, pega a URL atraves do HTTPService, passando:
    // path do arquivo, projeto e usuario
    HttpProxy.setTexts(getString("HTML_VIEWER_PROXY_TITLE"),
      getString("HTML_VIEWER_PROXY_FAILURE"),
      getString("HTML_VIEWER_PROXY_SUCCESS"));
    final String addr =
      HttpProxy.getDownloadURL(project.getId(), newFile.getPath());
    showURL(addr);
  }

  /**
   * Recebe o arquivo para ser aberto diretamente
   * 
   * @param name .
   * @param value .
   */
  @Override
  public void sendMessage(final String name, final Object value, String senderId) {
    if (value == null) {
      return;
    }
    if (name.equals(ApplicationProject.PROJECT_FILE_MESSAGE)) {
      final ClientProjectFile file = (ClientProjectFile) value;
      openFile(file);
    }
  }

  /**
   * Atualizao do label da URL corrente.
   * 
   * @param url a string a ser ajustada.
   * @param count .
   */
  private void setCurrentURL(final URL url, final boolean count) {
    if (url == null) {
      clearURL();
      return;
    }

    if (count) {
      history.add(url);
      histPnt++;
    }
    urlText.setText(url.toString());
    backButton.setEnabled(histPnt <= 0 ? false : true);
    forwardButton.setEnabled(histPnt < history.size() - 1 ? true : false);
  }

  /**
   * Exibe a janela de exibio do HTML
   */
  private void showDialog() {
    mainFrame.setSize(new Dimension(800, 600));
    mainFrame.center(DesktopFrame.getInstance().getDesktopFrame());
    mainFrame.setVisible(true);
  }

  /**
   * Mostra informaes de erro da URL
   * 
   * @param addr .
   * @param msg .
   */
  private void showErrorURL(final String addr, final String msg) {
    final String addStr = addr == null ? "???" : addr;
    final String errorMsg = msg == null ? "" : msg;
    htmlPane.setText("<html><table height=100% width=100%>"
      + "<tr valign=center height=100% width=100%><td>" + "<p>"
      + getString("HTML_VIEWER_CANNOT_FIND") + "</p>"
      + "<ul><i><font color=#ff0000>" + addStr + "</font></i></ul>" + "<p>"
      + getString("HTML_VIEWER_CONTACT") + "</p>" + "<br><hr><br> <pre>"
      + errorMsg + "</pre>" + "</td></tr>" + "</table></html>");
    urlText.setText(getString("HTML_VIEWER_ERROR"));
    backButton.setEnabled(false);
    forwardButton.setEnabled(false);
  }

  /**
   * Atualiza informaes da URL
   * 
   * @param addr .
   */
  private void showURL(final String addr) {
    final String address = addr;
    if (address == null) {
      showErrorURL(address, "No address specified!");
      return;
    }
    try {
      final URL url = mountURL(address);
      if (url == null) {
        showErrorURL(address, "No URL could be mounted!");
        return;
      }
      final LocalTask<Void> task = new LocalTask<Void>() {
        @Override
        protected void handleError(final Exception error) {
          showErrorURL(address, error.getMessage());
        }

        @Override
        public void performTask() throws Exception {
          htmlPane.setPage(url);
          setCurrentURL(url, true);
        }
      };
      task.execute(mainFrame, getString("HTML_VIEWER_WAIT_TITLE"),
        getString("HTML_VIEWER_WAIT_MESSAGE"));
    }
    catch (final Exception e) {
      showErrorURL(address, e.getMessage());
    }
  }

  /**
   * Mtodo que permite sempre ao usurio matar a aplicao.
   * 
   * @return o flag indicativo.
   */
  @Override
  public boolean userCanKillApplication() {
    return true;
  }

  /**
   * Constri e exibe o dilogo de execuo de algoritmos
   * 
   * @param id .
   */
  public HtmlViewer(final String id) {
    super(id);
    history = new Vector<URL>();
    histPnt = -1;
    createWidgets();
    showDialog();
  }
}
