/*
 * $Id$
 */
package csbase.client.applications;

import java.awt.Component;
import java.awt.Frame;
import java.awt.Toolkit;
import java.awt.Window;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.swing.ImageIcon;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.SwingConstants;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.core.lng.LNG.TranslationListener;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.StatusBar;
import csbase.client.applicationmanager.ApplicationException;
import csbase.client.applicationmanager.ApplicationManager;
import csbase.client.applicationmanager.ApplicationType;
import csbase.client.desktop.DesktopComponentFrame;
import csbase.client.desktop.DesktopFrame;
import csbase.client.preferences.PreferenceCategory;
import csbase.client.preferences.PreferenceManager;
import csbase.client.preferences.PreferenceValue;
import csbase.client.preferences.definition.PreferenceDefinition;
import csbase.client.preferences.util.PreferenceListener;
import csbase.client.preferences.util.PreferencesUtil;
import csbase.client.remote.srvproxies.EventLogProxy;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.CommonClientProject;
import csbase.logic.applicationservice.AppPropertyResourceBundle;
import csbase.logic.applicationservice.ApplicationRegistry;

/**
 * Classe bsica de aplicao com implementao para um suporte mnimo:
 * lanamento e encerramento de suas intncias, janela principal etc.
 *
 * @author Tecgraf
 */
public abstract class Application extends ApplicationType {

  /**
   * O identificar de mensagens enviadas no mtodo
   * {@link ApplicationType#sendMessage(String, Object, String)}.
   */
  @Deprecated
  public static final String PROJECT_FILE_MESSAGE = "PROJECT_FILE";

  /**
   * Usado para identificar objetos de preferncias de aplicao ao chamarmos o
   * mtodo {@link ApplicationType#sendMessage(String, Object, String)} .
   */
  public static final String PREFERENCE_MESSAGE = "PREFERENCE_MESSAGE";

  /** Janela principal */
  private final ApplicationFrame mainFrame;

  /** Barra de Status de uma aplicao */
  private JLabel statusBar;

  /** Vetor de janelas */
  private final ArrayList<Window> dependentWindows = new ArrayList<Window>();

  /** Estado inicial da janela da aplicao */
  private int initialFrameState;

  /** Mapa com os ouvintes de preferncias. */
  private Map<PreferenceDefinition, List<PreferenceListener<?>>> preferenceListeners;

  /**
   * Um mapa com os textos encontrados sem traduo em nenhum idioma carregado.
   */
  private Map<String, String> keyNotFoundInAnyIdiom;

  /**
   * Um mapa com os textos encontrados sem traduo para o idioma selecionado no
   * login do usurio.
   */
  private Map<String, String> keyNotFoundInSelectedIdiom;

  /**
   * Criacao da barra de status que pode ser adicionada em uma aplicao.
   */
  private void buildStatusBar() {
    this.statusBar = new JLabel();
    this.statusBar.setHorizontalAlignment(SwingConstants.LEFT);
  }

  /**
   * Obtm a barra de status para ser usada em uma aplicao.
   *
   * @return .
   * @deprecated Esse mtodo retorna apenas a instncia de um JLabel. Este no
   *             foi adicionado na janela da aplicao e  responsabilidade de
   *             quem utiliz-lo fazer isso. Ao invs deste utilize a
   *             {@link StatusBar barra de status} que j existe na
   *             {@link ApplicationFrame janela da aplicao}.
   * @see ApplicationFrame#getStatusBar()
   * @see ApplicationFrame#showStatusBar()
   * @see StatusBar
   */
  @Deprecated
  public JLabel getStatusBar() {
    return this.statusBar;
  }

  /**
   * Retorna o locale
   *
   * @return o locale
   */
  public Locale getLocale() {
    return ApplicationManager.getInstance().getLocale();
  }

  /**
   * Consulta ao projeto corrente.
   *
   * @return o projeto corrente (que necessariamente existe).
   *
   * @throws RuntimeException se no houver desktop e/ou projeto.
   */
  public final DesktopFrame getDesktopFrame() {
    final DesktopFrame dsk = DesktopFrame.getInstance();
    if (dsk == null) {
      final String err = "Internal Error: no desktop found for application!";
      throw new RuntimeException(err);
    }
    return dsk;
  }

  /**
   * Consulta ao projeto corrente.
   *
   * @return o projeto corrente (que necessariamente existe).
   *
   * @throws RuntimeException se no houver desktop e/ou projeto.
   */
  public final CommonClientProject getApplicationProject() {
    final DesktopFrame dsk = getDesktopFrame();
    final ApplicationRegistry reg = getApplicationRegistry();
    final CommonClientProject prj = dsk.getProject();
    if (prj != null) {
      return prj;
    }
    if (reg.requireProject()) {
      final String err = "No project found for application that requires one!";
      throw new IllegalStateException(err);
    }
    return null;
  }

  /**
   * Consulta as propriedades pr-configuradas no manager.
   *
   * @return as propriedades
   */
  public final String getApplicationCommand() {
    return this.getApplicationRegistry().getApplicationCommand();
  }

  /**
   * Consulta ao diretrio onde est o comando da aplicao.
   *
   * @return o diretrio onde est o comando da aplicao
   */
  public final String getApplicationCommandDir() {
    return this.getApplicationRegistry().getApplicationCommandDir();
  }

  /**
   * Indica se a aplicao est rodando em ambiente de execuo client-servidor;
   * ou seja, conectada a um servidor CSBASE.
   *
   * @return indicativo
   */
  final public boolean isConnectedToServer() {
    return true;
  }

  /**
   * Monta uma imagem do resource da aplicao.
   *
   * @param tag nome do arquivo de imagem.
   *
   * @return uma imagem (cone)
   */
  public final ImageIcon buildApplicationImage(final String tag) {
    final ImageIcon image = getImageIcon(tag + ".gif");
    if (image == null) {
      return ApplicationImages.ICON_NONE;
    }
    return image;
  }

  /**
   * Criao de um frame da aplicao.
   *
   * @return um frame da aplicao.
   */
  public final ApplicationComponentFrame buildApplicationFrame() {
    return new ApplicationComponentFrame(this);
  }

  /**
   * Retorna o dilogo principal da aplicao.
   *
   * @return a janela.
   */
  public final ApplicationFrame getApplicationFrame() {
    return this.mainFrame;
  }

  /**
   * Consulta ao bundle da aplicao.
   *
   * @return o bundle
   */
  public final AppPropertyResourceBundle getResourceBundle() {
    final ApplicationRegistry reg = this.getApplicationRegistry();
    return reg.getResourceBundle();
  }

  /**
   * Consulta a existncia de uma string de locale
   *
   * @param key a chave de consulta
   *
   * @return um indicativo
   */
  public final boolean hasString(final String key) {
    final ApplicationRegistry reg = this.getApplicationRegistry();
    return reg.hasString(key);
  }

  /**
   * Consulta a uma string de locale
   *
   * @param key a chave de consulta
   * @return a string baseada no locale.
   */
  public final String getString(final String key) {
    final ApplicationRegistry reg = this.getApplicationRegistry();
    return reg.getString(key);
  }

  /**
   * Obtm a verso localizada de uma mensagem que contenha parmetros.
   *
   * @param key chave para a mensagem que contm parmetros.
   * @param args argumentos para a mensagem.
   *
   * @return mensagem preenchida pelos argumentos e localizada.
   */
  public final String getString(final String key, final Object[] args) {
    final ApplicationRegistry reg = this.getApplicationRegistry();
    return reg.getString(key, args);
  }

  /**
   * Consulta a existncia de uma string de locale, com base em um objeto
   *
   * @param clazz classe de consulta.
   * @param key a chave de consulta
   *
   * @return um indicativo
   */
  public final boolean hasClassString(final Class<?> clazz, final String key) {
    final ApplicationRegistry reg = this.getApplicationRegistry();
    return reg.hasClassString(clazz, key);
  }

  /**
   * Consulta a uma string de locale, com base em um objeto
   *
   * @param clazz classe de consulta.
   * @param key a chave de consulta
   * @return a string baseada no locale.
   */
  public final String getClassString(final Class<?> clazz, final String key) {
    final ApplicationRegistry reg = this.getApplicationRegistry();
    return reg.getClassString(clazz, key);
  }

  /**
   * Obtm a verso localizada de uma mensagem que contenha parmetros, com base
   * em um objeto.
   *
   * @param clazz classe de consulta.
   * @param key chave para a mensagem que contm parmetros.
   * @param args argumentos para a mensagem.
   *
   * @return mensagem preenchida pelos argumentos e localizada.
   */
  public final String getClassString(final Class<?> clazz, final String key,
    final Object[] args) {
    final ApplicationRegistry reg = this.getApplicationRegistry();
    return reg.getClassString(clazz, key, args);
  }

  /**
   * Retorna o stream de entrada correspondente a um arquivo que se encontra no
   * repositrio da aplicao.
   *
   * NOTA:  necessrio fechar o stream de entrada retornado por este mtodo.
   *
   * @param resourcePath - path do arquivo.
   * @return stream de entrada correspondente ao arquivo.
   */
  public final InputStream getResource(String[] resourcePath) {
    ApplicationManager manager = ApplicationManager.getInstance();
    ApplicationRegistry registry = getApplicationRegistry();

    ApplicationFrame frame = getApplicationFrame();

    return manager.getApplicationResource(frame, registry, resourcePath);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void startApplication() throws ApplicationException {
    final DesktopFrame desktop = DesktopFrame.getInstance();
    final DesktopComponentFrame frame = desktop.getDesktopFrame();
    this.mainFrame.setTitle(getName());
    final ApplicationManager manager = ApplicationManager.getInstance();
    final ApplicationRegistry reg = manager.getApplicationRegistry(getId());
    final byte[] iconDefinition = reg.getIconDefinition();
    if (iconDefinition != null) {
      final ImageIcon imageIcon = new ImageIcon(iconDefinition);
      this.mainFrame.setIconImage(imageIcon.getImage());
    }

    if (this.initialFrameState != Frame.NORMAL) {
      this.mainFrame.pack();
      this.mainFrame.setExtendedState(this.initialFrameState);
      // usamos validate() para garantir que mudanas no tamanho (p.ex.
      // maximize) sejam consideradas antes do frame ser exibido
      this.mainFrame.validate();
    }
    // centralizamos a nova janela com relao ao seu pai ou ao fundo da tela
    if (frame == null) {
      final Toolkit defaultToolkit = Toolkit.getDefaultToolkit();
      this.mainFrame.center(defaultToolkit.getScreenSize(), 0, 0);
    }
    else {
      this.mainFrame.center(frame);
    }
    this.mainFrame.setVisible(reg.mainFrameIsVisible());
    this.mainFrame.toFront();

  }

  /**
   * {@inheritDoc}
   */
  @Override
  public final void finishApplication() {
    closeDependentWindows();
    this.mainFrame.setVisible(false);
    this.mainFrame.delApplicationManagerListener();
    final ApplicationManager manager = ApplicationManager.getInstance();
    manager.notifyDeath(this);
    this.mainFrame.close();

    final String[] aQueue = manager.getApplicationManagerEventQueue();

    // Loggando a inexistncia de chaves em qualquer idioma
    final String notAny = getTextNotFoundInAnyIdiom();
    if (notAny != null && !notAny.trim().isEmpty()) {
      String[] msg = { "Textos sem chave em qualquer idioma:\n", notAny };
      EventLogProxy.addClientInformation(aQueue, msg);
    }

    // Loggando a inexistncia de chaves no idioma selecionado.
    final String notIdiom = getTextNotFoundInSelectedIdiom();
    if (notIdiom != null && !notIdiom.trim().isEmpty()) {
      String[] msg =
        new String[] { "Textos sem chave no idioma " + LNG.getLocale() + ":\n",
        notIdiom };
      EventLogProxy.addClientInformation(aQueue, msg);
    }

    PreferenceManager pm = PreferenceManager.getInstance();
    if (pm.hasAppPreferences(getApplicationRegistry())) {
      removeAllPreferenceListeners();
    }
  }

  /**
   * Obtm as chaves de idioma no encontradas em todos os bundles carregados.
   *
   * @return um texto com as chaves no encontradas
   */
  private String getTextNotFoundInAnyIdiom() {
    StringBuffer textNotFound = new StringBuffer("");
    for (String key : keyNotFoundInAnyIdiom.keySet()) {
      textNotFound.append(key);
      textNotFound.append(": ");
      textNotFound.append(keyNotFoundInSelectedIdiom.get(key));
      textNotFound.append("\n");
    }
    return textNotFound.toString();
  }

  /**
   * Obtm as chaves de idioma no encontradas no idioma selecionado pelo
   * usurio. As chaves vem acompanhadas com o texto usado no idioma nativo.
   *
   * @return um texto com as chaves no traduzidas
   */
  private String getTextNotFoundInSelectedIdiom() {
    StringBuffer textWithoutTranslation = new StringBuffer("");
    for (String key : keyNotFoundInSelectedIdiom.keySet()) {
      textWithoutTranslation.append(key);
      textWithoutTranslation.append(": ");
      textWithoutTranslation.append(keyNotFoundInSelectedIdiom.get(key));
      textWithoutTranslation.append("\n");
    }
    return textWithoutTranslation.toString();
  }

  /**
   * Mtodo para log de informaes especficas do cliente (applicao).
   *
   * @param queue fila
   * @param info mensagem a ser exibida.
   */
  public final void logDetailedApplicationEvent(final String[] queue,
    final String[] info) {
    final ApplicationManager mgr = ApplicationManager.getInstance();
    final String[] aQueue = mgr.getApplicationEventQueue(this, queue);
    EventLogProxy.addClientInformation(aQueue, info);
  }

  /**
   * Mtodo para exibio de uma mensagem informativa.
   *
   * @param msg mensagem a ser exibida.
   */
  public final void showMessage(final Object msg) {
    final String title = getApplicationFrame().getTitle();
    showMessage(this.mainFrame, title, msg, JOptionPane.INFORMATION_MESSAGE);
  }

  /**
   * Mtodo para exibio de uma mensagem do tipo especificado.
   *
   * @param parent elemento (janela) pai.
   * @param title ttulo.
   * @param msg mensagem a ser exibida.
   * @param type tipo da mensagem.
   */
  final public void showMessage(final Component parent, final String title,
    final Object msg, final int type) {
    final Component father = (parent == null ? this.mainFrame : parent);
    JOptionPane.showMessageDialog(father, msg, title, type);
  }

  /**
   * Consulta a opes do programa.
   *
   * @param parent componente-pai do dilogo a ser exibido.
   * @param msg mensagem a ser exibida.
   * @param options array de opes.
   * @return a opo selecionada.
   */
  final public int showOptionDialog(final Component parent, final String msg,
    final String[] options) {
    final String title = getName();
    return StandardDialogs.showOptionDialog(parent, title, msg, options);
  }

  /**
   * Consulta a opes do programa.
   *
   * @param msg mensagem a ser exibida.
   * @param options array de opes.
   * @return a opo selecionada.
   */
  final public int showOptionDialog(final String msg, final String[] options) {
    return showOptionDialog(this.mainFrame, msg, options);
  }

  /**
   * Mtodo para exibio de uma pilha de erro (exceo).
   *
   * @param t a exceo.
   */
  final public void showExceptionStack(final Throwable t) {
    StandardErrorDialogs.showExceptionDialog(this.mainFrame, getName(), t);
  }

  /**
   * Mtodo para exibio de uma mensagem de erro, juntamente com a exceo
   * associada.
   *
   * @param msg uma mensagem.
   * @param t a exceo.
   */
  final public void showException(final String msg, final Throwable t) {
    StandardErrorDialogs.showErrorDialog(this.mainFrame, getName(), msg, t);
  }

  /**
   * Mtodo para exibio de uma mensagem de erro.
   *
   * @param msg o texto mensagem.
   */
  final public void showError(final Object msg) {
    final ApplicationFrame frame = getApplicationFrame();
    showError(frame, msg);
  }

  /**
   * Mtodo para exibio de uma mensagem de erro.
   *
   * @param parent elemento (janela) pai.
   * @param msg o texto da mensagem.
   */
  final public void showError(final Component parent, final Object msg) {
    showMessage(parent, getName(), msg, JOptionPane.ERROR_MESSAGE);
  }

  /**
   * Mtodo para exibio de uma mensagem informativa.
   *
   * @param parent elemento (janela) pai.
   * @param msg o texto da mensagem.
   */
  final public void showInformation(final Component parent, final Object msg) {
    showMessage(parent, getName(), msg, JOptionPane.INFORMATION_MESSAGE);
  }

  /**
   * Mtodo para exibio de uma mensagem informativa.
   *
   * @param msg o texto da mensagem.
   */
  final public void showInformation(final Object msg) {
    showInformation(null, msg);
  }

  /**
   * Mtodo para exibio de uma mensagem de aviso.
   *
   * @param parent elemento (janela) pai.
   * @param msg o texto da mensagem.
   */
  final public void showWarning(final Component parent, final Object msg) {
    showMessage(parent, getName(), msg, JOptionPane.WARNING_MESSAGE);
  }

  /**
   * Mtodo para exibio de uma mensagem de aviso.
   *
   * @param msg o texto da mensagem.
   */
  final public void showWarning(final Object msg) {
    showWarning(null, msg);
  }

  /**
   * Registra uma janela cliente na janela principal de modo que, ao fechar a
   * janela pricipal, todas as janelas registradas tambm so fechadas.
   *
   * @param win uma nova janela registrada
   */
  final public void addWindow(final Window win) {
    this.dependentWindows.add(win);
  }

  /**
   * Descadastra uma janela da cliente janela principal.
   *
   * @param win a janela que deve ser removida da janela principal.
   */
  final public void removeWindow(final Window win) {
    this.dependentWindows.remove(win);
  }

  /**
   * Fecha todas as janelas dependentes da janela principal da aplicao.
   */
  final public void closeDependentWindows() {
    final List<Window> aux = new ArrayList<Window>();
    aux.addAll(dependentWindows);
    for (Window window : aux) {
      window.setVisible(false);
      window.dispose();
    }
    dependentWindows.clear();
  }

  /**
   * Indica se a aplicao est visvel
   *
   * @return indicativo
   */
  final public boolean isVisible() {
    return mainFrame.isVisible();
  }

  /**
   * Torna status da aplicao visvel/invisvel
   *
   * @param flag indicativo
   */
  final public void setVisible(final boolean flag) {
    mainFrame.setVisible(flag);
  }

  /**
   * Retorna todas as preferncias da aplicao.
   *
   * @return preferncias da aplicao.
   */
  final public PreferenceCategory getPreferences() {
    ApplicationFrame frame = getApplicationFrame();
    ApplicationRegistry registry = getApplicationRegistry();

    PreferenceManager pm = PreferenceManager.getInstance();
    return pm.loadAppPreferences(registry, frame);
  }

  /** Salva as preferncias da aplicao. */
  final public void savePreferences() {
    PreferenceManager.getInstance().savePreferences(getApplicationFrame());
  }

  /**
   * Constri uma aplicao desktop
   *
   * @param id identificador da aplicao.
   */
  protected Application(final String id) {
    super(id);
    setInitialFrameState(Frame.NORMAL);
    this.keyNotFoundInAnyIdiom = new HashMap<String, String>();
    this.keyNotFoundInSelectedIdiom = new HashMap<String, String>();
    final ApplicationRegistry reg =
      ApplicationManager.getInstance().getApplicationRegistry(id);
    reg.setTranslationListener(new TranslationListener() {
      @Override
      public String keyNotFound(String key, String text) {
        keyNotFoundInAnyIdiom.put(key, text);
        return text;
      }

      @Override
      public String keyNotFoundInDefaultLanguage(String key, String text) {
        keyNotFoundInSelectedIdiom.put(key, text);
        return text;
      }
    });
    this.mainFrame = new ApplicationFrame(this, id);
    this.mainFrame.addApplicationManagerListener();
    this.mainFrame.addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(final WindowEvent e) {
        closeApplication();
      }
    });

    this.preferenceListeners =
      new HashMap<PreferenceDefinition, List<PreferenceListener<?>>>();

    // Criando a barra de status
    buildStatusBar();
  }

  /**
   * Adiciona ouvinte a uma preferncia do usurio. Assim que a aplicao for
   * fechada, todos os ouvintes sero removidos automaticamente.
   *
   * @param <T> - tipo do valor da preferncia.
   *
   * @param name - nome da preferncia.
   * @param listener - ouvinte da preferncia.
   */
  @SuppressWarnings("unchecked")
  protected <T> void addPreferenceListener(PreferenceDefinition name,
    PreferenceListener<T> listener) {

    PreferenceCategory pc = PreferencesUtil.deepSearch(name, getPreferences());

    PreferenceValue<T> preference = (PreferenceValue<T>) pc.getPreference(name);
    preference.addPreferenceListener(listener);

    if (!preferenceListeners.containsKey(name)) {
      preferenceListeners.put(name, new LinkedList<PreferenceListener<?>>());
    }
    preferenceListeners.get(name).add(listener);
  }

  /**
   * Remove todas os ouvintes de preferncias da aplicao.
   *
   * @param <T> - tipo da preferncia a ser removida.
   */
  @SuppressWarnings("unchecked")
  protected <T> void removeAllPreferenceListeners() {
    for (PreferenceDefinition name : preferenceListeners.keySet()) {
      PreferenceCategory cat =
        PreferencesUtil.deepSearch(name, getPreferences());

      PreferenceValue<T> preference =
        (PreferenceValue<T>) cat.getPreference(name);
      for (PreferenceListener<?> l : preferenceListeners.get(name)) {
        PreferenceListener<T> listener = (PreferenceListener<T>) l;
        preference.removePreferenceListener(listener);
      }
    }
  }

  /**
   * Mtodo padro de tratamento de excees da aplicao.
   *
   * @param ex a exceo.
   * @return <code>false</code> por default.
   */
  protected boolean handleError(final Exception ex) {
    return false;
  }

  /**
   * Indica que a janela da aplicao deve ser iniciada no estado indicado, que
   * pode ser {@link Frame#NORMAL}, {@link Frame#MAXIMIZED_BOTH}, etc..
   *
   * @param initState - o estado inicial.
   */
  protected final void setInitialFrameState(final int initState) {
    this.initialFrameState = initState;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public abstract void killApplication() throws ApplicationException;
}
