/*
 * $Id$
 */
package csbase.client.desktop;

import java.awt.Container;
import java.awt.FlowLayout;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.GBC;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.remote.srvproxies.MailProxy;
import csbase.client.remote.srvproxies.messageservice.MessageProxy;
import csbase.client.util.ClientUtilities;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.Notification;
import csbase.logic.User;
import csbase.logic.UserNotification;
import csbase.util.messages.Message;

/**
 * Classe bsica para telas de envio de mensagens de notificao.
 * 
 * @author Tecgraf/PUC-Rio
 */
abstract class AbstractNotificationFrame extends DesktopComponentDialog {

  /**
   * rea de edio
   */
  final protected JTextArea editionArea = new JTextArea();

  /**
   * Boto de envio da mensagem
   */
  final protected JButton sendButton = new JButton(
    LNG.get("notification.send.button"));

  /**
   * Opo de envio via email.
   */
  final protected JCheckBox mailToggle = new JCheckBox();

  /**
   * Opo de envio via popup.
   */
  final protected JCheckBox popUpToggle = new JCheckBox();

  /**
   * Boto para fechar a tela.
   */
  final protected JButton closeButton = new JButton(
    LNG.get("notification.cancel.button"));

  /**
   * Constri a interface. Este mtodo  tipicamente executado no construtor.
   */
  protected void buildUI() {
    addWindowListener(new WindowAdapter() {
      @Override
      public void windowClosing(WindowEvent we) {
        close();
      }
    });
    setTitle(getWindowTitle());
    Container contentPane = getContentPane();
    contentPane.setLayout(new GridBagLayout());
    addSubPanels(contentPane);
    pack();
  }

  /**
   * Inicializaes pr-exibio.
   * 
   * @see #start()
   */
  protected abstract void preShowInit();

  /**
   * Obtm o ttulo da janela.
   * 
   * @return ttulo da janela
   */
  protected abstract String getWindowTitle();

  /**
   * Exibe a janela.
   * 
   * @see #preShowInit()
   */
  public void start() {
    this.mailToggle.setSelected(false);
    this.popUpToggle.setSelected(false);
    this.editionArea.setText("");
    preShowInit();
    final DesktopFrame desktop = DesktopFrame.getInstance();
    final DesktopComponentFrame desktopFrame = desktop.getDesktopFrame();
    center(desktopFrame);
    setVisible(true);
  }

  /**
   * Adiciona os subpainis ao painel principal da janela. O <i>layout
   * manager</i> foi definido como sendo {@link GridBagLayout}.
   * 
   * @param mainPane painel principal
   */
  protected abstract void addSubPanels(Container mainPane);

  /**
   * Cria o painel de edio da mensagem.
   * 
   * @param title ttulo do painel
   * 
   * @return painel de edio
   */
  protected JPanel createEditionPanel(String title) {
    editionArea.setColumns(40);
    editionArea.setRows(10);
    editionArea.getDocument().addDocumentListener(new DocumentListener() {

      @Override
      public void changedUpdate(DocumentEvent e) {
        // vazio
      }

      @Override
      public void insertUpdate(DocumentEvent e) {
        messageTextChanged();
      }

      @Override
      public void removeUpdate(DocumentEvent e) {
        messageTextChanged();
      }
    });

    // Montagem de painel geral de edio.
    JPanel editionPanel = new JPanel(new GridBagLayout());
    editionPanel.setBorder(BorderFactory.createTitledBorder(title));

    // Montagem do scrollpane de edio.
    JScrollPane scrollEdition = new JScrollPane(editionArea);

    // Toggle de envio de email e volatilidade.
    mailToggle.setText(LNG.get("notification.mail.option"));
    mailToggle.setSelected(false);
    popUpToggle.setText(LNG.get("notification.pop_up.option"));
    popUpToggle.setSelected(false);

    Insets border = new Insets(0, 5, 0, 5);
    editionPanel.add(mailToggle, new GBC(0, 0).northwest().insets(border));
    editionPanel.add(popUpToggle, new GBC(0, 1).northwest().insets(border));
    editionPanel
      .add(scrollEdition, new GBC(0, 2).northwest().both().insets(10));

    return editionPanel;
  }

  /**
   * Mtodo de acesso aos proxies de notificao e mail para consulta aos
   * widgets da interface e envio das mensagens.
   * 
   * @param isVolatile indicativo de volatilidade
   */
  protected void sendMessage(boolean isVolatile) {
    DesktopFrame imf = DesktopFrame.getInstance();
    if (imf == null) {
      return;
    }
    User loggedUser = User.getLoggedUser();
    if (loggedUser == null) {
      return;
    }
    String content = editionArea.getText();
    if (content.trim().isEmpty()) {
      StandardDialogs.showErrorDialog(AbstractNotificationFrame.this, null,
        LNG.get("notification.no.message.error"));
      editionArea.requestFocus();
      return;
    }

    List<Object> users = getRecipientsIDs();
    final String[] usersLogin = users.toArray(new String[0]);
    Notification notification =
      new UserNotification(loggedUser.getLogin(), content,
        this.popUpToggle.isSelected(), isVolatile);
    final Message message = new Message(notification);
    DesktopTask task = new DesktopTask() {
      @Override
      public void performTask() throws Exception {
        MessageProxy.send(message, usersLogin);
      }
    };
    task.start(DesktopFrame.getInstance().getDesktopFrame(),
      LNG.get("notification.send.title"), LNG.get("notification.send.message"));

    // Tentamos mandar a mensagem com o primeiro email cadastrado.
    // Caso no haja nenhum, o login ser usado.
    String snd = null;
    final String[] emails = loggedUser.getEmails();
    if (emails == null || emails.length == 0) {
      snd = loggedUser.getLogin();
    }
    else {
      snd = emails[0];
    }
    boolean mailedAllUsers = true;
    for (Object userID : users) {
      if (mailToggle.isSelected()) {
        if (!MailProxy.mailTo(snd, userID, content, false)) {
          mailedAllUsers = false;
        }
      }
    }
    if (!mailedAllUsers) {
      JFrame jFrame = imf.getDesktopFrame();
      StandardErrorDialogs.showErrorDialog(jFrame,
        LNG.get("notification.failure.title"),
        LNG.get("notification.failure.mail.msg"));
    }
    close();
  }

  /**
   * Obtm os IDs dos destinatrios da mensagem.
   * 
   * @return IDs dos destinatrios da mensagem
   */
  protected abstract List<Object> getRecipientsIDs();

  /**
   * Consulta da existncia de uma mensagem digitada.
   * 
   * @return um flag indicativo.
   */
  public boolean hasMessage() {
    String content = editionArea.getText();
    return (!content.trim().isEmpty());
  }

  /**
   * Verifica se existe algum usurio selecionado.
   * 
   * @return <code>true</code> se existe algum usurio selecionado
   */
  protected abstract boolean hasRecipients();

  /**
   * Callback executada quando o contedo da mensagem  alterado.
   */
  public void messageTextChanged() {
    // vazio
  }

  /**
   * Obtm o {@link ActionListener} para o boto de envio da mensagem.
   * 
   * @return listener para o boto de envio da mensagem
   */
  protected abstract ActionListener getSendButtonActionListener();

  /**
   * Obtm o {@link ActionListener} para o boto de fechamento da tela.
   * 
   * @return listener para o boto de fechamento da tela
   */
  protected ActionListener getCloseButtonActionListener() {
    return new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent ev) {
        if (!editionArea.getText().trim().isEmpty()) {
          int option =
            StandardDialogs.showYesNoDialog(AbstractNotificationFrame.this,
              null, LNG.get("notification.question.discardContent"));
          if (option == StandardDialogs.NO_OPTION) {
            return;
          }
        }
        close();
      }
    };
  }

  /**
   * 
   * Cria o painel inferior com os botes.
   * 
   * @param addSendButton indica se o boto de "send" deve ser mostrado
   * @return o painel
   */
  protected JPanel createButtonsPane(boolean addSendButton) {
    closeButton.addActionListener(getCloseButtonActionListener());
    sendButton.addActionListener(getSendButtonActionListener());
    ClientUtilities.adjustEqualSizes(closeButton, sendButton);

    final JPanel panel = new JPanel(new FlowLayout());
    if (addSendButton) {
      panel.add(sendButton);
    }
    panel.add(closeButton);
    return panel;
  }

  /**
   * Cria o painel inferior com os botes (send e close)
   * 
   * @return painel inferior com os botes
   */
  protected JPanel createButtonsPane() {
    return createButtonsPane(true);
  }

  /**
   * Construtor.
   */
  public AbstractNotificationFrame() {
    super(null);
    setModalityType(ModalityType.MODELESS);
  }
}
