/*
 * PermissionManagerPanel.java
 * 
 * $Id$
 */
package csbase.client.ias;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.MessageFormat;
import java.util.Collections;
import java.util.Comparator;
import java.util.Vector;

import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.table.TableModel;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import tecgraf.javautils.gui.table.DefaultObjectTableProvider;
import tecgraf.javautils.gui.table.ObjectTableModel;
import tecgraf.javautils.gui.table.ObjectTableProvider;
import csbase.client.remote.srvproxies.PermissionProxy;
import csbase.client.util.ClientUtilities;
import csbase.client.util.StandardErrorDialogs;
import csbase.client.util.SwingObserverWrapper;
import csbase.logic.Permission;

/**
 * A classe <code>PermissionManagerPanel</code> representa o painel de
 * administrao das permisses de usurios. Mostra uma tabela com a
 * identificao e a descrio de cada permisso disponvel. O administrador
 * pode incluir, alterar e remover permisses. As permisses se dividem em 3
 * categorias bsicas:
 * <ul>
 * <li>simples</li>
 * <li>autenticadas (exigem usurio e senha)</li>
 * <li>com atributos (pode-se especificar um ou mais atributos de permisso)</li>
 * </ul>
 * 
 * @author Joaquim Cleto de Sousa
 * @version $Revision$
 */
public class PermissionManagerPanel extends ManagerPanel {
  /** A janela "pai" do painel */
  private JFrame parent;

  /** Boto Usurios */
  private JButton associateButton;

  // Constante utilizada para testes de condies de erro.
  // 0: funcionamento normal
  // 1: exibe a mensagem de comunicao com o servidor e dispara o tratamento 
  //    para permisso no encontrada
  private static final int debug = 0;

  /**
   * Cria um painel de gerncia de Permisses cadastradas no sistema
   * 
   * @param parent A janela "pai" do painel.
   */
  public PermissionManagerPanel(JFrame parent) {
    this.parent = parent;
    Vector<Permission> permissions =
      PermissionProxy.getAllPermissions(parent, parent.getTitle(),
        LNG.get("IAS_WAITING_ADMIN_DATA"));
    if (permissions == null) {
      permissions = new Vector<Permission>();
    }
    Collections.sort(permissions);
    ObjectTableProvider provider = new DefaultObjectTableProvider() {
      /**
       * {@inheritDoc}
       */
      @Override
      public Object[] getCellValues(Object item) {
        Permission permission = (Permission) item;
        if (permission == null) {
          return null;
        }
        return new Object[] { permission.getId(), permission.getName(),
            permission.getDescription() };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public String[] getColumnNames() {
        return new String[] { null, LNG.get("IAS_PERMISSION"),
            LNG.get("IAS_DESCRIPTION") };
      }

      /**
       * {@inheritDoc}
       */
      @Override
      public Class<?>[] getColumnClasses() {
        return new Class[] { Object.class, String.class, String.class };
      }
    };
    TableModel model = new ObjectTableModel<Permission>(permissions, provider);
    Comparator<?>[] comparators =
      { null, ClientUtilities.getStringComparatorIgnoreCase(),
          ClientUtilities.getStringComparatorIgnoreCase() };
    make(model, comparators, false);
    Permission.addObserver(new SwingObserverWrapper(this));
  }

  /**
   * Remove o observador de Permisses.
   */
  @Override
  public void beforeClose() {
    Permission.deleteObserver(new SwingObserverWrapper(this));
  }

  /**
   * Exibe a janela de incluso de Permisso.
   */
  @Override
  public void add() {
    new PermissionInfoDialog(parent);
  }

  /**
   * Exibe a janela de modificao de Permisso.
   * 
   * @param id identificador da Permisso a ser modificada.
   */
  @Override
  public void modify(Object id) {
    Permission permission = getPermission(id);
    if (permission == null) {
      return;
    }
    new PermissionInfoDialog(parent, permission);
  }

  /**
   * Exibe a janela de remoo de Permisso.
   * 
   * @param id identificador da Permisso a ser removida.
   */
  @Override
  public void delete(Object id) {
    Permission permission = getPermission(id);
    if (permission == null) {
      return;
    }
    String permissionName = permission.getName();
    int res =
      StandardDialogs.showYesNoDialog(parent, parent.getTitle(), MessageFormat
        .format(LNG.get("IAS_PERMISSION_MGR_PERMISSION_REMOVAL_CONFIRMATION"),
          new Object[] { permissionName }));
    if (res == JOptionPane.YES_OPTION) {
      PermissionProxy.deletePermission(parent, parent.getTitle(), MessageFormat
        .format(LNG.get("IAS_PERMISSION_MGR_WAITING_REMOVE_PERMISSION"),
          new Object[] { permissionName }), id);
    }
  }

  /**
   * Mostra uma janela com uma mensagem.
   * 
   * @param title ttulo da janela de mensagem
   * @param message mensagem a ser exibida
   */
  protected void displayMessage(String title, String message) {
    StandardDialogs.showInfoDialog(parent, title, message);
  }

  /**
   * Mostra uma janela com uma mensagem de erro.
   * 
   * @param message mensagem de erro a ser exibida
   */
  protected void displayError(String message) {
    StandardErrorDialogs.showErrorDialog(parent, parent.getTitle(), message);
  }

  /**
   * Redefine makeButtons para adicionar o boto Usuarios
   * 
   * @return .
   */
  @Override
  protected JPanel makeButtons() {
    JPanel panel = super.makeButtons();
    panel.add(new JPanel());
    this.createAssociateButton(panel, false);
    return panel;
  }

  /**
   * Cria o boto Usuarios
   * 
   * @param panel .
   * @param enabled .
   */
  private void createAssociateButton(JPanel panel, boolean enabled) {
    this.associateButton = new JButton(LNG.get("IAS_PERMISSION_USERS_BUTTON"));
    this.associateButton.addActionListener(new ActionListener() {
      @Override
      public void actionPerformed(ActionEvent e) {
        int row = getTable().getSelectedRow();
        Object id = getRowId(row);
        associate(id);
      }
    });
    this.associateButton.setEnabled(enabled);
    panel.add(this.associateButton);
    this.addButtonToSelectionListener(this.associateButton);
  }

  /**
   * Chama o dilogo de associao de usurios  permisso.
   * 
   * @param id .
   */
  protected void associate(Object id) {
    Permission permission = getPermission(id);
    if (permission == null) {
      return;
    }
    new PermissionUserDefinitionDialog(parent).showDialog(permission);
  }

  /**
   * Obtm a permisso identificada pelo parmetro especificado. Caso a
   * permisso no seja encontrada, exibe mensagem de erro adequada.
   * 
   * @param id identificador da permisso.
   * 
   * @return permisso ou <code>null</code> caso esta no seja encontrada.
   */
  private Permission getPermission(Object id) {
    Permission permission =
      PermissionProxy.getPermission(parent, parent.getTitle(),
        LNG.get("IAS_WAITING_PERMISSION"), id);
    if (debug == 1) { //@debug
      displayMessage(parent.getTitle(), LNG.get("IAS_WAITING_PERMISSION"));
      permission = null; // @debug
    } //@debug
    if (permission == null) {
      displayError(LNG.get("IAS_PERMISSION_MGR_PERMISSION_NOT_FOUND"));
    }
    return permission;
  }
}
