/*
 * SGAProxy.java $Author$ $Revision$ - $Date: 2008-06-06
 * 16:24:00 -0300 (Fri, 06 Jun 2008) $
 */
package csbase.client.remote.srvproxies;

import java.util.Collection;
import java.util.Hashtable;
import java.util.Vector;

import javax.swing.JOptionPane;

import tecgraf.javautils.core.lng.LNG;
import tecgraf.javautils.gui.StandardDialogs;
import csbase.client.desktop.DesktopTask;
import csbase.client.util.StandardErrorDialogs;
import csbase.logic.CommandInfo;
import csbase.logic.SGAInfo;
import csbase.logic.SGASet;
import csbase.logic.ServerGroupInfo;
import csbase.logic.User;
import csbase.remote.ClientRemoteLocator;
import csbase.remote.SGAServiceInterface;

/**
 * Classe que modela um proxy Cliente/SGAService.
 * 
 * @author Andr Luiz Clinio
 */
public class SGAProxy {
  /**
   * Mtodo de aviso de erro
   * 
   * @param msg mensagem a ser exibida.
   * @param e exceo causadora do erro (ou null)
   */
  private static void showError(String msg, Exception e) {
    String title = LNG.get("SGA_ACCESSING_TITLE");
    if (e != null) {
      StandardErrorDialogs.showErrorDialog(null, title, msg, e);
    }
    else {
      StandardErrorDialogs.showErrorDialog(null, title, msg);
    }
  }

  /**
   * Mtodo para busca dos SGAs registrados
   * 
   * @return uma hashtable com os dados dos SGAs
   */
  public static Hashtable<String, SGASet> getRegisteredServers() {
    final SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
    if (sgaService == null) {
      showError(LNG.get("SGA_NO_SERVER_ERROR"), null);
      return null;
    }
    DesktopTask task = new DesktopTask() {
      @Override
      public void performTask() throws Exception {
        Hashtable<String, SGASet> result = new Hashtable<String, SGASet>();
        Vector<String> sgaNames = sgaService.getAllSGANames();
        if ((sgaNames != null) && (sgaNames.size() > 0)) {
          for (int i = 0; i < sgaNames.size(); i++) {
            String sgaName = sgaNames.get(i);
            result.put(sgaName, sgaService.getSGASet(sgaName));
          }
        }
        this.setResult(result);
      }
    };
    task.start(null, LNG.get("SGA_ACCESSING_TITLE"),
      LNG.get("SGA_ACCESSING_MESSAGE"));

    /* Informa se ocorreu uma exceo */
    Exception exception = task.getException();
    if (exception != null) {
      showError(LNG.get("SGA_FAILURE_MESSAGE"), exception);
      return null;
    }
    return (Hashtable<String, SGASet>) task.getResult();
  }

  /**
   * Mtodo para busca dos nomes dos SGAs registrados
   * 
   * @return vetor de strings.
   */
  public static Vector<String> getAllSGANames() {
    final SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
    if (sgaService == null) {
      showError(LNG.get("SGA_NO_SERVER_ERROR"), null);
      return null;
    }
    DesktopTask task = new DesktopTask() {
      @Override
      public void performTask() throws Exception {
        setResult(sgaService.getAllSGANames());
      }
    };
    task.start(null, LNG.get("SGA_ACCESSING_TITLE"),
      LNG.get("SGA_ACCESSING_MESSAGE"));

    /* Informa se ocorreu uma exceo */
    Exception exception = task.getException();
    if (exception != null) {
      showError(LNG.get("SGA_FAILURE_MESSAGE"), exception);
      return null;
    }
    return (Vector<String>) task.getResult();
  }

  /**
   * Mtodo para obter as informaes de todos os ns de um SGA.
   * 
   * @param sgaName nome (identificao) do SGA
   * 
   * @return informaes do SGA ou null em caso de erro
   */
  public static SGAInfo[] getAllInfo(final String sgaName) {
    final SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
    if (sgaService == null) {
      showError(LNG.get("SGA_NO_SERVER_ERROR"), null);
      return null;
    }
    DesktopTask task = new DesktopTask() {
      @Override
      public void performTask() throws Exception {
        setResult(sgaService.getAllInfo(sgaName));
      }
    };
    task.start(null, LNG.get("SGA_ACCESSING_TITLE"),
      LNG.get("SGA_ACCESSING_MESSAGE"));

    /* Informa se ocorreu uma exceo */
    Exception exception = task.getException();
    if (exception != null) {
      showError(LNG.get("SGA_FAILURE_MESSAGE"), exception);
      return null;
    }
    return (SGAInfo[]) task.getResult();
  }

  /**
   * Mtodo para obter as informaes de um SGA.
   * 
   * @param sgaName nome (identificao) do SGA
   * 
   * @return informaes do SGA ou null em caso de erro
   */
  public static SGASet getSGASet(final String sgaName) {
    final SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
    if (sgaService == null) {
      showError(LNG.get("SGA_NO_SERVER_ERROR"), null);
      return null;
    }
    DesktopTask task = new DesktopTask() {
      @Override
      public void performTask() throws Exception {
        setResult(sgaService.getSGASet(sgaName));
      }
    };
    task.start(null, LNG.get("SGA_ACCESSING_TITLE"),
      LNG.get("SGA_ACCESSING_MESSAGE"));

    /* Informa se ocorreu uma exceo */
    Exception exception = task.getException();
    if (exception != null) {
      showError(LNG.get("SGA_FAILURE_MESSAGE"), exception);
      return null;
    }
    return (SGASet) task.getResult();
  }

  /**
   * Mtodo para obter o intervalo de atualizao das informaes dos comandos.
   * 
   * @return o intervalo de atualizao ou -1 em caso de erro
   */
  public static int getCommandsUpdateInterval() {
    final SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
    if (sgaService == null) {
      showError(LNG.get("SGA_NO_SERVER_ERROR"), null);
      return -1;
    }
    DesktopTask task = new DesktopTask() {
      @Override
      public void performTask() throws Exception {
        setResult(new Integer(sgaService.getCommandsUpdateInterval()));
      }
    };
    task.start(null, LNG.get("SGA_ACCESSING_TITLE"),
      LNG.get("SGA_ACCESSING_MESSAGE"));

    /* Informa se ocorreu uma exceo */
    Exception exception = task.getException();
    if (exception != null) {
      showError(LNG.get("SGA_FAILURE_MESSAGE"), exception);
      return -1;
    }
    return ((Integer) task.getResult()).intValue();
  }

  /**
   * Mtodo para obter o intervalo de atualizao das informaes
   * 
   * @return o intervalo de atualizao ou -1 em caso de erro
   */
  public static int getUpdateInterval() {
    final SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
    if (sgaService == null) {
      showError(LNG.get("SGA_NO_SERVER_ERROR"), null);
      return -1;
    }
    DesktopTask task = new DesktopTask() {
      @Override
      public void performTask() throws Exception {
        setResult(new Integer(sgaService.getUpdateInterval()));
      }
    };
    task.start(null, LNG.get("SGA_ACCESSING_TITLE"),
      LNG.get("SGA_ACCESSING_MESSAGE"));

    /* Informa se ocorreu uma exceo */
    Exception exception = task.getException();
    if (exception != null) {
      showError(LNG.get("SGA_FAILURE_MESSAGE"), exception);
      return -1;
    }
    return ((Integer) task.getResult()).intValue();
  }

  /**
   * Obtm as informaes de um comando em execuo.
   * 
   * @param sgaName nome do sga que executa o comando.
   * @param cmdId identificador do comando.
   * 
   * @return as informaes de um comando em execuo
   */
  public static CommandInfo getSGACommand(final String sgaName,
    final String cmdId) {
    final SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
    if (sgaService == null) {
      showError(LNG.get("SGA_NO_SERVER_ERROR"), null);
      return null;
    }
    DesktopTask task = new DesktopTask() {
      @Override
      public void performTask() throws Exception {
        setResult(sgaService.getSGACommand(sgaName, cmdId));
      }
    };
    task.start(null, LNG.get("SGA_ACCESSING_TITLE"),
      LNG.get("SGA_ACCESSING_MESSAGE"));

    /* Informa se ocorreu uma exceo */
    Exception exception = task.getException();
    if (exception != null) {
      showError(LNG.get("SGA_FAILURE_MESSAGE"), exception);
      return null;
    }
    return (CommandInfo) task.getResult();
  }

  /**
   * Mtodo para obter os grupos de servidores definidos por um usurio.
   * 
   * @return um array com as informaes dos grupos ou null em caso de erro
   */
  public static ServerGroupInfo[] getServerGroups() {
    final SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
    if (sgaService == null) {
      showError(LNG.get("SGA_NO_SERVER_ERROR"), null);
      return null;
    }
    DesktopTask task = new DesktopTask() {
      @Override
      public void performTask() throws Exception {
        setResult(sgaService.getServerGroups());
      }
    };
    task.start(null, LNG.get("SGA_ACCESSING_TITLE"),
      LNG.get("SGA_ACCESSING_MESSAGE"));

    /* Informa se ocorreu uma exceo */
    Exception exception = task.getException();
    if (exception != null) {
      showError(LNG.get("SGA_GETGROUP_FAILURE_MESSAGE"), exception);
      return null;
    }
    return (ServerGroupInfo[]) task.getResult();
  }

  /**
   * Mtodo para atualizar os grupos de servidores definidos por um usurio.
   * 
   * @param groups .
   * 
   * @return um array com as informaes dos grupos ou null em caso de erro
   */
  public static boolean setServerGroups(final ServerGroupInfo[] groups) {
    final SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
    if (sgaService == null) {
      showError(LNG.get("SGA_NO_SERVER_ERROR"), null);
      return false;
    }
    DesktopTask task = new DesktopTask() {
      @Override
      public void performTask() throws Exception {
        setResult(new Boolean(sgaService.setServerGroups(groups)));
      }
    };
    task.start(null, LNG.get("SGA_ACCESSING_TITLE"),
      LNG.get("SGA_ACCESSING_MESSAGE"));

    /* Informa se ocorreu uma exceo */
    Exception exception = task.getException();
    if (exception != null) {
      showError(LNG.get("SGA_SETGROUP_FAILURE_MESSAGE"), exception);
      return false;
    }
    return ((Boolean) task.getResult()).booleanValue();
  }

  /**
   * Mtodo para desativar (shutdown) um SGA especfico
   * 
   * @param sgaName .
   */
  public static void shutdownSGA(final String sgaName) {
    final SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
    if (sgaService == null) {
      showError(LNG.get("SGA_NO_SERVER_ERROR"), null);
      return;
    }
    DesktopTask task = new DesktopTask() {
      @Override
      public void performTask() throws Exception {
        setResult(sgaService.shutdownSGA(sgaName));
      }
    };
    task.start(null, LNG.get("SGA_MGMT_TITLE"), LNG.get("SGA_DISABLE_MESSAGE"));

    /* Informa se ocorreu uma exceo */
    Exception exception = task.getException();
    if (exception != null) {
      showError(LNG.get("SGA_FAILURE_MESSAGE"), exception);
      return;
    }
    if (!(Boolean) task.getResult()) {
      showError(String.format(LNG.get("SGA_COMMUNICATION_ERROR"), sgaName),
        null);
    }
  }

  /**
   * Reinicia o SGA.
   * 
   * @param sgaName nome do sga que se deseja reiniciar.
   */
  public static void restartSGA(final String sgaName) {
    final SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
    if (sgaService == null) {
      showError(LNG.get("SGA_NO_SERVER_ERROR"), null);
      return;
    }
    DesktopTask task = new DesktopTask() {
      @Override
      public void performTask() throws Exception {
        setResult(sgaService.restartSGA(sgaName));
      }
    };
    task.start(null, LNG.get("SGA_MGMT_TITLE"), LNG.get("SGA_RESTART_MESSAGE"));

    /* Informa se ocorreu uma exceo */
    Exception exception = task.getException();
    if (exception != null) {
      showError(LNG.get("SGA_FAILURE_MESSAGE"), exception);
      return;
    }
    if (!(Boolean) task.getResult()) {
      showError(String.format(LNG.get("SGA_COMMUNICATION_ERROR"), sgaName),
        null);
    }
  }

  /**
   * Mtodo para cancelar um comando.
   * 
   * @param sgaName nome do SGA.
   * @param cmdId identificador do comando.
   * @return indicativo de resultado.
   */
  public static boolean killCommand(final String sgaName, final String cmdId) {
    final SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
    if (sgaService == null) {
      showError(LNG.get("SGA_NO_SERVER_ERROR"), null);
      return false;
    }

    DesktopTask task = new DesktopTask() {
      @Override
      public void performTask() throws Exception {
        setResult(sgaService.killCommand(sgaName, cmdId));
      }
    };
    final String title = LNG.get("SGA_MGMT_TITLE");
    final String killMsg = LNG.get("SGA_COMMAND_KILL_MESSAGE");
    task.start(null, title, killMsg);

    /* Informa se ocorreu uma exceo */
    Exception exception = task.getException();
    if (exception != null) {
      final String err = LNG.get("SGA_FAILURE_MESSAGE");
      showError(err, exception);
      return false;
    }

    Object result = task.getResult();
    if (result != null) {
      final Boolean ok = (Boolean) result;
      if (ok) {
        return true;
      }
      else {
        if (User.getLoggedUser().isAdmin()) {
          final String confirmMsg = LNG.get("SGA_CONFIRM_KILL_ANYWAY");
          int answer = StandardDialogs.showYesNoDialog(null, title, confirmMsg);
          if (answer == JOptionPane.YES_OPTION) {
            task = new DesktopTask() {
              @Override
              public void performTask() throws Exception {
                sgaService.killCommandAnyway(sgaName, cmdId);
              }
            };
            task.start(null, title, killMsg);
            exception = task.getException();
            if (exception != null) {
              final String err = LNG.get("SGA_FAILURE_MESSAGE");
              showError(err, exception);
              return false;
            }
            return true;
          }
          return false;
        }

        final String comErrorMsg = LNG.get("SGA_COMMUNICATION_ERROR");
        showError(String.format(comErrorMsg, sgaName), null);
      }
    }
    return false;
  }

  /**
   * Mtodo para desativar (shutdown) todos os SGAs
   */
  public static void shutdownAllSGAs() {
    final SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
    if (sgaService == null) {
      showError(LNG.get("SGA_NO_SERVER_ERROR"), null);
      return;
    }
    DesktopTask task = new DesktopTask() {
      @Override
      public void performTask() throws Exception {
        setResult(sgaService.shutdownAllSGAs());
      }
    };
    task.start(null, LNG.get("SGA_MGMT_TITLE"),
      LNG.get("SGA_DISABLE_ALL_MESSAGE"));

    /* Informa se ocorreu uma exceo */
    Exception exception = task.getException();
    if (exception != null) {
      showError(LNG.get("SGA_FAILURE_MESSAGE"), exception);
      return;
    }
    Collection<String> sgasFailed = (Collection<String>) task.getResult();
    if (sgasFailed.size() > 0) {
      StringBuilder sgaNames = new StringBuilder();
      for (String sgaName : sgasFailed) {
        sgaNames.append("\n - ");
        sgaNames.append(sgaName);
      }
      String message =
        String.format(LNG.get("SGA_COMMUNICATION_ERROR"), sgaNames);
      showError(message, null);
    }
  }

  /**
   * Reinicia todos os SGAs em execuo.
   */
  public static void restartAllSGAs() {
    final SGAServiceInterface sgaService = ClientRemoteLocator.sgaService;
    if (sgaService == null) {
      showError(LNG.get("SGA_NO_SERVER_ERROR"), null);
      return;
    }
    DesktopTask task = new DesktopTask() {
      @Override
      public void performTask() throws Exception {
        setResult(sgaService.restartAllSGAs());
      }
    };
    task.start(null, LNG.get("SGA_MGMT_TITLE"),
      LNG.get("SGA_RESTART_ALL_MESSAGE"));

    /* Informa se ocorreu uma exceo */
    Exception exception = task.getException();
    if (exception != null) {
      showError(LNG.get("SGA_FAILURE_MESSAGE"), exception);
      return;
    }
    Collection<String> sgasFailed = (Collection<String>) task.getResult();
    if (sgasFailed.size() > 0) {
      StringBuilder sgaNames = new StringBuilder();
      for (String sgaName : sgasFailed) {
        sgaNames.append("\n - ");
        sgaNames.append(sgaName);
      }
      String message =
        String.format(LNG.get("SGA_COMMUNICATION_ERROR"), sgaNames);
      showError(message, null);
    }
  }
}
